package com.talpa.overlay.view.overlay

import android.content.Context
import android.content.Intent
import android.graphics.PixelFormat
import android.os.Build
import android.text.Html
import android.view.Gravity
import android.view.View
import android.view.WindowManager
import android.widget.TextView
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.overlay.R
import com.talpa.overlay.Transfer
import com.talpa.overlay.data.readOverlayTextLanguageTag
import com.talpa.overlay.tools.TextSpeech
import com.talpa.overlay.translate.translateForDetect
import com.talpa.overlay.view.overlayViewParams
import com.talpa.tengine.Trans
import io.reactivex.Single
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.functions.Consumer
import io.reactivex.schedulers.Schedulers
import org.jetbrains.anko.textColorResource
import java.lang.Exception

/**
 * @author CY 2019-10-21
 */
abstract class OverlayImpl(val context: Context) : OverlayInterface {


    companion object {

        /**
         *
         * 开始局部翻译
         *
         * putExtra(EXTRA_TEXT, sourceText)
         * putExtra(EXTRA_TARGET_LANGUAGE, targetLanguageLocaleTag)
         * putExtra(EXTRA_PACKAGE_NAME, nodeInfo.packageName)
         */
        const val ACTION_OVERLAY_TRANSLATE_START = "ACTION_OVERLAY_TRANSLATE_START"

        /**
         * putExtra(EXTRA_TEXT, sourceText)
         * putExtra(EXTRA_SOURCE_TARGET_LANGUAGE, "$sourceLanguage&&$targetLanguage")
         * putExtra(EXTRA_PACKAGE_NAME, packageName)
         * putExtra(EXTRA_CACHE, isCached)
         */
        const val ACTION_OVERLAY_TRANSLATE_SUCCESS = "ACTION_OVERLAY_TRANSLATE_SUCCESS"

        /**
         * putExtra(EXTRA_TEXT, sourceText)
         * putExtra(EXTRA_TARGET_LANGUAGE, targetLanguage)
         * putExtra(EXTRA_PACKAGE_NAME, packageName)
         */
        const val ACTION_OVERLAY_TRANSLATE_ERROR = "ACTION_OVERLAY_TRANSLATE_ERROR"

        /**
         * 开始输入框翻译
         */
        const val ACTION_EDIT_TEXT_TRANSLATE_START = "ACTION_EDIT_TEXT_TRANSLATE_START"
        const val ACTION_EDIT_TEXT_TRANSLATE_SUCCESS = "ACTION_EDIT_TEXT_TRANSLATE_SUCCESS"
        const val ACTION_EDIT_TEXT_TRANSLATE_ERROR = "ACTION_EDIT_TEXT_TRANSLATE_ERROR"
        const val ACTION_TEXT_SCREEN_TRANSLATE_SUCCESS = "ACTION_TEXT_SCREEN_TRANSLATE_SUCCESS"


        /**
         * 全局
         */
        const val ACTION_MULTI_TRANSLATE_START = "ACTION_MULTI_TRANSLATE_START"
        const val ACTION_MULTI_TRANSLATE_SUCCESS = "ACTION_MULTI_TRANSLATE_SUCCESS"
        const val ACTION_MULTI_TRANSLATE_ERROR = "ACTION_MULTI_TRANSLATE_ERROR"

        const val ACTION_SCREEN_TRANSLATE_FAIL = "action_screen_translate_fail"
        const val ACTION_SCREEN_TRANSLATE_SUCCESS = "action_screen_translate_success"


        const val EXTRA_TEXT = "text"
        const val EXTRA_TARGET_LANGUAGE = "target_language"
        const val EXTRA_PACKAGE_NAME = "package_name"
        const val EXTRA_SOURCE_TARGET_LANGUAGE = "source_target_language"
        const val EXTRA_CACHE = "cache"
        const val EXTRA_ERROR = "error"
        const val EXTRA_SOURCE = "translate_source"
        const val EXTRA_MODULE_TYPE = "module_type"
        const val MODULE_SCREEN = "module_screen"


    }

    val windowManager: WindowManager by lazy { context.getSystemService(Context.WINDOW_SERVICE) as WindowManager }

    val contentView: View by lazy {
        createContentView()
    }

    val textSpeech by lazy { TextSpeech }

    /**
     * 翻译结果
     */
    private var mTrans: Trans? = null

    /**
     * 本地广播
     */
    val localBroadcastManager by lazy { LocalBroadcastManager.getInstance(context) }

    abstract fun createContentView(): View

    override fun addContentView(x: Int, y: Int) {
        mTrans = null
    }

    /**
     * 悬浮视图Params
     */
    fun overlayParams(layoutParams: WindowManager.LayoutParams? = null): WindowManager.LayoutParams {
        var params = layoutParams ?: overlayViewParams()
        params.flags = params.flags or (WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL or WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE)

        params.x = 0
        params.y = 0
        params.width = 0
        params.height = 0

        params.gravity = Gravity.START or Gravity.TOP

        return params
    }

    /**
     * 处理悬浮　Text 翻译
     */
    fun translateForOverlayText(
        tvTranslation: TextView,
        sourceText: String,
        targetLanguageTag: String,
        onSuccess: ((trans: Trans) -> Unit)? = null,
        onError: ((trans: Trans) -> Unit)? = null
    ) {

        val onNext = Consumer<Trans> { trans ->

            mTrans = trans

            when (trans.result?.code) {
                Trans.SUCCESS -> {
                    val translation = trans.result?.translation
                    tvTranslation.text = Html.fromHtml(translation ?: "")
                    tvTranslation.textColorResource = R.color.color_floating_translation
                    tvTranslation.setTag(R.id.id_translation_view_trans_result, tvTranslation.text)

                    onSuccess?.invoke(trans)

                    if (translation != null) {
                        //readOverlayTextLanguageTag(context)
                        insertTranslateHistory(sourceText, translation, targetLanguageTag)
                    }
                }
                Trans.ERROR_NETWORK -> {
                    tvTranslation.setText(R.string.network_error)
                    tvTranslation.textColorResource = R.color.color_floating_failure
                    onError?.invoke(trans)
                }
                /*Trans.ERROR_NETWORK_TIMEOUT -> {
                    tvTranslation.setText(R.string.network_timeout)
                    tvTranslation.textColorResource = R.color.color_floating_failure
                    onError?.invoke(trans)
                }*/
                else -> {
                    tvTranslation.setText(R.string.text_translating_error)
                    tvTranslation.textColorResource = R.color.color_floating_failure
                    onError?.invoke(trans)
                }
            }
        }
        translateForDetect(
            tvTranslation,
            sourceText,
            targetLanguageTag,
            onNext
        )
    }


    override fun removeContentView() {
        if (textSpeech.isSpeaking()) {
            textSpeech.stopSpeak()
        }
    }


    fun insertTranslateHistory(
        text: String,
        translation: String,
        targetLanguageTag: String,
        isChecked: Boolean = false
    ) {

        val sourceLanguageTag = "AUTO"
        //val targetLanguageTag = readOverlayTextLanguageTag(context)
        val scene = 1
        //const val SCENE_DICTIONARY = 0
        //const val SCENE_FLOATING = 1

        val intent = Intent("BROADCAST_ACTION_INSERT_TRANSLATE_RECORD")
        intent.putExtra("isChecked", isChecked)
        intent.putExtra("text", text)
        intent.putExtra("translation", translation)
        intent.putExtra("source_language", sourceLanguageTag)
        intent.putExtra("target_language", targetLanguageTag)
        intent.putExtra("scene", scene)

        LocalBroadcastManager.getInstance(context).sendBroadcast(intent)
    }

    fun updateTranslateHistory(text: String, translation: String, isChecked: Boolean = true) {

        val sourceLanguageTag = "AUTO"
        val targetLanguageTag = readOverlayTextLanguageTag(context)
        val scene = 1
        //const val SCENE_DICTIONARY = 0
        //const val SCENE_FLOATING = 1

        val intent = Intent("BROADCAST_ACTION_UPDATE_TRANSLATE_RECORD")
        intent.putExtra("isChecked", isChecked)
        intent.putExtra("text", text)
        intent.putExtra("translation", translation)
        intent.putExtra("source_language", sourceLanguageTag)
        intent.putExtra("target_language", targetLanguageTag)
        intent.putExtra("scene", scene)

        LocalBroadcastManager.getInstance(context).sendBroadcast(intent)
    }

    /**
     *  it.text,
     *  it.translation,
     *  it.sourceLanguageLocaleTag,
     *  it.targetLanguageLocaleTag,
     *  it.star
     */
    fun readTranslateHistory(
        text: String,
        targetLanguageLocaleTag: String
    ): Single<Array<Any>> {

        val sourceLanguageLocaleTag = "AUTO"
        try {
            val single =
                Transfer.queryTranslateHistory(
                    text,
                    sourceLanguageLocaleTag,
                    targetLanguageLocaleTag
                )
            return single.subscribeOn(Schedulers.io()).observeOn(AndroidSchedulers.mainThread())
        } catch (e: Exception) {
            e.printStackTrace()
        }
        return Single.just(arrayOf())
    }


    /**
     * 打点开始翻译
     */
    fun logStartTranslate(
        action: String = ACTION_OVERLAY_TRANSLATE_START,
        sourceText: String,
        targetLanguageTag: String,
        packageName: String
    ) {

        val intent = Intent(action).apply {
            putExtra(EXTRA_TEXT, sourceText)
            putExtra(EXTRA_TARGET_LANGUAGE, targetLanguageTag)
            putExtra(EXTRA_PACKAGE_NAME, packageName)
        }

        localBroadcastManager.sendBroadcast(intent)
    }


    /**
     * 打点成功翻译
     */
    fun logSuccessTranslate(
        action: String = ACTION_OVERLAY_TRANSLATE_SUCCESS,
        sourceText: String,
        sourceLanguage: String,
        targetLanguage: String,
        packageName: String,
        isCached: Boolean = false,
        source: String
    ) {

        val intent = Intent(action).apply {
            putExtra(EXTRA_TEXT, sourceText)
            putExtra(EXTRA_SOURCE_TARGET_LANGUAGE, "$sourceLanguage&&$targetLanguage")
            putExtra(EXTRA_PACKAGE_NAME, packageName)
            putExtra(EXTRA_CACHE, isCached)
            putExtra(EXTRA_SOURCE, source)
        }

        localBroadcastManager.sendBroadcast(intent)
    }

    /**
     * 打点开始翻译
     */
    fun logFailTranslate(
        action: String = ACTION_OVERLAY_TRANSLATE_ERROR,
        sourceText: String,
        targetLanguageTag: String,
        packageName: String,
        errorMessage: String
    ) {

        val intent = Intent(action).apply {
            putExtra(EXTRA_TEXT, sourceText)
            putExtra(EXTRA_TARGET_LANGUAGE, targetLanguageTag)
            putExtra(EXTRA_PACKAGE_NAME, packageName)
            putExtra(EXTRA_ERROR, errorMessage)
        }

        localBroadcastManager.sendBroadcast(intent)
    }


}

/**
 * 悬浮视图Params
 */
fun WindowManager.overlayParams(layoutParams: WindowManager.LayoutParams? = null): WindowManager.LayoutParams {
    val type = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
        WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
    else WindowManager.LayoutParams.TYPE_PHONE
    val flags = WindowManager.LayoutParams.FLAG_WATCH_OUTSIDE_TOUCH or
            WindowManager.LayoutParams.FLAG_NOT_TOUCH_MODAL or
            WindowManager.LayoutParams.FLAG_NOT_FOCUSABLE or
            WindowManager.LayoutParams.FLAG_LAYOUT_IN_SCREEN or
            WindowManager.LayoutParams.FLAG_HARDWARE_ACCELERATED
    val params =
        layoutParams ?: WindowManager.LayoutParams(type, flags, PixelFormat.TRANSLUCENT)
    //
    params.x = 0
    params.y = 0
    params.width = 0
    params.height = 0

    params.gravity = Gravity.START or Gravity.TOP

    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.P) {
        params.layoutInDisplayCutoutMode =
            WindowManager.LayoutParams.LAYOUT_IN_DISPLAY_CUTOUT_MODE_SHORT_EDGES
    }

    params.format = PixelFormat.RGBA_8888
    return params
}

