package com.talpa.overlay.view.overlay

import android.annotation.SuppressLint
import android.content.Context
import android.graphics.Rect
import android.os.Bundle
import android.os.Handler
import android.text.TextUtils
import android.util.Log
import android.view.MotionEvent
import android.view.View
import android.view.WindowManager
import android.view.accessibility.AccessibilityNodeInfo
import android.widget.ProgressBar
import android.widget.TextView
import com.talpa.overlay.R
import com.talpa.overlay.data.readEditNodeInfo
import com.talpa.overlay.data.readOverlayEditTextLanguageTag
import com.talpa.overlay.data.writeEditNodeInfo
import com.talpa.overlay.service.nodeText
import com.talpa.overlay.translate.translateForEditText
import com.talpa.overlay.view.FloatingContainer
import com.talpa.tengine.STORE
import com.talpa.tengine.Trans
import com.talpa.tengine.UNKNOWN
import com.talpa.tengine.lang.LANG
import com.trello.rxlifecycle3.kotlin.bindToLifecycle
import io.reactivex.functions.Consumer
import org.jetbrains.anko.find
import org.jetbrains.anko.layoutInflater
import org.jetbrains.anko.textColorResource

/**
 * EDIT Overlay
 *
 * @author CY 2019-11-13
 *
 */
class EditTextOverlayView(context: Context, private val handler: Handler? = null) :
    OverlayImpl(context) {

    companion object {
        const val TAG = "EditTextOverlayView"
    }

    private val rectInScreen = Rect()

    @SuppressLint("ClickableViewAccessibility")
    override fun createContentView(): View {
        val view = context.layoutInflater.inflate(R.layout.layout_content_edit, null)
        view.setOnTouchListener { v, event ->

            when (event.action) {
                MotionEvent.ACTION_OUTSIDE -> {
                    removeContentView()
                }
            }

            return@setOnTouchListener false
        }

        (view as BaseOverlayView).setOnBackClickListener = object : BaseOverlayView.BackClickListener {
            override fun onBackClickListener(): Boolean {
                try {
                    if (view.isAttachedToWindow) {
                        windowManager.removeView(view)
                        return true
                    } else {
                        return false
                    }
                } catch (e: Exception) {
                    e.printStackTrace()
                }
                return false
            }
        }

        return view
    }

    override fun addContentView(x: Int, y: Int) {
        super.addContentView(x, y)
        val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)
        params.x = x
        params.y = y

        contentView.visibility = View.GONE
        try {
            if (!contentView.isAttachedToWindow && contentView.parent == null) {
                windowManager.addView(contentView, params)
            } else {
                windowManager.updateViewLayout(contentView, params)
            }
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    private fun addContentView(params: WindowManager.LayoutParams) {
        if (!contentView.isAttachedToWindow && contentView.parent == null) {
            try {
                windowManager.addView(contentView, params)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
    }

    override fun updateContentView(nodeInfo: AccessibilityNodeInfo) {

        editTextNodeInfo(nodeInfo)
    }


    private fun showContentView(nodeInfo: AccessibilityNodeInfo) {
        val params = overlayParams(contentView.layoutParams as? WindowManager.LayoutParams)

        addContentView(params)

        nodeInfo.getBoundsInScreen(rectInScreen)
        // val params = contentView.layoutParams as WindowManager.LayoutParams
        params.x = rectInScreen.left
        params.y = rectInScreen.top
        params.width = rectInScreen.width()
        params.height = rectInScreen.height()

        //contentView.setBackgroundResource(R.drawable.shape_edit_bg)
        if (contentView.visibility != View.VISIBLE) {
            contentView.visibility = View.VISIBLE
        }
        try {
            windowManager.updateViewLayout(contentView, params)
        } catch (e: Exception) {
            e.printStackTrace()
        }
    }

    override fun removeContentView() {
        if (contentView.isAttachedToWindow) {
            try {
                val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)
                tvStatus?.visibility = View.GONE
                tvStatus?.text = ""
                windowManager.removeViewImmediate(contentView)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }
        super.removeContentView()
    }

    /**
     * 处理文本编辑框节点
     */
    private fun editTextNodeInfo(nodeInfo: AccessibilityNodeInfo) {

        val text = nodeInfo.nodeText()

        val sourceText = text.toString().trim()

        val packageName = nodeInfo.packageName?.toString() ?: ""

        if (!TextUtils.isEmpty(sourceText)) {

            val targetText = restoreEditInfo(nodeInfo)
            val targetLanguageTag = readOverlayEditTextLanguageTag(context) ?: LANG.EN
            //targetLanguageTag
            //?: throw Exception("please set text translate targetLanguage,to see writeOverlayEditTextLanguageTag()")

            if (targetText == null) {
                dealNodeInfo(sourceText, targetLanguageTag, packageName, nodeInfo)
            } else {
                executorAction(nodeInfo, targetText)
            }
        }
    }

    /**
     * Restore Edit Text Node Info
     */
    private fun restoreEditInfo(
        nodeInfo: AccessibilityNodeInfo
    ): String? {
        val languageTag = readOverlayEditTextLanguageTag(context) ?: return null
        return readEditNodeInfo(context, nodeInfo, languageTag)
    }

    /**
     * Store Edit Text Node Info
     */
    private fun storeEditInfo(
        text: String,
        translation: String,
        toLanguageTag: String,
        nodeInfo: AccessibilityNodeInfo
    ) {
        writeEditNodeInfo(context, text, translation, toLanguageTag, nodeInfo)
    }

    /**
     * 执行无障碍服务动作
     */
    private fun executorAction(nodeInfo: AccessibilityNodeInfo, text: String) {
        try {
            val arguments = Bundle()
            arguments.putCharSequence(AccessibilityNodeInfo.ACTION_ARGUMENT_SET_TEXT_CHARSEQUENCE, text)
            //可能被回收
            nodeInfo.performAction(AccessibilityNodeInfo.ACTION_SET_TEXT, arguments)
        } catch (e: Exception) {
            //e.printStackTrace()
        }

    }

    private fun showProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.VISIBLE
    }

    private fun hideProgressBar(contentView: View?) {
        val progressBar = contentView?.find<ProgressBar>(R.id.loading_progress_bar)
        progressBar?.visibility = View.GONE
    }

    /**
     * 翻译前
     */
    private fun preTranslate(nodeInfo: AccessibilityNodeInfo): Boolean {

        showContentView(nodeInfo)

        val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)

        showProgressBar(contentView)
        return true
    }


    /**
     * Success Trans
     */
    private fun successTranslate(
        sourceText: String,
        translation: String,
        targetLanguageTag: String,
        nodeInfo: AccessibilityNodeInfo
    ) {

        executorAction(nodeInfo, translation)

        storeEditInfo(sourceText, translation, targetLanguageTag, nodeInfo)

        hideProgressBar(contentView)

        removeContentView()

        handler?.sendEmptyMessage(FloatingContainer.WHAT_TRANSLATE_SUCCESS)
    }

    /**
     * Failure Trans
     */
    private fun failureTranslate(
        sourceText: String,
        targetLanguageTag: String,
        packageName: String,
        errorMessage: String
    ) {
        val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)
        tvStatus?.setText(R.string.text_translating_error)
        tvStatus?.textColorResource = R.color.color_floating_failure
        tvStatus?.visibility = View.VISIBLE
        logFailTranslate(
            ACTION_EDIT_TEXT_TRANSLATE_ERROR,
            sourceText,
            targetLanguageTag,
            packageName,
            errorMessage = errorMessage
        )
        hideStatus(tvStatus)
        hideProgressBar(contentView)

        handler?.sendEmptyMessage(FloatingContainer.WHAT_TRANSLATE_FAILURE)

    }

    private fun dealNodeInfo(
        sourceText: String,
        targetLanguageTag: String,
        packageName: String,
        nodeInfo: AccessibilityNodeInfo
    ) {

        logStartTranslate(
            ACTION_EDIT_TEXT_TRANSLATE_START,
            sourceText,
            targetLanguageTag,
            packageName
        )

        /**
         *  it.text,
         *  it.translation,
         *  it.sourceLanguageLocaleTag,
         *  it.targetLanguageLocaleTag,
         *  it.star
         */
        val d = readTranslateHistory(sourceText, targetLanguageTag)
            .bindToLifecycle(contentView)
            .subscribe({

                if (it.size == 5 && it[4] is Boolean) {

                    val (text, translation, sourceLanguageLocaleTag, targetLanguageLocaleTag, isChecked) = it

                    logSuccessTranslate(
                        ACTION_EDIT_TEXT_TRANSLATE_SUCCESS,
                        sourceText,
                        sourceLanguageLocaleTag.toString(),
                        targetLanguageLocaleTag.toString(),
                        packageName,
                        isCached = true,
                        source = STORE
                    )
                    successTranslate(
                        sourceText,
                        translation.toString(),
                        targetLanguageTag,
                        nodeInfo
                    )
                } else {
                    translate(sourceText, targetLanguageTag, packageName, nodeInfo)
                }
            }, {
                it.printStackTrace()
                translate(sourceText, targetLanguageTag, packageName, nodeInfo)
            })
    }

    private fun translate(
        sourceText: String,
        targetLanguageTag: String,
        packageName: String,
        nodeInfo: AccessibilityNodeInfo
    ) {


        if (!preTranslate(nodeInfo)) {
            return
        }

        val onSuccess = Consumer { trans: Trans ->
            when (trans.result?.code) {
                Trans.SUCCESS -> {
                    val translation = trans.result?.translation

                    if (translation != null) {

                        val sourceLanguage = trans.result!!.detectLang ?: "AUTO"
                        logSuccessTranslate(
                            ACTION_EDIT_TEXT_TRANSLATE_SUCCESS,
                            sourceText,
                            sourceLanguage,
                            targetLanguageTag,
                            packageName,
                            source = trans.result?.source ?: UNKNOWN
                        )
                        insertTranslateHistory(sourceText, translation, targetLanguageTag)

                        successTranslate(sourceText, translation, targetLanguageTag, nodeInfo)
                    } else {
                        val errorMessage = "SUCCESS but translation==null"
                        failureTranslate(
                            sourceText,
                            targetLanguageTag,
                            packageName,
                            errorMessage
                        )
                    }

                }
                else -> {

                    val errorMessage = trans.result?.errorMessage ?: "unknown"
                    failureTranslate(
                        sourceText,
                        targetLanguageTag,
                        packageName,
                        errorMessage
                    )

                }
            }
            // hideProgressBar(contentView)

        }

        val onError = Consumer<Throwable> {
            /*val tvStatus = contentView.findViewById<TextView>(R.id.tv_status)
            tvStatus?.setText(R.string.text_translating_error)
            tvStatus?.textColorResource = R.color.color_floating_failure
            tvStatus?.visibility = View.VISIBLE
            logFailTranslate(
                ACTION_EDIT_TEXT_TRANSLATE_ERROR,
                sourceText,
                targetLanguageTag,
                packageName,
                errorMessage = it.message ?: "unknown"
            )*/

            val errorMessage = it.message ?: "unknown"
            failureTranslate(
                sourceText,
                targetLanguageTag,
                packageName,
                errorMessage
            )

            //hideProgressBar(contentView)

        }

        translateForEditText(sourceText, targetLanguageTag, onSuccess, onError, context = context)

    }

    private fun hideStatus(tvStatus: TextView?) {
        tvStatus?.postDelayed({
            removeContentView()
        }, 1200)
    }
}