package com.talpa.overlay.view

import android.content.Context
import android.graphics.Rect
import android.text.Selection
import android.text.Spannable
import android.text.method.ScrollingMovementMethod
import android.text.style.ClickableSpan
import android.util.Log
import android.view.GestureDetector
import android.view.MotionEvent
import android.view.VelocityTracker
import android.view.ViewConfiguration
import android.view.animation.Interpolator
import android.widget.OverScroller
import android.widget.TextView
import androidx.core.view.MotionEventCompat
import androidx.core.view.VelocityTrackerCompat
import androidx.core.view.ViewCompat

/**
 * Create by chenjunsheng on 2020/11/19
 */

class SmoothMovement constructor(context: Context) : ScrollingMovementMethod(), GestureDetector.OnGestureListener {

    private var mContext: Context
    /*private const val SCREEN_WIDTH = 0
    private const val SCREEN_HEIGHT = 0
    private const val mWidth = 0
    private const val mHeight = 0*/

    companion object {
        const val SCROLL_STATE_IDLE = 0
        const val SCROLL_STATE_DRAGGING = 1
        const val SCROLL_STATE_SETTLING = 2
        const val INVALID_POINTER = -1
    }

    private var mScrollState = SCROLL_STATE_IDLE
    private var mScrollPointerId = INVALID_POINTER
    private var mVelocityTracker: VelocityTracker? = null
    private var mLastTouchY = 0
    private val mTouchSlop: Int
    private val mMinFlingVelocity: Int
    private val mMaxFlingVelocity: Int
    private var mViewFlinger: ViewFlinger? = null
    private var mGestureDetector: GestureDetector
    private var mGestureListener: GestureDetector.OnGestureListener? = null

    init {
        mContext = context.applicationContext
        mGestureDetector = GestureDetector(context, this)

    }

    fun setGestureListener(listener: GestureDetector.OnGestureListener) {
        mGestureListener = listener
    }

    private fun setScrollState(state: Int) {
        if (state == mScrollState) {
            return
        }
        mScrollState = state
        if (state != SCROLL_STATE_SETTLING) {
            mViewFlinger?.stop()
        }
    }

    init {
        val vc = ViewConfiguration.get(mContext)
        mTouchSlop = vc.getScaledTouchSlop();
        mMinFlingVelocity = vc.getScaledMinimumFlingVelocity();
        mMaxFlingVelocity = vc.getScaledMaximumFlingVelocity();
    }

    private fun constrainScrollBy(view: TextView, dx: Int, dy: Int) {
        var dx = dx
        var dy = dy
        val viewport = Rect()
        view.getGlobalVisibleRect(viewport)
        val height: Int = viewport.height()
        val width: Int = viewport.width()
        val scrollX: Int = view.getScrollX()
        val scrollY: Int = view.getScrollY()

        //右边界
        if (view.width - scrollX - dx < width) {
            dx = view.width - scrollX - width
        }
        //左边界
        if (-scrollX - dx > 0) {
            dx = -scrollX
        }
        //下边界
        if (view.layout.height + view.paddingBottom + view.paddingTop - scrollY - dy < height) {
            dy = view.layout.height + view.paddingBottom + view.paddingTop - scrollY - height
        }
        //上边界
        if (scrollY + dy < 0) {
            dy = -scrollY
        }
        view.scrollBy(dx, dy)
    }

    //f(x) = (x-1)^5 + 1
    private val sQuinticInterpolator: Interpolator = Interpolator { t ->
        var t = t
        t -= 1.0f
        t * t * t * t * t + 1.0f
    }

    override fun onTouchEvent(widget: TextView, buffer: Spannable, event: MotionEvent): Boolean {
        mGestureDetector.onTouchEvent(event)
        if (mVelocityTracker == null) {
            mVelocityTracker = VelocityTracker.obtain()
        }
        if (mViewFlinger == null) {
            mViewFlinger = ViewFlinger(widget)
        }
        var eventAddedToVelocityTracker = false
        val action: Int = event.getActionMasked()
        val actionIndex: Int = event.getActionIndex()
        val vtev = MotionEvent.obtain(event)

        when (action) {
            MotionEvent.ACTION_DOWN -> {
                setScrollState(SCROLL_STATE_IDLE)
                mScrollPointerId = event.getPointerId(0)
                mLastTouchY = (event.y + 0.5f).toInt()

                var x = event.x.toInt()
                var y = event.y.toInt()

                x -= widget.totalPaddingLeft
                y -= widget.totalPaddingTop

                x += widget.scrollX
                y += widget.scrollY

                val layout = widget.layout
                val line = layout.getLineForVertical(y)
                val off = layout.getOffsetForHorizontal(line, x.toFloat())

                val links = buffer.getSpans(
                    off, off,
                    ClickableSpan::class.java
                )

                if (links.isNotEmpty()) {
                    val link = links[0]
                    Selection.setSelection(
                        buffer,
                        buffer.getSpanStart(link),
                        buffer.getSpanEnd(link)
                    )
                }

            }
            MotionEventCompat.ACTION_POINTER_DOWN -> {
                mScrollPointerId = event.getPointerId(actionIndex)
                mLastTouchY = (event.getY(actionIndex) + 0.5f).toInt()
            }
            MotionEvent.ACTION_MOVE -> {
                val index = event.findPointerIndex(mScrollPointerId)
                if (index < 0) {
                    Log.e(
                        "cjslog",
                        "Error processing scroll; pointer index for id " + mScrollPointerId.toString() + " not found. Did any MotionEvents get skipped?"
                    )
                    return false
                }
                val y = (event.getY(index) + 0.5f).toInt()
                var dy: Int = mLastTouchY - y
                if (mScrollState != SCROLL_STATE_DRAGGING) {
                    var startScroll = false
                    if (Math.abs(dy) > mTouchSlop) {
                        if (dy > 0) {
                            dy -= mTouchSlop
                        } else {
                            dy += mTouchSlop
                        }
                        startScroll = true
                    }
                    if (startScroll) {
                        setScrollState(SCROLL_STATE_DRAGGING)
                    }
                }
                if (mScrollState == SCROLL_STATE_DRAGGING) {
                    mLastTouchY = y
                    constrainScrollBy(widget, 0, dy)
                }
            }
            MotionEventCompat.ACTION_POINTER_UP -> {
                if (event.getPointerId(actionIndex) == mScrollPointerId) {
                    // Pick a new pointer to pick up the slack.
                    val newIndex = if (actionIndex == 0) 1 else 0
                    mScrollPointerId = event.getPointerId(newIndex)
                    mLastTouchY = (event.getY(newIndex) + 0.5f).toInt()
                }
            }
            MotionEvent.ACTION_UP -> {
                if (mScrollState == SCROLL_STATE_DRAGGING) {
                    mVelocityTracker!!.addMovement(vtev)
                    eventAddedToVelocityTracker = true
                    mVelocityTracker!!.computeCurrentVelocity(1000, mMaxFlingVelocity.toFloat())
                    var yVelocity: Float =
                        -VelocityTrackerCompat.getYVelocity(mVelocityTracker, mScrollPointerId)
                    yVelocity = if (Math.abs(yVelocity) < mMinFlingVelocity) {
                        0f
                    } else {
                        Math.max(
                            -mMaxFlingVelocity.toFloat(), Math.min(
                                yVelocity,
                                mMaxFlingVelocity.toFloat()
                            )
                        ).toFloat()
                    }
                    if (yVelocity != 0f) {
                        mViewFlinger?.fling(yVelocity.toInt())
                    } else {
                        setScrollState(SCROLL_STATE_IDLE)
                    }
                    resetTouch()
                } else {
                    var x = event.x.toInt()
                    var y = event.y.toInt()

                    x -= widget.totalPaddingLeft
                    y -= widget.totalPaddingTop

                    x += widget.scrollX
                    y += widget.scrollY

                    val layout = widget.layout
                    val line = layout.getLineForVertical(y)
                    val off = layout.getOffsetForHorizontal(line, x.toFloat())

                    val links = buffer.getSpans(
                        off, off,
                        ClickableSpan::class.java
                    )
                    if (links.isNotEmpty()) {
                        val link = links[0]
                        link.onClick(widget)
                    }

                }
            }
            MotionEvent.ACTION_CANCEL -> {
                resetTouch()
            }
        }
        if (!eventAddedToVelocityTracker) {
            mVelocityTracker?.addMovement(vtev)
        }
        vtev.recycle()
        return true
    }

    private fun resetTouch() {
        if (mVelocityTracker != null) {
            mVelocityTracker!!.clear()
        }
    }

    override fun onTrackballEvent(
        widget: TextView?,
        text: Spannable?,
        event: MotionEvent?
    ): Boolean {
        return super.onTrackballEvent(widget, text, event)
    }

    private inner class ViewFlinger(private var mView: TextView) : Runnable {
        private var mLastFlingY = 0
        private val mScroller: OverScroller
        private var mEatRunOnAnimationRequest = false
        private var mReSchedulePostAnimationCallback = false

        override fun run() {
            disableRunOnAnimationRequests()
            val scroller = mScroller
            if (scroller.computeScrollOffset()) {
                val y = scroller.currY
                val dy = y - mLastFlingY
                mLastFlingY = y
                constrainScrollBy(mView, 0, dy)
                postOnAnimation()
            }
            enableRunOnAnimationRequests()
        }

        fun fling(velocityY: Int) {
            mLastFlingY = 0
            setScrollState(SCROLL_STATE_SETTLING)
            mScroller.fling(
                0,
                0,
                0,
                velocityY,
                Int.MIN_VALUE,
                Int.MAX_VALUE,
                Int.MIN_VALUE,
                Int.MAX_VALUE
            )
            postOnAnimation()
        }

        fun stop() {
            mView.removeCallbacks(this)
            mScroller.abortAnimation()
        }

        private fun disableRunOnAnimationRequests() {
            mReSchedulePostAnimationCallback = false
            mEatRunOnAnimationRequest = true
        }

        private fun enableRunOnAnimationRequests() {
            mEatRunOnAnimationRequest = false
            if (mReSchedulePostAnimationCallback) {
                postOnAnimation()
            }
        }

        fun postOnAnimation() {
            if (mEatRunOnAnimationRequest) {
                mReSchedulePostAnimationCallback = true
            } else {
                mView.removeCallbacks(this)
                ViewCompat.postOnAnimation(mView, this)
            }
        }

        init {
            mScroller = OverScroller(mContext, sQuinticInterpolator)
        }
    }

    override fun onDown(e: MotionEvent?): Boolean {
        return mGestureListener?.onDown(e) ?: false
    }

    override fun onShowPress(e: MotionEvent?) {
        mGestureListener?.onShowPress(e)
    }

    override fun onSingleTapUp(e: MotionEvent?): Boolean {
        return mGestureListener?.onSingleTapUp(e) ?: false
    }

    override fun onScroll(
        e1: MotionEvent?,
        e2: MotionEvent?,
        distanceX: Float,
        distanceY: Float
    ): Boolean {
        return mGestureListener?.onScroll(e1, e2, distanceX, distanceY) ?: false
    }

    override fun onLongPress(e: MotionEvent?) {
        mGestureListener?.onLongPress(e)
    }

    override fun onFling(
        e1: MotionEvent?,
        e2: MotionEvent?,
        velocityX: Float,
        velocityY: Float
    ): Boolean {
        return mGestureListener?.onFling(e1, e2, velocityX, velocityY) ?: false
    }
}