package com.talpa.overlay.service

import android.accessibilityservice.AccessibilityService
import android.app.AlertDialog
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.graphics.Rect
import android.os.Build
import android.os.Message
import android.text.TextUtils
import android.util.Log
import android.view.KeyEvent
import android.view.WindowManager
import android.view.accessibility.AccessibilityEvent
import android.view.accessibility.AccessibilityNodeInfo
import android.webkit.WebView
import android.widget.ImageButton
import android.widget.ImageView
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.overlay.BuildConfig
import com.talpa.overlay.R
import com.talpa.overlay.RxRelay
import com.talpa.overlay.RxRelay.EVENT_ACTIVITY_CHANGE
import com.talpa.overlay.RxRelay.EVENT_NODE_INFO_DOWN
import com.talpa.overlay.RxRelay.EVENT_NODE_INFO_MULTI
import com.talpa.overlay.RxRelay.postByEventBus
import com.talpa.overlay.data.ActivityChangedEvent
import com.talpa.overlay.state.FloatingStateMachine
import com.talpa.overlay.tools.IntentRecord
import com.talpa.overlay.tools.canDrawOverlays
import com.talpa.overlay.view.FloatingManager
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.Single
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.functions.Consumer
import io.reactivex.functions.Function
import io.reactivex.functions.Predicate
import io.reactivex.schedulers.Schedulers
import java.util.concurrent.TimeUnit

/**
 * @author CY 2019-06-10
 */
class AccessService : AccessibilityService() {


    companion object {
        const val TAG = "AccessService"

        private const val ACTION_DOWN = "com.talpa.overlay.core.AccessService#ACTION_DOWN"
        private const val ACTION_POINT = "com.talpa.overlay.core.AccessService#ACTION_POINT"

        /**
         * 开始全局翻译
         */
        private const val ACTION_NODES_START =
            "com.talpa.overlay.core.AccessService#ACTION_NODES_START"

        /**
         * 停止全局翻译
         */
        private const val ACTION_NODES_STOP =
            "com.talpa.overlay.core.AccessService#ACTION_NODES_STOP"

        private const val EXTRA_POINT_X = "point_x"
        private const val EXTRA_POINT_Y = "point_y"

        /**
         * 无障碍服务已连接
         */
        const val BROADCAST_ACTION_SERVICE_CONNECTED =
            "com.talpa.overlay.service#BROADCAST_ACTION_CONNECTED"


        /**
         * 无障碍服务解除绑定
         */
        private const val BROADCAST_ACTION_UN_BIND =
            "com.talpa.overlay.service#BROADCAST_ACTION_UN_BIND"

        /**
         * 无障碍服务重新绑定
         */
        private const val BROADCAST_ACTION_RE_BIND =
            "com.talpa.overlay.service#BROADCAST_ACTION_RE_BIND"

        /**
         * 无障碍服务销毁
         */
        const val BROADCAST_ACTION_DESTROY =
            "com.talpa.overlay.service#BROADCAST_ACTION_DESTROY"

        /**
         * 请求无障碍服务
         */
        const val BROADCAST_ACTION_REQUEST = "com.talpa.overlay.service#BROADCAST_ACTION_REQUEST"

        /**
         * 全局间隔时间
         */
        private const val MILLIS_DELAY = 1500L//delayMillis

        /**
         * 无障碍服务是否正在运行
         */
        var isAccessibilityServiceRun = false

        /**
         * 打开悬浮球广播
         */
        private const val BROADCAST_ACTION_FLOATING_OPEN = "BROADCAST_ACTION_FLOATING_OPEN"

        /**
         * 关闭悬浮球广播
         */
        private const val BROADCAST_ACTION_FLOATING_CLOSE = "BROADCAST_ACTION_FLOATING_CLOSE"

        /**
         * 启动服务，传递屏幕坐标位置，返回当前位置所在的NodeInfo
         */
        fun startServiceForPoint(context: Context, x: Int, y: Int) {

            context.startService(Intent(context, AccessService::class.java).apply {
                action = ACTION_POINT
                putExtra(EXTRA_POINT_X, x)
                putExtra(EXTRA_POINT_Y, y)
            })
        }

        /**
         * 启动服务，返回当前屏幕所有可见的NodeInfo
         */
        fun startServiceForStartMultiNodes(context: Context) {
            context.startService(Intent(context, AccessService::class.java).apply {
                action = ACTION_NODES_START
            })
        }

        fun startServiceForStopMultiNodes(context: Context) {
            context.startService(Intent(context, AccessService::class.java).apply {
                action = ACTION_NODES_STOP
            })
        }

        /**
         * 启动服务，记录此时的 root view
         */
        fun startServiceForDown(context: Context) {

            context.startService(Intent(context, AccessService::class.java).apply {
                action = ACTION_DOWN
            })
        }
    }

    private val pointNodeBound = Rect()

    /**
     * 当前最顶部的应用包名
     */
    //private var currentPackageName: CharSequence? = null

    private var currentAction: String? = null

    private var multiDisposable: Disposable? = null

    /**
     * 是否触发AccessibilityEvent
     */
    private var isAccessibilityEvent = false

    /**
     * 节点信息
     */
    private var rootNodeInfo: AccessibilityNodeInfo? = null

    /**
     * 最新的包名
     */
    private var lastPackageName: CharSequence? = null

    /**
     * 节点Rect
     */
    private val nodeRectSet by lazy { hashSetOf<String>() }

    /**
     * 当前Version Code
     */
    //private var currentVersionCode = 0

    /**
     * 解除绑定提示 Dialog
     */
    private var unBindingDialog: AlertDialog? = null

    /**
     * 本地广播
     */
    private val localBroadcastManager by lazy { LocalBroadcastManager.getInstance(applicationContext) }

    private fun log(message: String) {
        Log.d(TAG, message)
    }

    override fun onCreate() {
        super.onCreate()
        /*if (BuildConfig.DEBUG)*/ log("onCreate")
    }


    override fun onStartCommand(intent: Intent?, flags: Int, startId: Int): Int {
        /*if (BuildConfig.DEBUG)*/ log("onStartCommand")

        currentAction = intent?.action

        when (intent?.action) {
            ACTION_POINT -> {

                handleActionPoint(intent)
            }
            ACTION_NODES_START -> {

                if (BuildConfig.DEBUG) log("开始全局翻译")
                handleActionStartMultiNode()
            }

            ACTION_NODES_STOP -> {
                if (BuildConfig.DEBUG) log("结束全局翻译")
                handleActionStopMultiNode()
            }
            ACTION_DOWN -> {
                handleActionDown()
            }

        }
        return super.onStartCommand(intent, flags, startId)
    }

    private fun handleActionPoint(intent: Intent) {

        val x = intent.getIntExtra(EXTRA_POINT_X, -1)
        val y = intent.getIntExtra(EXTRA_POINT_Y, -1)
        val nodeInfo = rootInActiveWindow
        if (x != -1 && y != -1 && nodeInfo != null) {

            val nodeInfoList = ArrayList<AccessibilityNodeInfo>()

            findNodeByPoint(nodeInfo, x, y, nodeInfoList)
        }

    }

    /**
     * 开始多节点识别
     */
    private fun handleActionStartMultiNode() {
        handleActionStopMultiNode()

        val mapper1 = Function<Long, AccessibilityNodeInfo> {
            rootNodeInfo()
        }

        //包名发生变化，跳过,退出全局翻译
        val predicate = Predicate { rootNodeInfo: AccessibilityNodeInfo? ->
            val packageNameChanged = this.lastPackageName != null &&
                    rootNodeInfo?.packageName != this.lastPackageName

            if (packageNameChanged) {
                FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_LIGHT_ENTER)
            }

            !packageNameChanged
        }

        val mapper2 =
            Function<AccessibilityNodeInfo?, LinkedHashSet<AccessibilityNodeInfo>> { rootNodeInfo: AccessibilityNodeInfo? ->
                try {
                    findNodes(rootNodeInfo)
                } catch (e: Throwable) {//StackOverflowError
                    //e.printStackTrace()
                    linkedSetOf()
                }
            }
        val initialDelay = 0L
        /*multiDisposable = Flowable
            .interval(initialDelay, MILLIS_DELAY, TimeUnit.MILLISECONDS)
            .onBackpressureLatest()
            .map(mapper1)
            .takeWhile(predicate)
            .map(mapper2)
            .filter { nodeSet -> shouldRefresh(nodeSet) }
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .doOnNext { nodes ->
                postByEventBus(Message.obtain(null, EVENT_NODE_INFO_MULTI, nodes))
            }
            .subscribe({}, Throwable::printStackTrace)*/
        multiDisposable = Single.create<LinkedHashSet<AccessibilityNodeInfo>> {
            val info = rootNodeInfo()
            it.onSuccess(findNodes(info))
        }.subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(Consumer { nodes ->
                postByEventBus(Message.obtain(null, EVENT_NODE_INFO_MULTI, nodes))
                FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_LIGHT_ENTER)
            }, Consumer {
                it.printStackTrace()
                FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_LIGHT_ENTER)
            })
    }

    /**
     * 停止多节点识别
     */
    private fun handleActionStopMultiNode() {
        if (multiDisposable?.isDisposed == false) {
            multiDisposable?.dispose()
        }
        this.rootNodeInfo?.recycle()
        this.rootNodeInfo = null
        nodeRectSet.clear()
        this.lastPackageName = null
    }

//    /**
//     * 全局翻译过滤包名
//     */
//    private fun filterPackageName(nodeInfo: AccessibilityNodeInfo): Boolean {
//        val packageNames = arrayOf("com.android.systemui", "com.sec.android.app.launcher")
//
//        val packageName = nodeInfo.packageName
//
//        packageNames.count { it.contains(packageName) }
//    }

    /**
     * 识别单个节点
     */
    private fun handleActionDown() {

        val dis = Flowable.create<LinkedHashSet<AccessibilityNodeInfo>>({ emitter ->

            try {
                if (!emitter.isCancelled) {
                    val rootNodeInfo = rootNodeInfo()
                    emitter.onNext(findNodes(rootNodeInfo))
                }

            } catch (t: Throwable) {
                t.printStackTrace()
                if (!emitter.isCancelled) {
                    emitter.tryOnError(t)
                }
            } finally {
                if (!emitter.isCancelled) {
                    emitter.onComplete()
                }
            }

        }, BackpressureStrategy.LATEST)
            .subscribeOn(Schedulers.io())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe({ nodes ->
                RxRelay.post(Message.obtain(null, EVENT_NODE_INFO_DOWN, nodes))
            }, Throwable::printStackTrace)
    }

    private fun rootNodeInfo(): AccessibilityNodeInfo? = this.rootNodeInfo ?: rootInActiveWindow

    private fun findNodes(rootNodeInfo: AccessibilityNodeInfo?): LinkedHashSet<AccessibilityNodeInfo> {

        val nodes = LinkedHashSet<AccessibilityNodeInfo>()
        /*val rootNodeInfo = if (this.rootNodeInfo == null) {
            rootInActiveWindow
        } else {
            this.rootNodeInfo
        }*/

        this.rootNodeInfo = rootNodeInfo
        this.lastPackageName = rootNodeInfo?.packageName

        rootNodeInfo?.apply {
            val nodeInfo = AccessibilityNodeInfo.obtain(this)
            nodeCate(nodeInfo, nodes)
        }

        if (BuildConfig.DEBUG) println("nodes=${nodes.size} Thread=${Thread.currentThread().name}")
        return nodes
    }

    /**
     * 通过x,y寻找节点信息
     */
    private fun findNodeByPoint(
        nodeInfo: AccessibilityNodeInfo,
        x: Int,
        y: Int,
        nodeInfoList: ArrayList<AccessibilityNodeInfo>
    ) {
        val childCount = nodeInfo.childCount
        val viewIdResourceName = nodeInfo.viewIdResourceName
        //val text = nodeInfo.text

        // log("nodeInfo=$nodeInfo   \n=viewIdResourceName=$viewIdResourceName")
        for (index in 0 until childCount) {
            val child = nodeInfo.getChild(index) ?: continue
            return findNodeByPoint(child, x, y, nodeInfoList)
        }

        nodeInfo.getBoundsInScreen(pointNodeBound)

        if (pointNodeBound.contains(x, y)) {
            nodeInfoList.add(nodeInfo)
        }
    }

    /**
     * 节点分类
     */
    private fun nodeCate(
        nodeInfo: AccessibilityNodeInfo,
        nodeSet: LinkedHashSet<AccessibilityNodeInfo>
    ) {
        val childCount = nodeInfo.childCount
        val text = nodeInfo.text?.toString()
        val contentDescription = nodeInfo.contentDescription
        val className = nodeInfo.className
        val packageName = nodeInfo.packageName
        val isVisibleToUser = nodeInfo.isVisibleToUser
        val isContentInvalid = nodeInfo.isContentInvalid
        val viewIdResourceName = nodeInfo.viewIdResourceName

        if (isVisibleToUser) {
            when (className) {
                ImageView::class.java.name -> {
                    //nodeSet.add(nodeInfo)
                }
                ImageButton::class.java.name -> {
                    //nodeSet.add(nodeInfo)
                }
                WebView::class.java.name -> {
                    //nodeSet.add(nodeInfo)
                }
                else -> {
                    //println("contentDescription=$contentDescription  childCount=${nodeInfo.childCount}")
                    if (!TextUtils.isEmpty(text)) {
                        nodeSet.add(nodeInfo)
                    } else if (!TextUtils.isEmpty(contentDescription) && nodeInfo.childCount == 0) {
                        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                            val drawingOrder = nodeInfo.drawingOrder
                            if (drawingOrder > 0) {
                                nodeSet.add(nodeInfo)
                            }
                        } else {
                            nodeSet.add(nodeInfo)
                        }


                    }
                }
            }
        }

        if (BuildConfig.DEBUG) {
            log(
                "nodeInfo#className=$className\ttext=$text\t" +
                        "packageName=$packageName\tcontentDescription=$contentDescription\t" +
                        "isVisibleToUser=$isVisibleToUser\t" +
                        "isContentInvalid=$isContentInvalid\t" +
                        "viewIdResourceName=$viewIdResourceName"
            )
        }
        for (index in 0 until childCount) {
            val child = nodeInfo.getChild(index) ?: continue
            nodeCate(child, nodeSet)
        }
    }

    /* *******************全局翻译刷新处理　START*************************/
    /**
     * 是否应该刷新,原理，20%以上的节点信息坐标发生变化，则认为应该刷新
     */
    private fun shouldRefresh(nodeSet: LinkedHashSet<AccessibilityNodeInfo>): Boolean {

        val size = nodeSet.size

        val percent = 0.5f

        var progress = 0f

        val temp = hashSetOf<String>()

        for (info in nodeSet) {
            val rectInScreen = Rect()
            info.getBoundsInScreen(rectInScreen)
            val key = info.nodeKey(rectInScreen)
            if (nodeRectSet.contains(key)) {
                progress++
            }
            temp.add(key)

        }
        nodeRectSet.clear()
        nodeRectSet.addAll(temp)
        temp.clear()


        if (BuildConfig.DEBUG) println("progress====$progress   nodeRectSet.size=${nodeRectSet.size}  ${progress / nodeRectSet.size}")
        if (size == 0) return true

        if (progress / nodeRectSet.size < percent) {
            return true
        }
        return false
    }


    /* *******************全局翻译刷新处理　END*************************/

    override fun onGesture(gestureId: Int): Boolean {
        if (BuildConfig.DEBUG) log("gestureId=$gestureId")
        return super.onGesture(gestureId)
    }

    override fun onInterrupt() {
        if (BuildConfig.DEBUG) log("onInterrupt")
    }

    override fun onAccessibilityEvent(event: AccessibilityEvent) {

        //currentVersionCode = BuildConfig.VERSION_CODE

        val packageName = event?.packageName
        val className = event?.className

        //println("packageName--------------$packageName")

        /*if (!TextUtils.isEmpty(packageName)
            && packageName != currentPackageName
        ) {
            //if (BuildConfig.DEBUG) println("EVENT_PACKAGE_NAME_CHANGE=$currentPackageName  ${event?.action}   ${event?.contentChangeTypes}   ")
            currentPackageName = packageName
            val message = Message.obtain(null, EVENT_PACKAGE_NAME_CHANGE, currentPackageName)
            RxRelay.post(message)
        }*/

        if (event.getEventType() == AccessibilityEvent.TYPE_WINDOW_CONTENT_CHANGED) {
            if (!TextUtils.isEmpty(packageName) && !TextUtils.isEmpty(className)) {
                if (packageName != "com.android.systemui" && packageName != application.packageName) {
                    val message = Message.obtain(null, EVENT_ACTIVITY_CHANGE,  ActivityChangedEvent(packageName.toString(), className = className.toString()))
                    RxRelay.post(message)
                }
            }
        }

        isAccessibilityEvent = true
        this.rootNodeInfo = null

    }

    override fun onServiceConnected() {
        super.onServiceConnected()
        localBroadcastManager.sendBroadcast(Intent(BROADCAST_ACTION_SERVICE_CONNECTED))

        //val floatingVisibleState = floatingVisibleState(visible = false)

        //if (BuildConfig.DEBUG) log("onServiceConnected  $floatingVisibleState")
        isAccessibilityServiceRun = true
    }


    /**
     * Accessibility Service Unbind
     */
    override fun onUnbind(intent: Intent?): Boolean {
        //if (BuildConfig.DEBUG) log("onUnbind")
        isAccessibilityServiceRun = false

        localBroadcastManager.sendBroadcast(Intent(BROADCAST_ACTION_UN_BIND))

        if (applicationContext.canDrawOverlays()) {
            showUnbindDialog(applicationContext)
        }
        return false
    }

    override fun onRebind(intent: Intent?) {
        super.onRebind(intent)
        localBroadcastManager.sendBroadcast(Intent(BROADCAST_ACTION_RE_BIND))
        isAccessibilityServiceRun = true
        if (unBindingDialog?.isShowing == true) {
            unBindingDialog?.dismiss()
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        if (BuildConfig.DEBUG) log("onDestroy")
        isAccessibilityServiceRun = false

        localBroadcastManager.sendBroadcast(Intent(BROADCAST_ACTION_DESTROY))
    }


    private fun visibleFloating() {

        if (!localBroadcastManager.sendBroadcast(Intent(BROADCAST_ACTION_FLOATING_OPEN))
            && canDrawOverlays()
        ) {
            FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_LIGHT_ENTER)
        }
    }

    private fun invisibleFloating() {
        if (!localBroadcastManager.sendBroadcast(Intent(BROADCAST_ACTION_FLOATING_CLOSE))) {
            FloatingStateMachine.sendMessage(FloatingStateMachine.WHAT_INVISIBLE_ENTER)
        }
    }

    private fun floatingVisible(): Boolean {
        return FloatingManager.isFloatingVisible(this) && FloatingStateMachine.currentState() != FloatingStateMachine.FloatingState.InvisibleState
    }

    /**
     * 显示解除绑定 Dialog
     */
    private fun showUnbindDialog(context: Context) {

        if (unBindingDialog?.isShowing == true) {
            return
        }

        val onPositionClick =
            DialogInterface.OnClickListener { dialog: DialogInterface, which: Int ->

                //invisibleFloating()
            }
        unBindingDialog = AlertDialog
            .Builder(context)
            .setTitle(android.R.string.dialog_alert_title)
            .setMessage(R.string.text_access_unbind_alert)
            .setPositiveButton(android.R.string.ok, onPositionClick)
            .setCancelable(false)
            .create()
        val type = if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.O)
            WindowManager.LayoutParams.TYPE_APPLICATION_OVERLAY
        else WindowManager.LayoutParams.TYPE_PHONE
        unBindingDialog?.window?.setType(type)
        try {
            unBindingDialog?.show()
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }

}

fun AccessibilityNodeInfo.nodeKey(rectInScreen: Rect? = null): String {
    val childCount = this.childCount
    val text = this.text//?.toString()
    val contentDescription = this.contentDescription
    val className = this.className
    val packageName = this.packageName
    val isVisibleToUser = this.isVisibleToUser
    val isContentInvalid = this.isContentInvalid
    val viewIdResourceName = this.viewIdResourceName

    return "$childCount$text$contentDescription$className$packageName$isVisibleToUser$isContentInvalid$viewIdResourceName$rectInScreen"
}

/**
 * 节点文本内容
 */
fun AccessibilityNodeInfo.nodeText(): CharSequence {
    val text = this.text ?: ""
    val contentDescription = this.contentDescription ?: ""
    return if (!TextUtils.isEmpty(text)) {
        text
    } else {
        contentDescription
    }
}

///**
// * 保存悬浮视图状态，在无障碍服务被关闭时有用
// */
//fun Context.saveFloatState(visible: Boolean) {
//
//    getSharedPreferences("access_service", Context.MODE_PRIVATE).edit()
//        .putBoolean("visible", visible).apply()
//}
//
///**
// * 读取悬浮视图状态
// */
//fun Context.readFloatState(): Boolean {
//
//    return getSharedPreferences("access_service", Context.MODE_PRIVATE).getBoolean("visible", false)
//}