package com.talpa.translate.kv

import android.app.ActivityManager
import android.content.Context
import android.content.SharedPreferences
import cafe.adriel.pufferdb.android.AndroidPufferDB
import cafe.adriel.pufferdb.core.Puffer
import cafe.adriel.pufferdb.core.PufferDB

/**
 * @author CY 2020-05-31
 */
class KeyValue(name: String) {

    val pufferDb: Puffer by lazy {
        PufferDB.with(
            AndroidPufferDB.getInternalFile("$name.db")
        )
    }

    inline fun <reified T : Any> put(key: String, value: T) {
        try {
            val puffer = pufferDb
            Companion.put(key, value, puffer)
        } catch (e: Exception) {
            e.printStackTrace()
        }

    }

    inline fun <reified T : Any> get(key: String, defaultValue: T? = null): T? {
        val puffer = pufferDb
        return get(key, defaultValue, puffer)
    }


    companion object {

        private lateinit var application: Context

        fun init(context: Context) {
            application = context

            try {
                AndroidPufferDB.init(context)
            } catch (e: Exception) {
                e.printStackTrace()
            }
        }

        inline fun <reified T : Any> put(
            key: String,
            value: T,
            puffer: Puffer? = null
        ) {

            if (ActivityManager.isUserAMonkey()) {
                return
            }

            try {
                (puffer ?: AndroidPufferDB.withDefault()).put(key, value)
            } catch (e: Exception) {
                e.printStackTrace()
                val editor = defaultSharedPrefer()?.edit()
                when (T::class) {
                    String::class -> {
                        editor?.putString(key, value as String)?.apply()
                    }
                    Int::class -> {
                        editor?.putInt(key, value as Int)?.apply()
                    }
                    Float::class -> {
                        editor?.putFloat(key, value as Float)?.apply()
                    }
                    Boolean::class -> {
                        editor?.putBoolean(key, value as Boolean)?.apply()
                    }
                    Long::class -> {
                        editor?.putLong(key, value as Long)?.apply()
                    }
                }
            }
        }

        inline fun <reified T : Any> get(
            key: String,
            defaultValue: T? = null,
            puffer: Puffer? = null
        ): T? {
            if (ActivityManager.isUserAMonkey()) {
                return defaultValue
            }
            try {
                val pufferDb = puffer ?: AndroidPufferDB.withDefault()
                if (pufferDb.contains(key)) {
                    return pufferDb.get(key, defaultValue)
                } else if (defaultValue == null) {
                    return null
                }
            } catch (e: Exception) {
                e.printStackTrace()
                val prefer = defaultSharedPrefer()
                when (T::class) {
                    String::class -> {
                        return prefer?.getString(key, defaultValue as? String) as? T
                    }
                    Int::class -> {
                        return prefer?.getInt(key, defaultValue as? Int ?: 0) as? T
                    }
                    Float::class -> {
                        return prefer?.getFloat(key, defaultValue as? Float ?: 0f) as? T
                    }
                    Boolean::class -> {
                        return prefer?.getBoolean(key, defaultValue as? Boolean ?: false) as? T
                    }
                    Long::class -> {
                        return prefer?.getLong(key, defaultValue as? Long ?: 0L) as? T
                    }
                }
            }
            return defaultValue
        }


        fun defaultSharedPrefer(): SharedPreferences? {
            if (::application.isInitialized) {
                return application.getSharedPreferences("default_key_value", Context.MODE_PRIVATE)
            }
            return null
        }

    }

}

