package com.talpa

import android.app.Activity
import android.content.Context
import android.graphics.Bitmap
import android.graphics.Point
import android.graphics.Rect
import android.view.MotionEvent
import androidx.annotation.Keep
import androidx.annotation.RestrictTo
import com.talpa.translate.activity.CameraActivity
import com.talpa.translate.base.BaseApp
import com.talpa.translate.base.utils.PT_enter
import com.talpa.translate.base.utils.logEvent
import com.talpa.translate.ocr.OcrDispatcher
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.record.*

/**
 * Create by chenjunsheng on 2020/4/26
 */
@Keep
object TranslationController {

    const val SCREENSHOT_PATH = "screenshot_path"
    const val SCREENSHOT_ID = "screenshot_id"
    var serverAppKey: String? = null
    var serverAppSecret: String? = null
    var googleKey: String? = null
    var microsoftKey: String? = null

    private var initialized = false

    fun hasInitialized() = initialized

    /**
     * SDK初始化，要求传入serverAppKey和serverAppSecret
     * 可多次调用，建议在Application启动时调用
     * 方法内部仅做了赋值操作，没有其他耗时调用
     */
    @JvmStatic
    @JvmOverloads
    fun initialize(context: Context, serverAppKey : String, serverAppSecret: String,
                   googleKey : String? = null, microsoftKey: String? = null) {
        initialized = true
        BaseApp.initialize(context.applicationContext)
        this.serverAppKey = serverAppKey
        this.serverAppSecret = serverAppSecret
        this.googleKey = googleKey
        this.microsoftKey = microsoftKey
    }

    /**
     * 控制GoogleOcr功能是否可用
     */
    @JvmStatic
    fun googleOcrEnable(enable: Boolean) {
        check(initialized)
        OcrDispatcher.setGoogleOcrEnable(enable)
    }

    /**
     * 启动CameraActivity进行拍照翻译，
     * 接口调用前请先确认是否有相机权限
     * 请用这个接口启动
     * 不要用startActivity的方式启动CameraActivity
     */
    @JvmOverloads
    @JvmStatic
    fun startCameraCollection(context: Activity, requestCode: Int = 0) {
        check(initialized)
        CameraActivity.start(context, requestCode)
        logEvent(PT_enter)
    }

    /**
     * 图片翻译
     * 图片由外部传入
     * @param context
     * @param screenShotPath 截图地址，需要传绝对路径
     */
    @JvmStatic
    fun startTranslateFromUri(context: Context, screenShotPath: String, id: Long) {
        check(initialized)
        CameraActivity.start(
            context = context,
            requestCode = 0,
            screenShotPath = screenShotPath,
            id = id
        )
    }

    private var mCaptureManager: ICaptureController? = null
    private var mTextRecognizeListener: TextRecognizeListener? = null

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    internal fun getCaptureManager() = mCaptureManager

    /**
     * 开始超级翻译
     * 首次调用会询问录屏权限，若权限存在，则返回截屏结果。
     * 三方可将结果保存下来，用于后续识别
     */
    @JvmStatic
    fun captureFrameSync(context: Context) : CaptureResult {
        check(initialized)
        if (mCaptureManager == null) {
            mCaptureManager = CaptureManager(context).apply {
                mTextRecognizeListener?.let {
                    setTextRecognizeListener(it)
                }
            }
        }
        return mCaptureManager!!.captureFrameSync(context)
    }


    fun setTextRecognizeListener(listener: TextRecognizeListener) {
        check(initialized)
        mTextRecognizeListener = listener
        mCaptureManager?.setTextRecognizeListener(listener)
    }

    /**
     *
     */
    @JvmStatic
    suspend fun doTextRecognize(point: Point, captureResult: CaptureResult) : OcrResult? {
        check(initialized)
        return doTextRecognize(point.x, point.y, captureResult)
    }

    @JvmStatic
    suspend fun doTextRecognize(x: Int, y: Int, captureResult: CaptureResult) : OcrResult? {
        check(initialized)
        check(captureResult.getStatus() == CAPTURE_BITMAP_SUCCESS) {
            "check fail capture result:${captureResult.getStatus()}"
        }
        var index = -1
        captureResult.rects?.let { rects ->
            rects.forEachIndexed { i, rect ->
                if (rect.contains(x, y)) {
                    index = i
                    return@let
                }
            }
        }
        if (index == -1) return null
        return mCaptureManager!!.doTextRecognize(index, captureResult)
    }

    /**
     * 销毁录屏服务，调用后不应该再调用其他录屏相关接口
     */
    @JvmStatic
    fun destroyRecognizeService() {
        check(initialized)
        mCaptureManager!!.destroyRecognizeService()
        mCaptureManager = null
    }

    @Keep
    interface TextRecognizeListener {

        fun onRecognizeServiceReady()

        fun onRecognized(index: Int, result: CaptureTransResult)

        fun onRecognizedFail(rect: Rect)

        /**
         * 识别区域确定时回调
         * @param rect 指示区域，起始坐标为相对截图坐标起始点的距离
         */
        fun onAreaConfirmed(rect: Rect)

        fun getTransTargetLanguage() : String

    }

    /**
     * 将触摸事件委托给SDK
     * 翻译结果通过 [TextRecognizeListener] 回调返回
     */
    @JvmStatic
    fun delegateRecTouchEvent(e: MotionEvent) {
        check(initialized)
        if (mCaptureManager == null) {
            mCaptureManager = CaptureManager(BaseApp.appContext)
                .apply {
                    mTextRecognizeListener?.let {
                        setTextRecognizeListener(it)
                    }
                }
        }
        mCaptureManager!!.delegateRecTouchEvent(e)
    }

    var mLastBitmap: Bitmap? = null
}