package com.talpa.translate.base.utils

import android.content.Context
import android.content.pm.PackageManager
import android.content.pm.Signature
import android.graphics.Rect
import android.net.ConnectivityManager
import android.os.Environment
import android.os.StatFs
import android.text.Layout
import android.text.StaticLayout
import android.text.TextPaint
import android.util.TypedValue
import androidx.core.os.LocaleListCompat
import com.google.common.io.BaseEncoding
import java.security.MessageDigest
import java.security.NoSuchAlgorithmException
import java.util.*


/**
 * Create by chenjunsheng on 2020/4/27
 */

const val PREFER_KEY_SOURCE_LANG = "key_source_lang"
const val PREFER_NAME = "prefer_camera"

const val MAX_TEXT_SIZE = 16
const val MIN_TEXT_SIZE = 4

/**
 * Save Language Tag
 */
fun saveLanguage(
    context: Context,
    preferKey: String = PREFER_KEY_SOURCE_LANG,
    value: String
) {
    val prefer = context.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE)
    prefer.edit().putString(preferKey, value).apply()
    //val mmkv = MMKV.defaultMMKV()
    //mmkv.encode(preferKey, value)
}

fun Context.isNetworkConnected(): Boolean {
    val connectivityManager =
        getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

    val networkInfo = connectivityManager.activeNetworkInfo
    return networkInfo?.isAvailable == true && networkInfo.isConnected
}

fun crypt(str: String): String {
    if (str.isEmpty()) {
        return ""
    }
    val hexString = StringBuffer()
    try {
        val md = MessageDigest.getInstance("MD5")
        md.update(str.toByteArray())

        val hash = md.digest()

        repeat(hash.size) {
            if ((0xff and hash[it].toInt()) < 0x10) {
                hexString.append("0" + Integer.toHexString((0xFF and hash[it].toInt())))
            } else {
                hexString.append(Integer.toHexString(0xFF and hash[it].toInt()))
            }
        }
    } catch (e: NoSuchAlgorithmException) {
        e.printStackTrace()
    }
    return hexString.toString()
}

fun isSDCardEnable(): Boolean {
    return Environment.MEDIA_MOUNTED == Environment.getExternalStorageState()
}

fun getFreeSpace(): Long {
    if (!isSDCardEnable()) {
        return -1
    }
    val stat = StatFs(Environment.getExternalStorageDirectory().path)
    val blockSize = stat.blockSizeLong
    val availableBlocks = stat.availableBlocksLong
    val size = availableBlocks * blockSize / 1024L
    return size
}

fun getLocaleListCompat(vararg languages: String): LocaleListCompat {
    val supportList = StringBuilder()
        .also { builder ->
            languages.forEach {
                builder.append(it)
                builder.append(",")
            }
        }.toString()
    return LocaleListCompat.forLanguageTags(getOrderedLanguageList(Locale.getDefault().language, supportList))

}

/**
 * Gets the SHA1 signature, hex encoded for inclusion with Google Cloud Platform API requests
 *
 * @param packageName Identifies the APK whose signature should be extracted.
 * @return a lowercase, hex-encoded
 */
fun getSignature(pm: PackageManager, packageName: String): String? {
    return try {
        val packageInfo = pm.getPackageInfo(packageName, PackageManager.GET_SIGNATURES)
        if (packageInfo?.signatures == null || packageInfo.signatures.isEmpty() || packageInfo.signatures[0] == null
        ) {
            null
        } else
            signatureDigest(packageInfo.signatures[0])
    } catch (e: PackageManager.NameNotFoundException) {
        null
    }
}

private fun signatureDigest(sig: Signature): String? {
    val signature = sig.toByteArray()
    return try {
        val md = MessageDigest.getInstance("SHA1")
        val digest = md.digest(signature)
        BaseEncoding.base16().lowerCase().encode(digest)
    } catch (e: NoSuchAlgorithmException) {
        null
    }
}

fun getAutofitTextSize(text: String, paint: TextPaint, rect: Rect, maxTextSize: Int, context: Context) : Int {

    var textsize = maxTextSize
    val displayMetrics = context.resources.displayMetrics
    paint.textSize = TypedValue.applyDimension(
        TypedValue.COMPLEX_UNIT_SP, textsize.toFloat(),
        displayMetrics
    )

    var staticLayout = StaticLayout(text, paint, rect.width(),
        Layout.Alignment.ALIGN_NORMAL,
        1.0f,
        0.0f,
        true)


    while (staticLayout.height > rect.height()) {
        if (textsize <= MIN_TEXT_SIZE) {
            break
        }
        textsize --
        paint.textSize = TypedValue.applyDimension(
            TypedValue.COMPLEX_UNIT_SP, textsize.toFloat(),
            displayMetrics
        )
        staticLayout = StaticLayout(text, paint, rect.width(),
            Layout.Alignment.ALIGN_NORMAL,
            1.0f,
            0.0f,
            true)
    }
    return textsize
}