package com.talpa.translate.test

import android.content.Intent
import android.graphics.Bitmap
import android.graphics.ImageFormat
import android.net.Uri
import android.os.Bundle
import android.view.View
import androidx.appcompat.app.AppCompatActivity
import androidx.lifecycle.lifecycleScope
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.photo.translation.R
import com.talpa.translate.camera.CameraFragment
import com.talpa.translate.camera.view.CameraListener
import com.talpa.translate.camera.view.CameraView
import com.talpa.translate.camera.view.PictureResult
import com.talpa.translate.camera.view.controls.*
import com.talpa.translate.camera.view.gesture.Gesture
import com.talpa.translate.camera.view.gesture.GestureAction
import com.talpa.translate.camera.view.markers.DefaultAutoFocusMarker
import kotlinx.android.synthetic.main.test_layout.*
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.launch
import kotlinx.coroutines.withContext

/**
 * Create by chenjunsheng on 2020/8/18
 */
class TestActivity : AppCompatActivity() {

    companion object {
        init {
            System.loadLibrary("c++_shared")
            System.loadLibrary("imgprocessor")
        }
    }

    private lateinit var mCamera: CameraView

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        setContentView(R.layout.test_layout)

        mCamera = camera
            .apply {
                engine = Engine.CAMERA2
                setExperimental(true)
                preview = Preview.GL_SURFACE
                playSounds = false
                grid = Grid.OFF
                flash = Flash.OFF
                audio = Audio.OFF
                facing = Facing.BACK
                mapGesture(Gesture.TAP, GestureAction.AUTO_FOCUS)
                mapGesture(Gesture.LONG_TAP, GestureAction.NONE)
                mapGesture(Gesture.PINCH, GestureAction.ZOOM)
                mode = Mode.PICTURE
                setAutoFocusMarker(DefaultAutoFocusMarker())
                useDeviceOrientation = false
                frameProcessingFormat = ImageFormat.YUV_420_888
            }
        //parent.addView(mCamera, FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT))
        mCamera.setLifecycleOwner(this)
        mCamera.addCameraListener(object : CameraListener() {
            override fun onPictureTaken(result: PictureResult) {
                mCamera.visibility = View.INVISIBLE
                lifecycleScope.launch {
                    val bitmap = withContext(Dispatchers.IO) {
                        Glide.with(this@TestActivity)
                            .asBitmap()
                            .load(result.data)
                            .submit()
                            .get()
                    }
                    imageView3.setImageBitmap(bitmap)
                    val dealBitmap = withContext(Dispatchers.IO) {
                        dealPicture(bitmap)
                    }
                    imageView4.setImageBitmap(dealBitmap)
                }

                imageView4.visibility = View.VISIBLE
                imageView3.visibility = View.VISIBLE

                imageView3.setOnClickListener {
                    if (imageView4.visibility == View.VISIBLE) {
                        imageView4.visibility = View.INVISIBLE
                    } else {
                        imageView4.visibility = View.VISIBLE
                    }
                }

                imageView4.setOnClickListener {
                    if (it.visibility == View.VISIBLE) {
                        it.visibility = View.INVISIBLE
                    }
                }
            }
        })

        button.setOnClickListener { mCamera.takePictureSnapshot() }

        button2.setOnClickListener {
            val intent = Intent(Intent.ACTION_PICK)

            intent.type = "image/*"

            startActivityForResult(
                intent,
                CameraFragment.REQUEST_CODE_SELECT_IMAGE
            )
        }
    }

    private fun setImageFromGallery(uri: Uri) {
        mCamera.close()
        mCamera.visibility = View.INVISIBLE
        imageView4.visibility = View.VISIBLE
        imageView3.visibility = View.VISIBLE

        imageView3.setOnClickListener {
            if (imageView4.visibility == View.VISIBLE) {
                imageView4.visibility = View.INVISIBLE
            } else {
                imageView4.visibility = View.VISIBLE
            }
        }

        imageView4.setOnClickListener {
            if (it.visibility == View.VISIBLE) {
                it.visibility = View.INVISIBLE
            }
        }
        Glide.with(this)
            .asBitmap()
            .load(uri)
            .listener(object : RequestListener<Bitmap> {
                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Bitmap>?,
                    isFirstResource: Boolean
                ): Boolean {
                    return false
                }

                override fun onResourceReady(
                    resource: Bitmap,
                    model: Any?,
                    target: Target<Bitmap>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    lifecycleScope.launchWhenCreated {
                        val dealBitmap = withContext(Dispatchers.Default) {
                            dealPicture(resource)
                        }
                        imageView4.setImageBitmap(dealBitmap)
                    }
                    return false
                }

            })
            .into(imageView3)
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)
        when (requestCode) {
            CameraFragment.REQUEST_CODE_SELECT_IMAGE -> {
                val uri = data?.data
                //println("uri===$uri")

                if (uri != null) {
                    setImageFromGallery(uri)
                }

            }
        }
    }

    external fun dealPicture(bitmap: Bitmap) : Bitmap
}