package com.talpa.translate.activity

import android.app.Activity
import android.content.Context
import android.content.Intent
import android.graphics.Bitmap
import android.net.Uri
import android.os.Bundle
import android.provider.MediaStore
import android.view.View
import android.widget.Toast
import androidx.activity.result.contract.ActivityResultContract
import androidx.appcompat.app.AppCompatActivity
import androidx.constraintlayout.motion.widget.MotionLayout
import androidx.core.view.doOnPreDraw
import androidx.lifecycle.Observer
import androidx.lifecycle.ViewModelProvider
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.photo.translation.R
import com.photo.translation.databinding.GalleryPickerActivityBinding
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.GR_gallery_picker_submit
import com.talpa.translate.base.utils.logEvent
import com.talpa.translate.base.view.drag.DragSelectTouchListener
import com.talpa.translate.base.view.drag.DragSelectionProcessor
import com.talpa.translate.camera.PickerParagraphAdapter
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.camera.view.controls.PictureFormat
import com.talpa.translate.ocr.ImageTranslate
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.ocr.result.OcrTransferWrapper

/**
 * Create by chenjunsheng on 2021/3/19
 */
class GalleryPickerActivity : AppCompatActivity(), View.OnClickListener {

    private val pickPhoto = registerForActivityResult(PickPhoto()) { result ->
        result?.let {
            doOcrOnly(it)
        } ?: kotlin.run {
            finish()
        }
    }
    internal val translateViewModel: ImageTranslate by lazy {
        ViewModelProvider(
            this,
            ViewModelProvider.AndroidViewModelFactory.getInstance(application)
        ).get(ImageTranslate::class.java)
    }

    private fun doOcrOnly(uri: Uri) {
        Glide.with(this)
            .asBitmap()
            .load(uri)
            .listener(object : RequestListener<Bitmap> {
                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Bitmap>?,
                    isFirstResource: Boolean
                ): Boolean {
                    return false
                }

                override fun onResourceReady(
                    resource: Bitmap,
                    model: Any?,
                    target: Target<Bitmap>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    binding.loading.root.visibility = View.VISIBLE
                    binding.ocrResult.root.visibility = View.VISIBLE
                    translateViewModel.doOcrOnly(
                        resource, FrameMetadata.Builder()
                            .setWidth(resource.width)
                            .setHeight(resource.height)
                            .setCameraFacing(Facing.BACK)
                            .setPictureFormat(PictureFormat.JPEG)
                            .setRotation(0)
                            .build()
                    ).observe(this@GalleryPickerActivity, Observer {
                        binding.loading.root.visibility = View.GONE
                        if (it.isSuccess) {
                            val ocrResult = it.getOrNull()
                            if (ocrResult != null) {
                                ocrOnlySuccess(ocrResult)
                            } else {
                                ocrOnlyFail()
                            }
                        } else {
                            ocrOnlyFail()
                        }
                    })
                    return false
                }

            })
            .into(binding.ocrResult.ivPreview)
    }

    internal fun ocrOnlySuccess(ocrResult: OcrResult) {
        val adapter = PickerParagraphAdapter(ocrResult)
        // 2) Add the DragSelectListener
        val dragSelectionProcessor = DragSelectionProcessor(object :
            DragSelectionProcessor.ISelectionHandler {

            override fun getSelection(): MutableSet<Int> {
                return adapter.getSelection()
            }

            override fun isSelected(index: Int): Boolean {
                return adapter.getSelection().contains(index)
            }

            override fun updateSelection(
                start: Int,
                end: Int,
                isSelected: Boolean,
                calledFromOnStart: Boolean
            ) {
                adapter.selectRange(start, end, isSelected)
                binding.ocrResult.submit.isEnabled = adapter.getSelection().isNotEmpty()

            }
        })
            .withMode(DragSelectionProcessor.Mode.Simple)
        var dragSelectTouchListener = DragSelectTouchListener()
            .withSelectListener(dragSelectionProcessor)
        adapter.setClickListener(object : PickerParagraphAdapter.ItemClickListener {
            override fun onItemClick(view: View, position: Int) {
                adapter.toggleSelection(position)
                binding.ocrResult.submit.isEnabled = adapter.getSelection().isNotEmpty()
            }

            override fun onItemLongClick(view: View, position: Int): Boolean {
                // if one item is long pressed, we start the drag selection like following:
                // we just call this function and pass in the position of the first selected item
                // the selection processor does take care to update the positions selection mode correctly
                // and will correctly transform the touch events so that they can be directly applied to your adapter!!!
                dragSelectTouchListener.startDragSelection(position)
                return true
            }
        })
        binding.ocrResult.recyclerView.addOnItemTouchListener(dragSelectTouchListener)
        binding.ocrResult.recyclerView.adapter = adapter
        binding.ocrResult.recyclerView.doOnPreDraw {
            adapter.selectAll()
            binding.ocrResult.submit.isEnabled = adapter.getSelection().isNotEmpty()
        }
        binding.ocrResult.root.setTransitionListener(null)
        binding.ocrResult.root.transitionToEnd()
        binding.ocrResult.toolbar.setNavigationOnClickListener {
            binding.ocrResult.root.setTransitionListener(object : MotionLayout.TransitionListener {
                override fun onTransitionStarted(p0: MotionLayout?, p1: Int, p2: Int) {

                }

                override fun onTransitionChange(p0: MotionLayout?, p1: Int, p2: Int, p3: Float) {
                }

                override fun onTransitionCompleted(p0: MotionLayout?, p1: Int) {
                    finish()
                }

                override fun onTransitionTrigger(
                    p0: MotionLayout?,
                    p1: Int,
                    p2: Boolean,
                    p3: Float
                ) {
                }

            })
            binding.ocrResult.root.transitionToStart()
        }
    }

    internal fun ocrOnlyFail() {
        Toast.makeText(this, R.string.no_content_identified, Toast.LENGTH_SHORT).show()
        setResult(Activity.RESULT_CANCELED)
        finish()
    }

    private lateinit var binding: GalleryPickerActivityBinding

    override fun onCreate(savedInstanceState: Bundle?) {
        super.onCreate(savedInstanceState)
        binding = GalleryPickerActivityBinding.inflate(layoutInflater)
        setContentView(binding.root)
        pickPhoto.launch(null)
        binding.ivFinish.setOnClickListener(this)
        binding.ocrResult.submit.setOnClickListener(this)
    }

    override fun onClick(v: View) {
        when (v.id) {
            R.id.iv_finish -> {
                setResult(Activity.RESULT_CANCELED)
                finish()
            }
            R.id.submit -> {
                val adapter = binding.ocrResult.recyclerView.adapter as PickerParagraphAdapter
                val blocks = adapter.getSelection().map {
                    adapter.getItem(it)
                }.toTypedArray()
                val bundle = Bundle()
                bundle.putBinder(OCR_RESULT, OcrTransferWrapper(blocks))
                setResult(Activity.RESULT_OK, Intent().apply {
                    putExtra(OCR_RESULT, bundle)
                })
                logEvent(GR_gallery_picker_submit)
                finish()
            }
        }
    }
}

class PickPhoto : ActivityResultContract<Void?, Uri?>() {
    override fun createIntent(context: Context, input: Void?): Intent {
        return Intent(Intent.ACTION_PICK).setType(MediaStore.Images.Media.CONTENT_TYPE)
    }

    override fun parseResult(resultCode: Int, intent: Intent?): Uri? {
        return if (intent == null || resultCode != Activity.RESULT_OK) null else intent.data
    }
}