package com.talpa.translate.record

import android.content.ComponentName
import android.content.Context
import android.content.Intent
import android.content.ServiceConnection
import android.graphics.Bitmap
import android.graphics.Rect
import android.os.IBinder
import android.util.Log
import android.view.MotionEvent
import android.view.VelocityTracker
import androidx.annotation.RestrictTo
import com.talpa.TranslationController
import com.talpa.translate.activity.ScreenRecordActivity
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.camera.view.controls.PictureFormat
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.OcrDispatcher
import com.talpa.translate.ocr.Recognizer
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.service.RecordService
import kotlinx.coroutines.*
import java.util.*
import kotlin.math.abs
import kotlin.math.max
import kotlin.math.min
import kotlin.math.sqrt


/**
 * Create by chenjunsheng on 2020/8/5
 */
class CaptureManager(context: Context) : ServiceConnection, ICaptureController {

    companion object {

        const val RECT_GAP = 12
        const val MIN_DISTANCE = 15

        const val STATE_INITIALIZE = 0
        const val STATE_ASKING_PERMISSION = 1
        const val STATE_PERMISSION_GRANTED = 2
        const val STATE_SERVICE_PREPARED = 3
        const val STATE_CAPTURING_PIC = 4
        const val STATE_CAPTURED_PIC = 5
        const val STATE_RECOGNIZE_START = 6
        const val STATE_RECOGNIZING = 7
        const val STATE_TRANSLATING = 8
        const val STATE_RECOGNIZED_FINISH = 9
        const val STATE_TRANSLATE_FINISH = 10
        const val STATE_ACTION_CLEAR = 11

        private const val VELOCITY_TIME_MILLIS = 300

        /**
         * 最小速率
         */
        private const val MIN_VELOCITY = 80
    }

    private var mResultListener: TranslationController.TextRecognizeListener? = null
    private var mRecordService: IRecordService? = null
    private val mTranslator: Recognizer
    private val mContext = context.applicationContext
    private var mCurrentState = STATE_INITIALIZE

    init {
        mTranslator = OcrDispatcher.dispatchOcrTranslator(mContext, Locale.getDefault().language)
    }

    override fun setTextRecognizeListener(listener: TranslationController.TextRecognizeListener) {
        mResultListener = listener
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    override fun setCurrentState(state: Int) {
        mCurrentState = state
        when (mCurrentState) {
            STATE_INITIALIZE -> {
                Log.d("cjslog", "STATE_INITIALIZE")
            }
            STATE_ASKING_PERMISSION -> {
                Log.d("cjslog", "STATE_ASKING_PERMISSION")
            }
            STATE_PERMISSION_GRANTED -> {
                Log.d("cjslog", "STATE_PERMISSION_GRANTED")
            }
            STATE_SERVICE_PREPARED -> {
                Log.d("cjslog", "STATE_SERVICE_PREPARED")
            }
            STATE_CAPTURING_PIC -> {
                Log.d("cjslog", "STATE_CAPTURING_PIC")
            }
            STATE_CAPTURED_PIC -> {
                Log.d("cjslog", "STATE_CAPTURED_PIC")
            }
            STATE_RECOGNIZE_START -> {
                Log.d("cjslog", "STATE_RECOGNIZE_START")
            }
            STATE_RECOGNIZING -> {
                Log.d("cjslog", "STATE_RECOGNIZING")
            }
            STATE_RECOGNIZED_FINISH -> {
                Log.d("cjslog", "STATE_RECOGNIZED_FINISH")
            }
            STATE_TRANSLATE_FINISH -> {
                Log.d("cjslog", "STATE_TRANSLATE_FINISH")
            }
            STATE_ACTION_CLEAR -> {
                Log.d("cjslog", "STATE_ACTION_CLEAR")
            }
        }
    }

    override fun captureFrameSync(context: Context): CaptureResult {
        if (mRecordService == null) {
            ScreenRecordActivity.start(context)
            //RequestPermissionActivity.start(context)
            return CaptureResult(START_CAPTURE_PERMISSION_REQUEST)
        } else {
            val result = try {
                mRecordService!!.captureFrameSync()
            } catch (e: Exception) {
                e.printStackTrace()
                null
            } ?: return CaptureResult(PROCESSOR_BITMAP_FAIL)
            fastAnalyzeArea(result)
            return result
        }
    }

    private fun isRectangleOverlap(current: Rect, other: Rect): Boolean {
        return current.left < other.right
                && other.left < current.right
                && current.top < other.bottom
                && other.top < current.bottom
    }

    private fun fastAnalyzeArea(result: CaptureResult) {
        if (result.bitmap == null) {
            return
        }
        val list = result.rects?.toMutableList() ?: return
        if (list.isEmpty()) return
        list.forEach {
            //1.先向外扩张
            it.inset(-RECT_GAP, -RECT_GAP)
            //2.检查是否越界
            it.left = max(0, it.left)
            it.top = max(0, it.top)
            it.right = min(result.bitmap!!.width, it.right)
            it.bottom = min(result.bitmap!!.height, it.bottom)
        }
        //3.检查是否重叠
        var i = 0
        while (i < list.size) {
            var j = i + 1
            while (j < list.size) {
                if (isRectangleOverlap(current = list[i], other = list[j])) {
                    list[i].union(list[j])
                    list.removeAt(j)
                    j--
                }
                j++
            }
            i++
        }
        result.rects = list
    }

    @RestrictTo(RestrictTo.Scope.LIBRARY_GROUP)
    override fun startRecordService(context: Context, resultCode: Int, data: Intent) {
        val serviceIntent = Intent(context, RecordService::class.java)
        serviceIntent.putExtra(RecordService.PROJECT_INTENT, data)
        serviceIntent.putExtra(RecordService.PROJECT_CODE, resultCode)
        context.applicationContext.bindService(serviceIntent, this, Context.BIND_AUTO_CREATE)
        //context.startForegroundService(serviceIntent)
        //Log.d("cjslog", "start bind service")
    }

    override suspend fun doTextRecognize(index: Int, captureResult: CaptureResult): OcrResult? {
        val rect = captureResult.rects!![index]
        mTranslator.setup(
            TransParams(
                FrameMetadata.Builder()
                    .setWidth(rect.width())
                    .setHeight(rect.height())
                    .setCameraFacing(Facing.BACK)
                    .setPictureFormat(PictureFormat.JPEG)
                    .setRotation(0)
                    .build(), Locale.getDefault().language
            )
        )
        return try {
            mTranslator.doOcr(
                Bitmap.createBitmap(
                    captureResult.bitmap!!,
                    rect.left,
                    rect.top,
                    rect.width(),
                    rect.height()
                )
            )
        } catch (e: Exception) {
            e.printStackTrace()
            null
        }
    }

    override fun destroyRecognizeService() {
        mRecordService?.destroyService()
        mRecordService = null
    }

    override fun onServiceDisconnected(name: ComponentName?) {
        //mRecordServiceMessenger = null
        mRecordService = null
    }

    override fun onServiceConnected(name: ComponentName?, service: IBinder) {
        mRecordService = IRecordService.Stub.asInterface(service)
            .also {
                try {
                    it.registerListener(mRemoteCallback)
                } catch (e: Exception) {
                    e.printStackTrace()
                }
            }
        //mRecordServiceMessenger = Messenger(service)
        //mRecordServiceMessenger?.send(getMessage(mClientMessenger, SEND_CLIENT_MESSENGER))
        setCurrentState(STATE_SERVICE_PREPARED)
    }

    private var mLastX = 0f
    private var mLastY = 0f
    private var mCaptureJob: Job? = null
    private var mRecognizeJob: Job? = null
    private var mLastResult: CaptureTransResult? = null
    private var velocityTracker: VelocityTracker? = null

    override fun delegateRecTouchEvent(e: MotionEvent) {
        if (velocityTracker == null) {
            velocityTracker = VelocityTracker.obtain()
        }
        //Log.d("cjslog", "original2:${e.x} ${e.y}")
        velocityTracker!!.addMovement(e)

        when (e.action) {
            MotionEvent.ACTION_DOWN -> {
                mLastX = e.rawX
                mLastY = e.rawY
                if (mRecordService != null) {
                    mRecognizeJob?.cancel()
                    mCaptureJob?.cancel()
                    setCurrentState(STATE_SERVICE_PREPARED)
                }
            }
            MotionEvent.ACTION_MOVE -> {
                when (mCurrentState) {
                    STATE_INITIALIZE -> {
                        val dx = abs(e.rawX - mLastX)
                        val dy = abs(e.rawY - mLastY)
                        if (sqrt(dx * dx + dy * dy) > MIN_DISTANCE) {
                            Log.d("cjslog", "asking permission")
                            captureFrameSync(mContext)
                            setCurrentState(STATE_ASKING_PERMISSION)
                        }
                    }
                    STATE_SERVICE_PREPARED -> {
                        val dx = abs(e.rawX - mLastX)
                        val dy = abs(e.rawY - mLastY)
                        if (sqrt(dx * dx + dy * dy) > MIN_DISTANCE) {
                            Log.d("cjslog", "start capture picture")
                            setCurrentState(STATE_CAPTURING_PIC)
                            mCaptureJob?.cancel()
                            mCaptureJob = CoroutineScope(Dispatchers.IO).launch {
                                val captureResult = captureFrameSync(mContext)
                                mLastResult = CaptureTransResult(captureResult)
                                    .also {
                                        it.ocrResultMap = hashMapOf()
                                        it.transMap = hashMapOf()
                                    }
                                Log.d(
                                    "cjslog",
                                    "picture capture finish:${mLastResult?.captureResult?.bitmap}" +
                                            " ${mLastResult?.captureResult?.rects}"
                                )

                                withContext(Dispatchers.Main) {
                                    //val intent = Intent(mContext, Test2Activity::class.java)
                                    //intent.putExtra("bitmap", mLastResult!!.bitmap!!)
                                    //mLastBitmap = mLastResult!!.bitmap
                                    //intent.addFlags(Intent.FLAG_ACTIVITY_NEW_TASK)
                                    //mContext.startActivity(intent)
                                    setCurrentState(STATE_CAPTURED_PIC)
                                }
                            }
                        }
                    }
                    STATE_CAPTURED_PIC -> {
                        velocityTracker!!.computeCurrentVelocity(VELOCITY_TIME_MILLIS)
                        val xVelocity = velocityTracker!!.xVelocity
                        val yVelocity = velocityTracker!!.yVelocity
                        if (xVelocity < MIN_VELOCITY && yVelocity < MIN_VELOCITY) {
                            mLastResult?.captureResult?.let { result ->
                                var index = -1
                                result.rects?.let letInner@{ rects ->
                                    rects.forEachIndexed { i, rect ->
                                        if (rect.contains(e.rawX.toInt(), e.rawY.toInt())) {
                                            index = i
                                            return@letInner
                                        }
                                    }
                                }
                                if (index != -1) {
                                    mLastResult!!.workingIndex = index
                                    setCurrentState(STATE_RECOGNIZE_START)
                                    mResultListener?.onAreaConfirmed(result.rects!![index])
                                    Log.d(
                                        "cjslog",
                                        "recognize picture start: ${result.rects!![index]}"
                                    )
                                    mRecognizeJob?.cancel()
                                    mRecognizeJob = CoroutineScope(Dispatchers.Main).launch {
                                        setCurrentState(STATE_RECOGNIZING)
                                        val ocrResult = withContext(Dispatchers.IO) {
                                            doTextRecognize(index, result)
                                        }
                                        setCurrentState(STATE_RECOGNIZED_FINISH)
                                        if (ocrResult != null) {
                                            mLastResult!!.ocrResultMap!![index] = ocrResult

                                            val targetLanguage =
                                                mResultListener?.getTransTargetLanguage()
                                                    ?: Locale.ENGLISH.language
                                            val list = arrayListOf<String>()
                                            ocrResult.blocks.forEach {
                                                list.add(it.text)
                                            }
                                            setCurrentState(STATE_TRANSLATING)

                                            val transResult = withContext(Dispatchers.IO) {
                                                try {
                                                    HiTranslator.getInstance(mContext).postTranslate(
                                                        Locale.getDefault().language,
                                                        targetLanguage, list
                                                    )
                                                } catch (e: Exception) {
                                                    null
                                                }
                                            }

                                            mLastResult!!.workingIndex = -1
                                            mLastResult!!.transMap!![index] = transResult
                                            mResultListener?.onRecognized(index, mLastResult!!)
                                            if (mCurrentState == STATE_ACTION_CLEAR) {
                                                setCurrentState(STATE_SERVICE_PREPARED)
                                                mLastResult = null
                                            } else {
                                                setCurrentState(STATE_TRANSLATE_FINISH)
                                            }
                                        } else {
                                            if (mCurrentState == STATE_ACTION_CLEAR) {
                                                setCurrentState(STATE_SERVICE_PREPARED)
                                                mLastResult = null
                                            } else {
                                                setCurrentState(STATE_TRANSLATE_FINISH)
                                            }
                                            mResultListener?.onRecognizedFail(result.rects!![index])
                                        }
                                    }
                                }
                            }
                        }
                    }
                    STATE_RECOGNIZED_FINISH, STATE_TRANSLATING, STATE_TRANSLATE_FINISH -> {
                        mLastResult?.let { transResult ->
                            velocityTracker!!.computeCurrentVelocity(VELOCITY_TIME_MILLIS)
                            val xVelocity = velocityTracker!!.xVelocity
                            val yVelocity = velocityTracker!!.yVelocity
                            if (xVelocity < MIN_VELOCITY && yVelocity < MIN_VELOCITY) {
                                transResult.captureResult.let { captureResult ->
                                    var index = -1
                                    captureResult.rects?.let letInner@{ rects ->
                                        rects.forEachIndexed { i, rect ->
                                            if (rect.contains(e.rawX.toInt(), e.rawY.toInt())) {
                                                index = i
                                                return@letInner
                                            }
                                        }
                                    }

                                    if (index != -1 && transResult.workingIndex != index) {
                                        transResult.workingIndex = index
                                        mResultListener?.onAreaConfirmed(captureResult.rects!![index])
                                        //setCurrentState(STATE_TRANSLATING)
                                        val targetLanguage =
                                            mResultListener?.getTransTargetLanguage()
                                                ?: Locale.ENGLISH.language
                                        if (transResult.ocrResultMap != null) {
                                            val ocrR = transResult.ocrResultMap!![index]
                                            if (ocrR != null) {
                                                mRecognizeJob?.cancel()
                                                mRecognizeJob =
                                                    CoroutineScope(Dispatchers.Main).launch {
                                                        setCurrentState(STATE_TRANSLATING)
                                                        val list = arrayListOf<String>()
                                                        ocrR.blocks.forEach {
                                                            list.add(it.text)
                                                        }

                                                        val transR =
                                                            withContext(Dispatchers.IO) {
                                                                HiTranslator.getInstance(mContext)
                                                                    .postTranslate(
                                                                        Locale.getDefault().language,
                                                                        targetLanguage,
                                                                        list
                                                                    )
                                                            }
                                                        setCurrentState(STATE_TRANSLATE_FINISH)
                                                        transResult.workingIndex = -1
                                                        transResult.transMap!![index] = transR
                                                        mResultListener?.onRecognized(
                                                            index,
                                                            transResult
                                                        )
                                                    }
                                            } else {
                                                mRecognizeJob?.cancel()
                                                mRecognizeJob =
                                                    CoroutineScope(Dispatchers.Main).launch {
                                                        setCurrentState(STATE_RECOGNIZING)
                                                        val ocrResult =
                                                            withContext(Dispatchers.IO) {
                                                                doTextRecognize(
                                                                    index,
                                                                    captureResult
                                                                )
                                                            }
                                                        setCurrentState(STATE_RECOGNIZED_FINISH)
                                                        if (ocrResult != null) {
                                                            transResult.ocrResultMap!![index] =
                                                                ocrResult
                                                            val targetLanguage =
                                                                mResultListener?.getTransTargetLanguage()
                                                                    ?: Locale.ENGLISH.language
                                                            val list = arrayListOf<String>()
                                                            ocrResult.blocks.forEach {
                                                                list.add(it.text)
                                                            }
                                                            setCurrentState(STATE_TRANSLATING)

                                                            val transR =
                                                                withContext(Dispatchers.IO) {
                                                                    HiTranslator.getInstance(
                                                                        mContext
                                                                    )
                                                                        .postTranslate(
                                                                            Locale.getDefault().language,
                                                                            targetLanguage, list
                                                                        )
                                                                }
                                                            setCurrentState(STATE_TRANSLATE_FINISH)
                                                            transResult.workingIndex = -1
                                                            transResult.transMap!![index] = transR
                                                            mResultListener?.onRecognized(
                                                                index,
                                                                transResult
                                                            )
                                                        } else {
                                                            setCurrentState(STATE_TRANSLATE_FINISH)
                                                            transResult.workingIndex = -1
                                                            mResultListener?.onRecognizedFail(
                                                                captureResult.rects!![index]
                                                            )
                                                        }
                                                    }
                                            }
                                        } else {
                                            mRecognizeJob?.cancel()
                                            mRecognizeJob =
                                                CoroutineScope(Dispatchers.Main).launch {
                                                    setCurrentState(STATE_RECOGNIZING)
                                                    val ocrResult = withContext(Dispatchers.IO) {
                                                        doTextRecognize(
                                                            index,
                                                            captureResult
                                                        )
                                                    }
                                                    setCurrentState(STATE_RECOGNIZED_FINISH)
                                                    if (ocrResult != null) {
                                                        transResult.ocrResultMap!![index] =
                                                            ocrResult
                                                        val targetLanguage =
                                                            mResultListener?.getTransTargetLanguage()
                                                                ?: Locale.ENGLISH.language
                                                        val list = arrayListOf<String>()
                                                        ocrResult.blocks.forEach {
                                                            list.add(it.text)
                                                        }
                                                        setCurrentState(STATE_TRANSLATING)

                                                        val transR =
                                                            withContext(Dispatchers.IO) {
                                                                HiTranslator.getInstance(mContext)
                                                                    .postTranslate(
                                                                        Locale.getDefault().language,
                                                                        targetLanguage, list
                                                                    )
                                                            }
                                                        setCurrentState(STATE_TRANSLATE_FINISH)
                                                        transResult.workingIndex = -1
                                                        transResult.transMap!![index] = transR
                                                        mResultListener?.onRecognized(
                                                            index,
                                                            transResult
                                                        )
                                                    } else {
                                                        setCurrentState(STATE_TRANSLATE_FINISH)
                                                        transResult.workingIndex = -1
                                                        mResultListener?.onRecognizedFail(
                                                            captureResult.rects!![index]
                                                        )
                                                    }
                                                }
                                        }
                                    }
                                }

                            }
                        }
                    }
                }
            }
            MotionEvent.ACTION_UP -> {
                if (mRecordService != null) {
                    when (mCurrentState) {
                        STATE_TRANSLATE_FINISH -> {
                            setCurrentState(STATE_SERVICE_PREPARED)
                            mLastResult = null
                        }
                        STATE_TRANSLATING, STATE_RECOGNIZING, STATE_CAPTURING_PIC -> {
                            setCurrentState(STATE_ACTION_CLEAR)
                        }
                    }

                } else {
                    setCurrentState(STATE_INITIALIZE)
                }

                velocityTracker?.recycle()
                velocityTracker = null
            }
        }
    }

    private val mRemoteCallback = object : IRecordListener.Stub() {

        override fun onRecordServiceReady() {
            Log.d("cjslog", "onRecordServiceReady")
            mResultListener?.onRecognizeServiceReady()
        }

    }
}