package com.talpa.translate.render

import android.content.Context
import android.graphics.*
import android.text.Layout
import android.text.StaticLayout
import android.text.TextPaint
import android.util.Log
import android.util.TypedValue
import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.appcompat.widget.AppCompatTextView
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.*
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.ocr.PhotoAnalyzer
import com.talpa.translate.ocr.Recognizer
import com.talpa.translate.ocr.datasource.DataSource
import com.talpa.translate.ocr.datasource.FirebaseDataSource
import com.talpa.translate.ocr.datasource.GoogleDataSource
import com.talpa.translate.ocr.result.OcrResult
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.withContext

/**
 * Create by chenjunsheng on 2020/7/14
 */
class TranslatorRender(val context: Context, private val photoAnalyzer: PhotoAnalyzer) : Render {

    private var mTextPaint: TextPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)

    init {
        mTextPaint.color = Color.BLACK
    }

    override fun initialize(originalBitmap: Bitmap) {
        release() //first release avoid memory leak
        photoAnalyzer.initialize(originalBitmap)
    }

    override fun release() {
        photoAnalyzer.cleanUp()
    }

    override suspend fun renderOverlay(translator: Recognizer, transResult: List<String>): Bitmap? {
        return withContext(Dispatchers.Main) {
            val ocrResult = translator.getHistory() ?: return@withContext null
            val task = async(Dispatchers.IO) {

                val rects = arrayListOf<Rect>()
                ocrResult.blocks.forEach {
                    rects.add(it.rect)
                }
                photoAnalyzer.postAnalyzeArea(rects.toTypedArray())
            }

            val originBitmap = handleVisionTextToBitmap(
                task.await() ?: return@withContext null,
                ocrResult,
                translator.getFrameMetadata() ?: return@withContext null,
                transResult
            )
            val overlayBitmap = BitmapUtils.reRotateBitmap(
                originBitmap,
                translator.getFrameMetadata()?.rotation ?: 0,
                Facing.BACK
            )
            originBitmap.recycle()
            overlayBitmap
        }
    }

    override fun getBase64FromRender(): String? {
        return photoAnalyzer.getBase64String()
    }

    override fun createDataSource(
        bitmap: Bitmap,
        type: Int,
        metadata: FrameMetadata,
        targetLanguage: String
    ): DataSource {
        return when (type) {
            TYPE_FIREBASE -> {
                FirebaseDataSource(bitmap, metadata, targetLanguage)
            }
            TYPE_GOOGLE_VISION -> {

                val test = BitmapUtils.scaleBitmapDown(
                    bitmap,
                    MAX_DIMENSION
                )
                Log.d(
                    "cjslog",
                    "test bitmap:${test.width} ${test.height} ${bitmap.width} ${bitmap.height}"
                )
                GoogleDataSource(
                    bitmap
                    /*BitmapUtils.scaleBitmapDown(
                        bitmap,
                        MAX_DIMENSION
                    )*/, photoAnalyzer.getBase64String(), metadata
                    , targetLanguage
                )
            }
            else -> {
                throw IllegalArgumentException("unknown type for ocr translator")
            }
        }
    }

    /**
     * Vision Text To Bitmap
     */
    private fun handleVisionTextToBitmap(
        background: Bitmap,
        ocrResult: OcrResult,
        metadata: FrameMetadata,
        transResult: List<String?>
    ): Bitmap {

        /*val newBitmap = Bitmap.createBitmap(
            metadata.width, metadata.height,
            Bitmap.Config.ARGB_8888
        )*/
        //newBitmap.setHasAlpha(true)

        val canvas = Canvas(background)
        //canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)

        //canvas.drawBitmap(bitmap, 0f, 0f, null)

        //canvas.drawRect()
        //val backgroundPaint = Paint()
        //backgroundPaint.color = Color.WHITE

        val textPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)
            .also { it.density = context.resources.displayMetrics.density }
        textPaint.color = Color.BLACK
        //textPaint.textAlign = Paint.Align.LEFT

        ocrResult.blocks.forEachIndexed { index, block ->
            val rect = block.rect
            /*val bit = createBlockView(
                context,
                rect,
                transResult.getOrNull(index) ?: return@forEachIndexed
            ) ?: return@forEachIndexed*/
            val textSize = getAutofitTextSize(
                transResult[index] ?: "",
                mTextPaint,
                rect,
                MAX_TEXT_SIZE,
                context
            )
            mTextPaint.textSize = TypedValue.applyDimension(
                TypedValue.COMPLEX_UNIT_SP, textSize.toFloat(),
                context.resources.displayMetrics
            )
            val staticLayout = StaticLayout(
                transResult[index] ?: "",
                mTextPaint, rect.width(),
                Layout.Alignment.ALIGN_NORMAL,
                1.0f,
                0.0f,
                true
            )

            canvas.save()
            canvas.translate(rect.left.toFloat(), rect.top.toFloat())

            staticLayout.draw(canvas)
            //canvas.drawBitmap(bit, rect.left.toFloat(), rect.top.toFloat(), null)
            canvas.restore()

        }

        //val blockFrame = block.boundingBox
        return background
    }

    private fun createBlockView(
        context: Context,
        rect: Rect,
        text: String,
        backgroundColor: Int = Color.WHITE
    ): Bitmap? {

        //val rect = block.boundingBox ?: return null
        //val blockText = block.text

        val textView =
            LayoutInflater.from(context).inflate(R.layout.layout_text, null) as AppCompatTextView
        textView.apply {
            layout(rect.left, rect.top, rect.right, rect.bottom)
        }
        //val textView = view.findViewById<AppCompatTextView>(R.id.tv_content)
        textView.width = rect.width()
        textView.height = rect.height()
        textView.layoutParams = ViewGroup.LayoutParams(rect.width(), rect.height())

        textView.measure(0, 0)

        textView.setBackgroundColor(backgroundColor)

        textView.text = text

        val bitmap = Bitmap.createBitmap(rect.width(), rect.height(), Bitmap.Config.ARGB_8888)

        val canvas = Canvas(bitmap)
        textView.draw(canvas)

        return bitmap
    }

}