package com.talpa.translate.network

import android.content.Context
import android.util.Log
import com.jakewharton.retrofit2.adapter.kotlin.coroutines.CoroutineCallAdapterFactory
import com.talpa.TranslationController
import com.talpa.tengine.Trans
import com.talpa.tengine.lang.LANG
import com.talpa.tengine.lang.languageTagToLangMap
import com.talpa.translate.base.utils.crypt
import com.talpa.translate.network.service.ApiService
import com.talpa.translate.ocr.exception.NoContentException
import okhttp3.OkHttpClient
import retrofit2.Retrofit
import retrofit2.converter.gson.GsonConverterFactory
import java.util.concurrent.TimeUnit
import com.talpa.translator.link.translate
import kotlinx.coroutines.async
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.flow.*

/**
 * Hi Translator Sync
 *
 * @author CY 20-3-5
 */
class HiTranslator private constructor(val context: Context) {
    companion object {

        //private const val DEBUG_BASE_URL = "http://tapi.translasion.com/"
        private const val RELEASE_BASE_URL = "https://api.translasion.com/"
        private const val CONNECT_TIME_OUT = 6000
        private const val READ_TIME_OUT = 6000
        private const val REQUEST_METHOD = "POST"
        private const val CONTENT_TYPE = "application/json"
        private const val APP_KEY = "app_key"
        private const val PREFER_TRANSLATOR = "prefer_translator_data"
        const val SERVER_SECRET = "1e98215f-929b-7cd9-9c20-4c523d4b493b"
        const val SERVER_KEY = "e292400433a4"

        @Volatile
        private var mInstance: HiTranslator? = null

        fun getInstance(context: Context): HiTranslator {
            if (mInstance == null) {
                synchronized(HiTranslator::class.java) {
                    if (mInstance == null) {
                        mInstance = HiTranslator(context.applicationContext)
                    }
                }
            }
            return mInstance!!
        }
    }

    private val apiService: ApiService

    init {

        val okHttpClient = OkHttpClient.Builder()
            .connectTimeout(CONNECT_TIME_OUT.toLong(), TimeUnit.MILLISECONDS)
            .readTimeout(READ_TIME_OUT.toLong(), TimeUnit.MILLISECONDS)
            .addInterceptor {
                val newRequest = it.request().newBuilder()
                    .addHeader("content-type",
                        CONTENT_TYPE
                    )
                    .build()
                Log.d("cjslog", "new Request:$newRequest" +
                        " body:${newRequest.body?.toString()}")
                val response = it.proceed(newRequest)
                Log.d("cjslog", "response:${response}")
                response
            }
            .build()

        apiService = Retrofit.Builder()
            .baseUrl(RELEASE_BASE_URL)
            .client(okHttpClient)
            .addConverterFactory(GsonConverterFactory.create())
            .addCallAdapterFactory(CoroutineCallAdapterFactory.invoke())
            .build().create(ApiService::class.java)
    }

    suspend fun postTranslate(from: String, to: String, texts: List<String>): TransResponse {
//        str = sprintf("%s&%s&%s&%s&%s", appId, to, timestamp, secret, nonce)
//        sig = md5(str)
        val serverAppKey =
            TranslationController.serverAppKey ?: throw NoContentException("app key not set")
        val serverSecret =
            TranslationController.serverAppSecret ?: throw NoContentException("app secret not set")
        val timeStamp = System.currentTimeMillis()
        val random = ((Math.random() * 9 + 1) * 1000).toInt()
        val randomKey = "${serverAppKey}&$to&$timeStamp&${serverSecret}&$random"

        val requestbody =
            TransRequest(
                app_key = serverAppKey,
                from = from,
                to = to,
                texts = texts,
                sig = crypt(randomKey),
                nonce = random.toString(),
                timestamp = timeStamp
            )
        val langMap = languageTagToLangMap()

        Log.d("cjslog", "translating to source:${langMap[from]} target:${langMap[to]}")

        val responses = texts.map {
            Trans(langMap[from] ?: LANG.AUTO, langMap[to] ?: LANG.EN, it)
        }.asFlow().map {
            context.translate(it).blockingSingle()
        }.map {
            val code = it.result?.code ?: 0
            val message = it.result?.errorMessage
            val result = Result(arrayListOf(it.result?.translation ?: ""))
            Log.d("cjslog", "response message:${it.result?.errorMessage}")
            TransResponse(code, message, result)
        }.toList()

        if (true) {
            val translations = arrayListOf<String>()
            responses.forEach {
                translations.add(it.result?.texts?.get(0) ?: "0")
            }
            val result = Result(translations)
            return TransResponse(responses.first().code, responses.first().message, result)
        }

        Log.d("cjslog", "nothing do")

        return apiService.translate(
            requestbody
        )
    }
}