package com.talpa.translate.render

import android.content.Context
import android.graphics.Bitmap
import android.graphics.Rect
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.BitmapUtils
import com.talpa.translate.base.utils.TYPE_FIREBASE
import com.talpa.translate.base.utils.TYPE_GOOGLE_VISION
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.ocr.IAnalyzer
import com.talpa.translate.ocr.Recognizer
import com.talpa.translate.ocr.datasource.DataSource
import com.talpa.translate.ocr.datasource.FirebaseDataSource
import com.talpa.translate.ocr.datasource.GoogleDataSource
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.withContext

/**
 * Create by chenjunsheng on 2020/7/14
 */
class TranslatorRender(val context: Context, private val photoAnalyzer: IAnalyzer) : Render {


    override fun initialize(originalBitmap: Bitmap) {
        release() //first release avoid memory leak
        photoAnalyzer.initialize(originalBitmap)
    }

    override fun release() {
        photoAnalyzer.cleanUp()
    }

    override suspend fun renderOverlay(translator: Recognizer, transResult: List<String>): Bitmap? {
        return withContext(Dispatchers.Main) {
            val ocrResult = translator.getHistory() ?: return@withContext null
            val task = async(Dispatchers.IO) {

                val rects = arrayListOf<Rect>()
                ocrResult.blocks.forEach {
                    rects.add(it.rect)
                }
                photoAnalyzer.postAnalyzeArea(rects.toTypedArray())
            }

            /*val originBitmap = handleVisionTextToBitmap(
                task.await() ?: return@withContext null,
                ocrResult,
                translator.getFrameMetadata() ?: return@withContext null,
                transResult
            )*/

            val originBitmap = photoAnalyzer.getTextDrawer().draw(
                task.await() ?: return@withContext null,
                translator.getFrameMetadata() ?: return@withContext null,
                ocrResult,
                transResult
            )

            val overlayBitmap = BitmapUtils.reRotateBitmap(
                originBitmap,
                translator.getFrameMetadata()?.rotation ?: 0,
                Facing.BACK
            )
            originBitmap.recycle()
            overlayBitmap
        }
    }

    override fun getBase64FromRender(): String? {
        return photoAnalyzer.getBase64String()
    }

    override fun createDataSource(
        bitmap: Bitmap,
        type: Int,
        metadata: FrameMetadata,
        targetLanguage: String
    ): DataSource {
        return when (type) {
            TYPE_FIREBASE -> {
                FirebaseDataSource(bitmap, metadata, targetLanguage)
            }
            TYPE_GOOGLE_VISION -> {

                /*val test = BitmapUtils.scaleBitmapDown(
                    bitmap,
                    MAX_DIMENSION
                )
                Log.d(
                    "cjslog",
                    "test bitmap:${test.width} ${test.height} ${bitmap.width} ${bitmap.height}"
                )*/
                GoogleDataSource(
                    bitmap
                    /*BitmapUtils.scaleBitmapDown(
                        bitmap,
                        MAX_DIMENSION
                    )*/, photoAnalyzer.getBase64String(), metadata
                    , targetLanguage
                )
            }
            else -> {
                throw IllegalArgumentException("unknown type for ocr translator")
            }
        }
    }

}