package com.talpa.camera.translate.photo

import android.content.Context
import android.graphics.ImageFormat.YUV_420_888
import android.util.Log
import android.view.ViewGroup
import android.widget.FrameLayout
import androidx.lifecycle.LifecycleOwner
import com.talpa.base.common.FrameMetadata
import com.talpa.camera.cameraview.CameraListener
import com.talpa.camera.cameraview.CameraView
import com.talpa.camera.cameraview.PictureResult
import com.talpa.camera.cameraview.controls.*
import com.talpa.camera.cameraview.gesture.Gesture
import com.talpa.camera.cameraview.gesture.GestureAction
import com.talpa.camera.cameraview.markers.DefaultAutoFocusMarker
import com.talpa.camera.translate.ocr.NoContentException

/**
 * Create by chenjunsheng on 2020/5/9
 */
class CameraCompatSource : ICameraSource {

    private lateinit var mCamera: CameraView
    private var mStatus: ICameraSource.Status? = null

    override fun initialize(context: Context, camera: CameraView, lifecycleOwner: LifecycleOwner) {
        mCamera = camera
            .apply {
                engine = Engine.CAMERA2
                setExperimental(true)
                preview = Preview.GL_SURFACE
                playSounds = false
                grid = Grid.OFF
                flash = Flash.OFF
                audio = Audio.OFF
                facing = Facing.BACK
                mapGesture(Gesture.TAP, GestureAction.AUTO_FOCUS)
                mapGesture(Gesture.LONG_TAP, GestureAction.NONE)
                mapGesture(Gesture.PINCH, GestureAction.ZOOM)
                mode = Mode.PICTURE
                setAutoFocusMarker(DefaultAutoFocusMarker())
                useDeviceOrientation = false
                frameProcessingFormat = YUV_420_888
            }
        //parent.addView(mCamera, FrameLayout.LayoutParams(ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT))
        mCamera.setLifecycleOwner(lifecycleOwner)
        mCamera.addCameraListener(L())
    }

    override fun setStatusCallback(status: ICameraSource.Status) {
        mStatus = status
    }

    override fun open() {
        if (!mCamera.isOpened) {
            mCamera.open()
        }
    }

    override fun pause() {
        if (mCamera.isOpened) {
            mCamera.close()
        }
    }

    override fun release() {
        mCamera.destroy()
    }

    override fun takePicture() {
        mCamera.takePictureSnapshot()
    }

    override fun torch(switch: Boolean) {
        mCamera.set(if (switch) {Flash.TORCH} else {Flash.OFF})
        mStatus?.onStatusChange(if (switch) {ICameraSource.TORCH_ON} else {ICameraSource.TORCH_OFF})
    }

    private var mOrientation = 0

    private inner class L : CameraListener() {
        override fun onPictureTaken(result: PictureResult) {
            if (result.data != null) {
                Log.d("cjslog", "result:${result.rotation}")
                val metadata = FrameMetadata.Builder()
                    .setWidth(mCamera.width)
                    .setHeight(mCamera.height)
                    .setRotation(mOrientation)
                    .setCameraFacing(mCamera.facing)
                    .setPictureFormat(result.format)
                    .build()
                mStatus?.onPicturetaked(result.data, metadata)
            } else {
                mStatus?.onPicturetakeFail(NoContentException())
            }
        }

        override fun onOrientationChanged(orientation: Int) {
            super.onOrientationChanged(orientation)
            //Log.d("cjslog", "on orientation change:${orientation}")
            mOrientation = orientation
        }
    }
}