package com.talpa.camera.translate.photo


import android.annotation.SuppressLint
import android.content.*
import android.graphics.Bitmap
import android.graphics.RectF
import android.net.Uri
import android.os.Bundle
import android.util.Log
import android.view.KeyEvent
import android.view.LayoutInflater
import android.view.View
import android.view.ViewGroup
import android.widget.CheckBox
import android.widget.Checkable
import android.widget.ImageView
import android.widget.TextView
import androidx.annotation.Keep
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.toBitmap
import androidx.core.os.LocaleListCompat
import androidx.fragment.app.Fragment
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.lifecycleScope
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.airbnb.lottie.LottieAnimationView
import com.bumptech.glide.Glide
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.snackbar.Snackbar
import com.google.firebase.ml.common.FirebaseMLException
import com.photo.translation.R
import com.talpa.base.common.FrameMetadata
import com.talpa.base.utils.*
import com.talpa.base.utils.BitmapUtils.getBitmap
import com.talpa.base.view.WheelView
import com.talpa.camera.cameraview.CameraView
import com.talpa.camera.edmodo.cropper.CropImageView
import com.talpa.camera.edmodo.cropper.cropwindow.edge.Edge
import com.talpa.camera.translate.ocr.ImageTranslate
import com.talpa.camera.translate.utils.*
import com.talpa.camera.translate.view.CaptureButton
import com.talpa.tengine.lang.languageTagToLangMap
import com.talpa.translate.photo.KEY_EVENT_ACTION
import com.talpa.translate.photo.KEY_EVENT_EXTRA
import com.talpa.translate.photo.localeToGoogleLanguage
import com.talpa.translation.activity.CameraActivity
import com.tencent.mmkv.MMKV
import com.trello.rxlifecycle3.android.lifecycle.kotlin.bindUntilEvent
import io.reactivex.Flowable
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.disposables.Disposable
import io.reactivex.functions.Consumer
import io.reactivex.schedulers.Schedulers
import kotlinx.android.synthetic.main.fragment_camera.*
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.io.IOException
import java.lang.ref.WeakReference
import java.util.*


private const val TAG = "CameraFragment"

/**
 * Camera Preview
 *
 * @author CY 2020-02-03
 */
@Keep
class CameraFragment : Fragment(), View.OnClickListener, ICameraSource.Status {

    private lateinit var mControlView: CaptureButton
    private var mCropImageView: CropImageView? = null
    private lateinit var mSelectImage: ImageView

    //private var lastRectF: RectF? = null

    /**
     * 当前状态
     */
    private var currentStatus = STATE_CAMERA

    private var translateDisposable: Disposable? = null

    private val sourceLiveData: MutableLiveData<String> = MutableLiveData<String>("en")
    private val targetLiveData: MutableLiveData<String> = MutableLiveData<String>("en")

    private var imgTranslate: ImageTranslate? = null

    /**
     * 刷选出的所有拉丁语
     * [{"d":"英语","t":"en"},{"d":"德语","t":"de"},{"d":"立陶宛语","t":"lt"},{"d":"克罗地亚语","t":"hr"},{"d":"拉脱维亚语","t":"lv"},{"d":"匈牙利语","t":"hu"},{"d":"约鲁巴语","t":"yo"},{"d":"马拉加斯语","t":"mg"},{"d":"伊博语","t":"ig"},{"d":"夏威夷语","t":"haw"},{"d":"南非荷兰语","t":"af"},{"d":"乌兹别克语","t":"uz-Latn"},{"d":"印度尼西亚语","t":"id-ID"},{"d":"马来语","t":"ms"},{"d":"马耳他语","t":"mt"},{"d":"世界语","t":"eo"},{"d":"冰岛语","t":"is"},{"d":"意大利语","t":"it"},{"d":"西班牙语","t":"es"},{"d":"爱沙尼亚语","t":"et"},{"d":"巴斯克语","t":"eu"},{"d":"越南语","t":"vi"},{"d":"阿塞拜疆语","t":"az-Latn"},{"d":"祖鲁语","t":"zu"},{"d":"罗马尼亚语","t":"ro"},{"d":"荷兰语","t":"nl"},{"d":"法语","t":"fr"},{"d":"波斯尼亚语","t":"bs-Latn"},{"d":"西弗里西亚语","t":"fy"},{"d":"斯洛伐克语","t":"sk"},{"d":"斯洛文尼亚语","t":"sl"},{"d":"绍纳语","t":"sn"},{"d":"索马里语","t":"so"},{"d":"苏格兰盖尔语","t":"gd"},{"d":"加泰罗尼亚语","t":"ca"},{"d":"阿尔巴尼亚语","t":"sq"},{"d":"斯瓦希里语","t":"sw"},{"d":"捷克语","t":"cs"},{"d":"科萨语","t":"xh"},{"d":"威尔士语","t":"cy"},{"d":"豪萨语","t":"ha"},{"d":"波兰语","t":"pl"},{"d":"丹麦语","t":"da"},{"d":"土耳其语","t":"tr"}]
     * Source Support Languages Tag
     */
    //private val sourceSupportLanguages by lazy { "en,de,lt,hr,lv,hu,yo,mg,ig,haw,af,uz-Latn,id-ID,ms,mt,eo,is,it,es,et,eu,vi,az-Latn,zu,ro,nl,fr,bs-Latn,fy,sk,sl,sn,so,gd,ca,sq,sw,cs,xh,cy,ha,pl,da,tr" }

    /**
     * 进一步筛选
     */
    private val sourceSupportLanguages by lazy { "ca,da,nl,en,fi,fr,de,hu,it,la,no,pl,pt,ro,es,sv,tr,sw" }

    /**
     * Target Support Languages Tag
     */
    private val targetSupportLanguages by lazy { "en,hi,de,lo,ps,pt,lt,hr,lv,hu,hy,uk,yo,mg,ur,ig,mk,haw,ml,mn,af,mr,uz-Latn,id-ID,ms,el,mt,eo,is,it,am,my,es,zh-CN,he-IL,et,eu,ar,vi,nb-NO,ja,ne,az-Latn,fa,zu,ro,nl,be,fi,ru,bg,bn,fr,bs-Latn,fy,ka,si,sk,sl,ga,sn,so,gd,ca,sq,sr-Latn,kk,km,kn,sv,ko,sw,gl,ta,gu,ky,cs,xh,pa-Guru,te,tg,th,cy,lb,ha,pl,da,tr" }

    private lateinit var cameraBroadcastReceiver: CameraBroadcastReceiver

    private var singleClickTime: Long = 0

    //private var ivToCamera: View? = null
    private var tvSourceLanguage: TextView? = null
    private var tvTargetLanguage: TextView? = null
    private var flashLight: CheckBox? = null
    private lateinit var mCameraSource: ICameraSource

    companion object {

        /*Camera state*/
        const val STATE_CAMERA = 100

        /*Cropper*/
        const val STATE_CROP_FROM_CAMERA = 200

        /*Camera state*/
        const val STATE_SELECT_IMAGE = 300

        const val STATE_RESULT_FROM_CAMERA = 400
        const val STATE_RESULT_FROM_GALLERY = 401

        const val STATE_CROP_FROM_GALLERY = 500

        const val REQUEST_CODE_SELECT_IMAGE = 1000
        const val REQUEST_CODE_SELECT_SOURCE_LANGUAGE = 1100
        const val REQUEST_CODE_SELECT_TARGET_LANGUAGE = 1200

        private const val PREFER_KEY_TARGET_LANG = "key_target_lang"

    }

    override fun onAttach(context: Context) {
        super.onAttach(context)
        checkSourceLegal(context)
    }

    private fun checkSourceLegal(context: Context) {
        val result = context as? CameraActivity
        checkNotNull(result) {
            "must call TranslationController.startCameraCollection to start CameraFragment"
        }
    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        return inflater.inflate(R.layout.fragment_camera, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        mCameraSource = CameraCompatSource()
        //mainExecutor = ContextCompat.getMainExecutor(requireContext())

        initPreviewView(view)

        initView(view)

        cameraBroadcastReceiver = CameraBroadcastReceiver(this)

        registerBroadcast()

        enterCameraState()

        readLanguage(view)

        handleIntent()

        mCameraSource.setStatusCallback(this)
    }

    override fun cameraInitializeFail(exception: Exception) {
        showErrorAlert(requireContext(), exception)
    }

    override fun onPicturetaked(data: ByteArray, metadata: FrameMetadata) {
        if (!isDetached && activity?.isDestroyed == false) {
            onImageSaved(data, metadata)
        }
    }

    override fun onPicturetakeFail(exception: java.lang.Exception) {
        showErrorAlert(requireActivity(), exception)
    }

    override fun onStatusChange(state: Int) {
        when(state) {
            ICameraSource.TORCH_OFF -> {
                flashLight?.isChecked = false
            }
            ICameraSource.TORCH_ON -> {
                flashLight?.isChecked = true
            }
        }
    }

    /**
     * Handle Intent
     */
    private fun handleIntent() {
        val intent = requireActivity().intent
        when (intent?.action) {
            Intent.ACTION_SEND -> {
                val uri = intent.getParcelableExtra<Uri?>(Intent.EXTRA_STREAM)

                if (uri != null) {
                    setImageFromUri(uri)
                }
            }
        }
    }

    override fun onPause() {
        super.onPause()
        mCameraSource.torch(false)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        unregisterBroadcast()
        if (translateDisposable?.isDisposed == false) {
            translateDisposable?.dispose()
            translateDisposable = null
        }

        mCameraSource.release()
    }


    private fun enterCropState() {
        //currentStatus = STATE_CROP_FROM_CAMERA
        mCameraSource.pause()
        mControlView.setState(mControlView.cropState)

        if (mCropImageView?.visibility != View.VISIBLE) {
            mCropImageView?.visibility = View.VISIBLE
        }

        flashLight?.visibility = View.INVISIBLE
        mSelectImage.visibility = View.INVISIBLE
    }

    private fun enterSelectImageState() {
        //currentStatus = STATE_SELECT_IMAGE
        //cameraHandler.sendEmptyMessage(WHAT_ENABLE_CONTROL_VIEW)
        hideProgressBar()
        tvSourceLanguage?.isClickable = true
        tvTargetLanguage?.isClickable = true
    }

    private fun enterCameraState() {
        if (screenShotMode) {
            activity?.finish()
            return
        }

        mCameraSource.open()
        if (translateDisposable?.isDisposed == false) {
            translateDisposable?.dispose()
            translateDisposable = null
        }
        //imgTranslate?.reset()

        currentStatus = STATE_CAMERA
        mControlView.visibility = View.VISIBLE
        mControlView.setState(mControlView.cameraState)
        mControlView.revertAnim()
        mControlView.isClickable = true

        //mControlView.setBackgroundResource(R.drawable.ic_camera_take)
        if (mCropImageView?.visibility != View.GONE) {
            lifecycleScope.launch(Dispatchers.Main) {
                delay(500)
                mCropImageView?.visibility = View.GONE
            }
        }
        mCropImageView?.isEnabled = true
        mCropImageView?.setState(CropImageView.State.NORMAL)

        /*if (mPreviewView.visibility != View.VISIBLE) {
            mPreviewView.visibility = View.VISIBLE
        }*/

        flashLight?.visibility = View.VISIBLE
        mSelectImage.visibility = View.VISIBLE
        //source_img?.visibility = View.GONE

        hideProgressBar()

        tvSourceLanguage?.isClickable = true
        tvTargetLanguage?.isClickable = true

        tvSourceLanguage?.setTextColor(ContextCompat.getColor(requireContext(), R.color.camera_tool_bg))
        tvTargetLanguage?.setTextColor(ContextCompat.getColor(requireContext(), R.color.camera_tool_bg))
        iv_exchange?.alpha = 1.0f

        //cameraHandler.sendEmptyMessage(WHAT_ENABLE_CONTROL_VIEW)
    }

    private fun enterTranslateState() {
        tvSourceLanguage?.isClickable = false
        tvTargetLanguage?.isClickable = false
        mControlView.setState(mControlView.cameraState)
        mControlView.startAnim()
        mControlView.isClickable = false
        language_selector?.background?.alpha = 0x66
        language_selector?.isClickable = false
        tvSourceLanguage?.setTextColor(ContextCompat.getColor(requireContext(), R.color.camera_tool_bg2))
        tvTargetLanguage?.setTextColor(ContextCompat.getColor(requireContext(), R.color.camera_tool_bg2))
        iv_exchange?.alpha = 0.5f
        mCropImageView?.isEnabled = false
    }

    private fun enterResultState() {
        tvSourceLanguage?.isClickable = true
        tvTargetLanguage?.isClickable = true
        mControlView.revertAnim()
        mControlView.setState(mControlView.resultState)
        mControlView.isClickable = true
        mControlView.visibility = View.VISIBLE
        currentStatus = if (currentStatus == STATE_CROP_FROM_GALLERY) {
            STATE_RESULT_FROM_GALLERY
        } else {
            STATE_RESULT_FROM_CAMERA
        }
        //cameraHandler.sendEmptyMessageDelayed(WHAT_ENABLE_CONTROL_VIEW, 100)
        language_selector?.background?.alpha = 0xff
        language_selector?.isClickable = true
        tvSourceLanguage?.setTextColor(ContextCompat.getColor(requireContext(), R.color.camera_tool_bg))
        tvTargetLanguage?.setTextColor(ContextCompat.getColor(requireContext(), R.color.camera_tool_bg))
        iv_exchange?.alpha = 1.0f
        //lastRectF = null
        mCropImageView?.isEnabled = false
        mCropImageView?.setState(CropImageView.State.RESULT)
        //source_img?.visibility = View.VISIBLE
    }

    private fun initView(view: View) {

        mControlView = view.findViewById(R.id.btn_control)

        mControlView.setOnClickListener(this)

        mCropImageView = view.findViewById(R.id.iv_preview)

        mSelectImage = view.findViewById(R.id.iv_select_image)

        mSelectImage.setOnClickListener(this)
        iv_exchange?.setOnClickListener(this)

        view.findViewById<View>(R.id.language_selector).setOnClickListener(this)

        val finishView = view.findViewById<View>(R.id.iv_finish)

        finishView.setOnClickListener(this)

        flashLight = view.findViewById(R.id.torch_selector)

        flashLight?.setOnClickListener(this)

        tvSourceLanguage = view.findViewById(R.id.tv_source_language)
        tvTargetLanguage = view.findViewById(R.id.tv_target_language)

        tvSourceLanguage?.setOnClickListener(this)
        tvTargetLanguage?.setOnClickListener(this)

        sourceLiveData.observe(viewLifecycleOwner, androidx.lifecycle.Observer { language ->
            tvSourceLanguage?.text = Locale.forLanguageTag(language).displayLanguage
        })
        targetLiveData.observe(viewLifecycleOwner, androidx.lifecycle.Observer { language ->
            tvTargetLanguage?.text = Locale.forLanguageTag(language).displayLanguage
            exchangeState()
        })

        /*view.apply {
            setPadding(paddingLeft, paddingTop, paddingRight, context.navigationBarHeight())
        }*/
    }

    private fun readLanguage(view: View) {
        val context: Context = view.context
        val mmkv = MMKV.defaultMMKV()
        //val prefer = context.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE)
        val index = supportSourceLanguages().indexOf(Locale.getDefault())
        val defaultTargetLang: String
        if (index == -1) {
            defaultTargetLang = "en"
        } else {
            defaultTargetLang = Locale.getDefault().toLanguageTag()
        }

        val sourceLang = mmkv.decodeString(PREFER_KEY_SOURCE_LANG, defaultTargetLang) ?: "en"
        //val sourceLang = prefer.getString(PREFER_KEY_SOURCE_LANG, Locale.ENGLISH.language) ?: "en"
        val targetLang = mmkv.decodeString(PREFER_KEY_TARGET_LANG, Locale.ENGLISH.language) ?: defaultTargetLang
        //val targetLang =
        //    prefer.getString(PREFER_KEY_TARGET_LANG, defaultTargetLang) ?: defaultTargetLang
        sourceLiveData.value = sourceLang
        targetLiveData.value = targetLang
    }

    /***************************LANGUAGE START*********************************/

    /**
     * Picker for Language
     */
    @SuppressLint("InflateParams")
    private fun languagePicker(context: Context, anchorView: View) {
        val bottomSheetDialog = BottomSheetDialog(context)

        val contentView =
            LayoutInflater.from(context).inflate(R.layout.layout_language_picker, null)

        //val bmp: Bitmap = BitmapFactory.decodeResource(resources, R.drawable.picker_bg)

        bottomSheetDialog.setContentView(contentView)
        bottomSheetDialog.getWindow()?.findViewById<View>(R.id.design_bottom_sheet)
            ?.setBackgroundResource(android.R.color.transparent)

        val sourcePicker = contentView.findViewById<WheelView<String>>(R.id.np_source_picker)
        val targetPicker = contentView.findViewById<WheelView<String>>(R.id.np_target_picker)
        val submit = contentView.findViewById<View>(R.id.submit_language)

        /* ***********Source Picker*************/
        val supportSourceLanguages = supportSourceLanguages()

        val supportSourceLanguageSize = supportSourceLanguages.size()
        val sourceValues = Array(supportSourceLanguageSize) {
            supportSourceLanguages[it].displayLanguage
        }

        val sourceLanguageTag = sourceLiveData.value
        val defaultSourceIndex = if (sourceLanguageTag != null) {
            var index = 0
            for (i in 0..supportSourceLanguageSize) {
                val locale = supportSourceLanguages[i] ?: continue
                if (locale.toLanguageTag() == sourceLanguageTag) {
                    index = i
                    break
                }
            }
            index
        } else 0

        sourcePicker.data = sourceValues.toList()
        sourcePicker.selectedItemPosition = defaultSourceIndex

        /* ***********Target Picker*************/

        val supportTargetLanguages = supportTargetLanguages()
        val supportTargetLanguageSize = supportTargetLanguages.size()
        val targetValues = Array(supportTargetLanguageSize) {
            supportTargetLanguages[it].displayLanguage
        }
        //println("supportTargetLanguageSize=$supportTargetLanguageSize")
        val targetLanguageTag = targetLiveData.value
        val defaultTargetIndex = if (targetLanguageTag != null) {
            var index = 0
            for (i in 0..supportTargetLanguageSize) {
                val locale = supportTargetLanguages.get(i) ?: continue
                val tempTag = locale.toLanguageTag()
                if (tempTag == targetLanguageTag) {
                    index = i
                    break
                }
            }

            index

        } else 0

        targetPicker.data = targetValues.toList()
        targetPicker.selectedItemPosition = defaultTargetIndex

        var sourceLocale = supportSourceLanguages[sourcePicker.selectedItemPosition]
        val sourceSelectedListener = WheelView.OnItemSelectedListener<String> { wheelView, data, position ->
            /*val type = if (sourcePicker == wheelView) {
                PREFER_KEY_SOURCE_LANG
            } else {
                PREFER_KEY_TARGET_LANG
            }*/
            /*val locale = if (sourcePicker == wheelView) {
                supportSourceLanguages[position]
            } else {
                supportTargetLanguages[position]
            }*/
            sourceLocale = supportSourceLanguages[position]
            //saveLanguage(context, type, locale.toLanguageTag())
        }

        var targetLocale = supportTargetLanguages[targetPicker.selectedItemPosition]
        val targetSelectedListener = WheelView.OnItemSelectedListener<String> { wheelView, data, position ->
            targetLocale = supportTargetLanguages[position]
        }

        sourcePicker.onItemSelectedListener = sourceSelectedListener
        targetPicker.onItemSelectedListener = targetSelectedListener

        submit.setOnClickListener {
            //readLanguage(submit)
            saveLanguage(PREFER_KEY_SOURCE_LANG, sourceLocale.toLanguageTag())
            saveLanguage(PREFER_KEY_TARGET_LANG, targetLocale.toLanguageTag())
            sourceLiveData.value = sourceLocale.toLanguageTag()
            targetLiveData.value = targetLocale.toLanguageTag()
            if (bottomSheetDialog.isShowing) {
                bottomSheetDialog.dismiss()
            }

            val onNext = TranslateSubscribe(this)
            val onError = TranslateError(this)

            if (currentStatus == STATE_RESULT_FROM_CAMERA || currentStatus == STATE_RESULT_FROM_GALLERY) {
                mCropImageView?.drawable?.toBitmap()?.let {
                    imgTranslate?.changeLanguage(sourceLiveData.value ?: return@let,
                        targetLiveData.value ?: return@let)
                        ?.bindUntilEvent(this@CameraFragment, Lifecycle.Event.ON_DESTROY)
                        ?.map { translationBitmap -> replaceImage(it , translationBitmap) }
                        ?.subscribeOn(Schedulers.single())
                        ?.observeOn(AndroidSchedulers.mainThread())
                        ?.doOnSubscribe {
                            currentStatus = if (currentStatus == STATE_RESULT_FROM_CAMERA) {
                                STATE_CROP_FROM_CAMERA
                            } else {
                                STATE_CROP_FROM_GALLERY
                            }
                            enterTranslateState()
                            progress?.visibility = View.VISIBLE
                        }
                        ?.doFinally {
                            enterResultState()
                            progress?.visibility = View.GONE
                        }
                        ?.subscribe(onNext, onError)
                }
            }

            changeLanguageEvent(sourceLiveData.value ?: return@setOnClickListener,
                targetLiveData.value ?: return@setOnClickListener)
        }

        bottomSheetDialog.setOnShowListener {
            language_selector?.visibility = View.INVISIBLE
        }
        bottomSheetDialog.setOnDismissListener {
            language_selector?.visibility = View.VISIBLE
        }

        targetPicker.setVibratorEnable(true)
        sourcePicker.setVibratorEnable(true)

        bottomSheetDialog.show()
    }

    /**
     * Support Source Language
     */
    private fun supportSourceLanguages(): LocaleListCompat {
        val localeListCompat = LocaleListCompat.forLanguageTags(sourceSupportLanguages)
        return localeListCompat

    }

    /**
     * Support Target Language
     */
    private fun supportTargetLanguages(): LocaleListCompat {

        return LocaleListCompat.forLanguageTags(targetSupportLanguages)
    }

    /**
     * Exchange Language
     */
    private fun exchangeLanguage(context: Context) {
        val sourceLanguageTag = targetLiveData.value ?: return
        val targetLanguageTag = sourceLiveData.value ?: return
        saveLanguage(PREFER_KEY_SOURCE_LANG, sourceLanguageTag)
        saveLanguage(PREFER_KEY_TARGET_LANG, targetLanguageTag)
        sourceLiveData.value = sourceLanguageTag
        targetLiveData.value = targetLanguageTag
    }

    /***************************LANGUAGE END*********************************/

    /***************************CAMERA START*********************************/

    private fun initPreviewView(view: View) {
        val cameraView = view.findViewById<CameraView>(R.id.preview_container)
        mCameraSource.initialize(requireContext(), cameraView, this)
        //mPreviewView = view.findViewById(R.id.preview)
    }

    /**
     * Take Photo
     */
    private fun takePicture(context: Context) {
        mCameraSource.takePicture()
    }

    private var mMetadata: FrameMetadata? = null

    /**
     * Take Photo Success
     */
    private fun onImageSaved(data: ByteArray, metadata: FrameMetadata) {
        currentStatus = STATE_CROP_FROM_CAMERA
        cropImageMatchParent()
        showCropImageView(data = data, metadata = metadata)
        mMetadata = metadata
        /*source_img?.let {
            Glide.with(this).load(data.copyOf()).into(it)
        }*/
    }

    private fun cropImageMatchParent() {
        val params = mCropImageView?.layoutParams ?: return
        params.width = 0 //ViewGroup.LayoutParams.MATCH_PARENT
        params.height = 0//ViewGroup.LayoutParams.WRAP_CONTENT
        mCropImageView?.layoutParams = params
        //mCropImageView?.scaleType = ImageView.ScaleType.FIT_CENTER
        mCropImageView?.scaleType = ImageView.ScaleType.CENTER_CROP
    }

    private fun cropImageWrapContent() {
        val params = mCropImageView?.layoutParams ?: return
        params.width = ViewGroup.LayoutParams.MATCH_PARENT
        params.height = ViewGroup.LayoutParams.MATCH_PARENT
        mCropImageView?.layoutParams = params
        mCropImageView?.scaleType = ImageView.ScaleType.FIT_CENTER
    }

    /**
     * 显示裁剪
     */
    private fun showCropImageView(data: ByteArray? = null, uri: Uri? = null, metadata: FrameMetadata? = null) {
        val cropImageView = mCropImageView ?: return
        if (data != null && metadata != null) {
            Glide.with(this).load(data).into(cropImageView)
        } else {
            Glide.with(this).load(uri).into(cropImageView)
        }

        cropImageView.postDelayed({ enterCropState() }, 100)
    }

    private fun showErrorAlert(context: Context, exc: java.lang.Exception) {
        val onClickListener = DialogInterface.OnClickListener { dialog, which ->
            activity?.finish()
        }

        AlertDialog.Builder(context)
            .setTitle(android.R.string.dialog_alert_title)
            .setMessage(R.string.camera_unavailable)
            .setPositiveButton(android.R.string.ok, onClickListener)
            .setCancelable(false)
            .create()
            .show()
    }

    /***************************CAMERA END*********************************/


    /***************************CROPPER WITH TRANSLATE START*********************************/

    /**
     * Cropped Image
     */
    private fun croppedImage(context: Context) {

        val currentRectF = RectF(
            Edge.LEFT.coordinate,
            Edge.TOP.coordinate,
            Edge.RIGHT.coordinate,
            Edge.BOTTOM.coordinate
        )

        //if (currentRectF == lastRectF) return

        var croppedBitmap: Bitmap? = null
        var completedBitmap: Bitmap? = null
        try {
            croppedBitmap = mCropImageView?.croppedImage
            completedBitmap = mCropImageView?.drawable?.toBitmap()
        } catch (e: Exception) {
            e.printStackTrace()
        } finally {
            if (croppedBitmap == null || completedBitmap == null) {
                recognitionFailure()
                return
            }
        }

        showProgressBar()
        enterTranslateState()

        val sourceLanguageTag = sourceLiveData.value ?: "en"
        val targetLanguageTag = targetLiveData.value ?: "en"
        val map = languageTagToLangMap()
        val sourceLanguage = map[sourceLanguageTag] ?: "en"
        val targetLanguage = map[targetLanguageTag] ?: "en"

        val translateFlow = translateImage(context, croppedBitmap, sourceLanguage, targetLanguage)

        val onNext = TranslateSubscribe(this)
        val onError = TranslateError(this)

        translateDisposable = translateFlow
            .bindUntilEvent(this, Lifecycle.Event.ON_DESTROY)
            .map { translationBitmap -> replaceImage(completedBitmap, translationBitmap, ContextCompat.getColor(requireContext(), R.color.result_mask)) }
            .subscribeOn(Schedulers.single())
            .observeOn(AndroidSchedulers.mainThread())
            .subscribe(onNext, onError)
    }

    private class TranslateSubscribe(cameraFragment: CameraFragment) : Consumer<Bitmap> {
        private val weakRefer = WeakReference(cameraFragment)
        override fun accept(newBitmap: Bitmap?) {
            val cameraFragment = weakRefer.get() ?: return
            cameraFragment.apply {
                if (isDetached) return
                recognitionSuccess(newBitmap)
                enterResultState()
                translationSuccessEvent(currentStatus)
            }

        }
    }

    private class TranslateError(cameraFragment: CameraFragment) : Consumer<Throwable> {
        private val weakRefer = WeakReference(cameraFragment)
        override fun accept(it: Throwable?) {

            val cameraFragment = weakRefer.get() ?: return
            cameraFragment.apply {
                if (isDetached) return
                val context = requireContext()
                it?.printStackTrace()
                recognitionFailure(it)
                enterResultState()
                //enterCameraState()
            }
        }
    }

    /**
     * 翻译图片
     */
    private fun translateImage(
        context: Context,
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String
    ): Flowable<Bitmap> {
        Log.d("cjslog", "source:${sourceLanguage} ${targetLanguage}")
        imgTranslate = ImageTranslate(requireContext(), bitmap, sourceLanguage, targetLanguage, mMetadata)
        return imgTranslate!!.translate()
    }

    private fun recognitionFailure(throwable: Throwable? = null) {
        hideProgressBar()
        showTranslateError(throwable)
    }

    /**
     * 识别翻译成功
     */
    private fun recognitionSuccess(newBitmap: Bitmap?) {
        if (isDetached) return
        mCropImageView?.setImageBitmap(newBitmap)

        hideProgressBar()

        /*lastRectF = RectF(
            Edge.LEFT.coordinate,
            Edge.TOP.coordinate,
            Edge.RIGHT.coordinate,
            Edge.BOTTOM.coordinate
        )*/

    }


    /**
     * Show Progress
     */
    private fun showProgressBar() {
        val progressBar = view?.findViewById<LottieAnimationView>(R.id.progress)
        if (progressBar?.visibility != View.VISIBLE) {
            progressBar?.visibility = View.VISIBLE
        }

    }

    /**
     * Hide Progress
     */
    private fun hideProgressBar() {
        val progressBar = view?.findViewById<LottieAnimationView>(R.id.progress)
        if (progressBar?.visibility != View.GONE) {
            progressBar?.visibility = View.GONE
        }
    }

    private fun showTranslateError(throwable: Throwable? = null) {
        val v = view ?: return
        val resId = when (throwable) {
            is FirebaseMLException -> {
                R.string.waiting_recognition_model_download
            }
            is IOException -> {
                R.string.network_unavailable
            }
            else -> {
                R.string.no_content_identified
            }
        }

        fail_toast?.visibility = View.VISIBLE
        fail_toast_text?.setText(resId)
        fail_toast?.postDelayed({fail_toast?.visibility = View.GONE}, 2000)

        translationFailEvent(currentStatus, throwable)
    }

    /***************************CROPPER WITH TRANSLATE END*********************************/

    /***************************SELECT START*********************************/
    /**
     * 设置图片
     */
    fun setImageFromUri(uri: Uri) {
        currentStatus = STATE_CROP_FROM_GALLERY
        cropImageWrapContent()
        showCropImageView(uri = uri)
        enterSelectImageState()
    }

    private var screenShotMode = false

    fun setScreenShotMode(mode: Boolean) {
        screenShotMode = mode
    }

    /***************************SELECT END*********************************/

    override fun onClick(v: View) {

        val currentTime = System.currentTimeMillis()
        if (currentTime - singleClickTime < 300) {
            singleClickTime = currentTime
            return
        }

        when (v.id) {
            R.id.btn_control -> {

                if (!v.context.isNetworkConnected()) {
                    Snackbar.make(v, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return
                }

                //cameraHandler.sendEmptyMessage(WHAT_UNABLE_CONTROL_VIEW)
                mControlView.startAnim()
                when (currentStatus) {

                    STATE_CAMERA -> {
                        takePicture(v.context)
                        logEvent(PT_camera_click)
                    }
                    STATE_CROP_FROM_CAMERA, STATE_CROP_FROM_GALLERY -> {
                        croppedImage(v.context)
                        startTranslateEvent(currentStatus)
                    }
                    STATE_RESULT_FROM_CAMERA, STATE_RESULT_FROM_GALLERY -> {
                        enterCameraState()
                    }
                }
            }
            R.id.iv_select_image -> {

                val intent = Intent(Intent.ACTION_PICK)

                intent.type = "image/*"

                startActivityForResult(
                    intent,
                    REQUEST_CODE_SELECT_IMAGE
                )
                logEvent(PT_gallery_click)
            }
            R.id.torch_selector -> {
                val isChecked = (v as? Checkable)?.isChecked
                mCameraSource.torch(isChecked == true)
                //mCamera?.cameraControl?.enableTorch(isChecked == true)
                logEvent(PT_flash_click)
            }
            R.id.iv_finish -> {
                //requireActivity().finish()
                if (currentStatus == STATE_CROP_FROM_CAMERA
                    || currentStatus == STATE_CROP_FROM_GALLERY
                    || currentStatus == STATE_RESULT_FROM_GALLERY
                    || currentStatus == STATE_RESULT_FROM_CAMERA) {
                    enterCameraState()
                } else {
                    activity?.finish()
                }
            }
            R.id.tv_source_language, R.id.tv_target_language, R.id.language_selector -> {
                /*startActivityForSelectLanguage(
                    v.context,
                    REQUEST_CODE_SELECT_SOURCE_LANGUAGE
                )*/
                languagePicker(v.context, v)
                pickerClickedEvent(state = currentStatus)
            }
            R.id.iv_exchange -> {
                exchangeLanguage(v.context)
            }

        }
    }

    private fun exchangeState() {
        val targetLanguageTag = targetLiveData.value ?: return
        iv_exchange?.isEnabled = sourceSupportLanguages.contains(targetLanguageTag)
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)

        when (requestCode) {
            REQUEST_CODE_SELECT_IMAGE -> {
                val uri = data?.data
                //println("uri===$uri")

                if (uri != null) {
                    currentStatus = STATE_CROP_FROM_GALLERY
                    setImageFromUri(uri)
                }

            }
            REQUEST_CODE_SELECT_SOURCE_LANGUAGE -> {
                val languageTag = data?.getStringExtra("languageTag") ?: return
                saveLanguage(PREFER_KEY_SOURCE_LANG, languageTag)
            }
            REQUEST_CODE_SELECT_TARGET_LANGUAGE -> {
                val languageTag = data?.getStringExtra("languageTag") ?: return
                saveLanguage(PREFER_KEY_TARGET_LANG, languageTag)
            }
        }
    }

    private fun registerBroadcast() {
        LocalBroadcastManager.getInstance(requireContext())
            .registerReceiver(cameraBroadcastReceiver, IntentFilter(KEY_EVENT_ACTION))
    }

    private fun unregisterBroadcast() {
        LocalBroadcastManager.getInstance(requireContext())
            .unregisterReceiver(cameraBroadcastReceiver)
    }

    private class CameraBroadcastReceiver(cameraFragment: CameraFragment) : BroadcastReceiver() {
        private val weakReference = WeakReference(cameraFragment)
        override fun onReceive(context: Context, intent: Intent) {
            val fragment = weakReference.get() ?: return
            when (intent.action) {
                KEY_EVENT_ACTION -> {

                    when (intent.getIntExtra(KEY_EVENT_EXTRA, KeyEvent.KEYCODE_BACK)) {
                        KeyEvent.KEYCODE_BACK -> {
                            if (fragment.currentStatus == STATE_CROP_FROM_CAMERA
                                || fragment.currentStatus == STATE_CROP_FROM_GALLERY
                                || fragment.currentStatus == STATE_RESULT_FROM_GALLERY
                                || fragment.currentStatus == STATE_RESULT_FROM_CAMERA) {
                                fragment.enterCameraState()
                            } else {
                                fragment.activity?.finish()
                            }
                        }
                    }
                }
            }
        }
    }
}