package com.talpa.translation.activity

import android.Manifest
import android.app.Activity
import android.content.*
import android.content.pm.PackageManager
import android.content.pm.PackageManager.PERMISSION_GRANTED
import android.net.Uri
import android.os.Build
import android.os.Bundle
import android.provider.MediaStore
import android.text.TextUtils
import android.util.Log
import android.view.KeyEvent
import android.view.View
import androidx.annotation.RestrictTo
import androidx.appcompat.app.AppCompatActivity
import androidx.core.app.ActivityCompat
import androidx.core.content.ContextCompat
import androidx.core.content.FileProvider
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.photo.translation.R
import com.talpa.TranslationController
import com.talpa.base.utils.getFreeSpace
import com.talpa.camera.translate.photo.CameraFragment
import com.talpa.translate.photo.KEY_EVENT_ACTION
import com.talpa.translate.photo.KEY_EVENT_EXTRA
import com.talpa.translation.dialog.DiskAlertDialog
import com.talpa.translation.engine.TranslationStateEngine
import com.talpa.translation.event.Event
import com.talpa.translation.handler.TranslationHandler
import com.talpa.translation.session.TranslationSession
import com.talpa.translation.session.TranslationSessionFactory
import java.io.File


private val PERMISSIONS_REQUIRED = arrayOf(
    Manifest.permission.CAMERA,
    Manifest.permission.WRITE_EXTERNAL_STORAGE,
    Manifest.permission.READ_EXTERNAL_STORAGE
)

private const val REQUEST_CODE_PERMISSIONS = 100

/**
 * Camera Preview
 *
 * @author CY 2020-02-03
 */
class CameraActivity : AppCompatActivity(),
    TranslationHandler, DiskAlertDialog.Callback {

    @RestrictTo(RestrictTo.Scope.LIBRARY)
    companion object {

        fun start(context: Activity, requestCode: Int) {
            val intent = Intent(context, CameraActivity::class.java)
            context.startActivityForResult(intent, requestCode)
        }

        fun start(context: Context, requestCode: Int, screenShotPath: String, id: Long) {
            val intent = Intent(context, CameraActivity::class.java)
            intent.putExtra(TranslationController.SCREENSHOT_PATH, screenShotPath)
            intent.putExtra(TranslationController.SCREENSHOT_ID, id)
            intent.flags = Intent.FLAG_ACTIVITY_NEW_TASK
            context.startActivity(intent)
        }
    }

    private lateinit var translationSession: TranslationSession
    private var mHomeReceiver: KeyEventReceiver = KeyEventReceiver()

    override fun onCreate(savedInstanceState: Bundle?) {
        //window.decorView.systemUiVisibility = View.SYSTEM_UI_FLAG_HIDE_NAVIGATION or View.SYSTEM_UI_FLAG_IMMERSIVE_STICKY
        super.onCreate(savedInstanceState)

        if (startHiTranslate()) {
            finish()
            return
        }

        translationSession = TranslationSessionFactory.buildDefaultSession()
        TranslationStateEngine.setHandler(translationSession, this)
        TranslationStateEngine.post(translationSession)

        val homeFilter = IntentFilter(Intent.ACTION_CLOSE_SYSTEM_DIALOGS)
        registerReceiver(mHomeReceiver, homeFilter)
    }

    private fun startHiTranslate(): Boolean {
        val intent = Intent("com.talpa.translate.CAMERA")
        intent.setPackage("com.zaz.translate")
        val list = packageManager.queryIntentActivities(intent, PackageManager.GET_META_DATA)
        if (list.isNotEmpty()) {
            startActivity(intent)
            return true
        }
        return false
    }

    override fun handle(translationSession: TranslationSession) {
        if (translationSession.getEvent() == Event.START_REQUEST_PERMISSION) {
            if (!hasPermission(this)) {
                ActivityCompat.requestPermissions(
                    this,
                    PERMISSIONS_REQUIRED,
                    REQUEST_CODE_PERMISSIONS
                )
            } else {
                setContentView(R.layout.activity_camerax)
                //TranslationStateEngine.post(TranslationSessionFactory.buildDefaultSession())
                dealIntent()

                TranslationStateEngine.post(translationSession
                    .apply {
                        setEvent(Event.CHECK_EVN)
                        setState(TranslationStateEngine.stateMachine.next(translationSession))
                    })

            }
        } else if (translationSession.getEvent() == Event.CHECK_EVN) {
            if (getFreeSpace() < 30 * 1024) {
                //todo 环境检查失败
                showNoticeDialog()
            } else {
                TranslationStateEngine.post(translationSession
                    .apply {
                        setEvent(Event.PREVIEW)
                        setState(TranslationStateEngine.stateMachine.next(translationSession))
                    })
            }
        }
    }

    override fun onDestroy() {
        super.onDestroy()
        unregisterReceiver(mHomeReceiver)
    }

    override fun onLackUsageConfirm() {
        finish()
    }

    private fun showNoticeDialog() {
        // Create an instance of the dialog fragment and show it
        val dialog = DiskAlertDialog()
            .also {
                it.isCancelable = false
            }
        dialog.show(supportFragmentManager, "DiskAlertDialog")
    }

    private fun dealIntent() {

        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.Q) {
            val screenshotId = intent.getLongExtra(TranslationController.SCREENSHOT_ID, -1L)
            if (screenshotId != -1L) {
                Log.d("cjslog", "screen shot:${screenshotId}")
                val f = supportFragmentManager.findFragmentById(R.id.container) as? CameraFragment
                if (f != null) {
                    val uri: Uri = ContentUris.withAppendedId(MediaStore.Images.Media.EXTERNAL_CONTENT_URI, screenshotId.toLong())

                    f.setScreenShotMode(true)
                    f.setImageFromUri(uri)
                }
            }
        } else {
            val screenshot = intent.getStringExtra(TranslationController.SCREENSHOT_PATH)
            if (!TextUtils.isEmpty(screenshot)) {
                Log.d("cjslog", "screen shot:${screenshot}")
                val f = supportFragmentManager.findFragmentById(R.id.container) as? CameraFragment
                if (f != null) {
                    val uri: Uri
                    val file = File(screenshot)
                    Log.d("cjslog", "file exsist:${file.exists()}")
                    if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
                        uri =
                            FileProvider.getUriForFile(this, getPackageName() + ".fileProvider", file);
                    } else {
                        uri = Uri.fromFile(file)
                    }

                    f.setScreenShotMode(true)
                    f.setImageFromUri(uri)
                }
            }
        }


    }

    private fun hasPermission(context: Context): Boolean {
        return PERMISSIONS_REQUIRED.all {
            PERMISSION_GRANTED == ContextCompat.checkSelfPermission(context, it)
        }
    }

    override fun onRequestPermissionsResult(
        requestCode: Int,
        permissions: Array<out String>,
        grantResults: IntArray
    ) {

        when (requestCode) {
            REQUEST_CODE_PERMISSIONS -> {
                //&& !supportFragmentManager.isStateSaved
                if (hasPermission(applicationContext)) {
                    //TranslationStateEngine.post(TranslationSessionFactory.buildDefaultSession())
                    setContentView(R.layout.activity_camerax)
                    dealIntent()
                } else {
                    finish()
                }
            }
        }

        super.onRequestPermissionsResult(requestCode, permissions, grantResults)
    }

    private inner class KeyEventReceiver : BroadcastReceiver() {

        override fun onReceive(context: Context, intent: Intent) {
            when(intent.action) {
                Intent.ACTION_CLOSE_SYSTEM_DIALOGS -> {
                    //Log.d("cjslog", "key home")
                    finish()
                }
            }
        }

    }

    override fun onKeyDown(keyCode: Int, event: KeyEvent?): Boolean {

        return when (keyCode) {
            KeyEvent.KEYCODE_BACK -> {
                val intent = Intent(KEY_EVENT_ACTION).apply {
                    putExtra(
                        KEY_EVENT_EXTRA, keyCode
                    )
                }
                LocalBroadcastManager.getInstance(applicationContext).sendBroadcast(intent)
                true
            }
            else -> super.onKeyDown(keyCode, event)
        }
    }

}
