package com.talpa.translate.camera


import android.annotation.SuppressLint
import android.content.*
import android.graphics.Bitmap
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.Bundle
import android.util.Log
import android.view.*
import android.widget.CheckBox
import android.widget.Checkable
import android.widget.ImageView
import android.widget.TextView
import androidx.annotation.Keep
import androidx.annotation.RestrictTo
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.toBitmap
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.*
import androidx.lifecycle.Observer
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.snackbar.Snackbar
import com.google.firebase.ml.common.FirebaseMLException
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.*
import com.talpa.translate.base.view.CaptureButton
import com.talpa.translate.base.view.wheel.WheelView
import com.talpa.translate.camera.view.CameraView
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.camera.view.controls.PictureFormat
import com.talpa.translate.datasource.CameraCompatSource
import com.talpa.translate.datasource.ICameraSource
import com.talpa.translate.ocr.ImageTranslate
import com.talpa.translate.ocr.OcrDispatcher
import kotlinx.android.synthetic.main.fragment_camera.*
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.delay
import kotlinx.coroutines.launch
import java.io.IOException
import java.lang.ref.WeakReference
import java.util.*


private const val TAG = "CameraFragment"

/**
 * Camera Preview
 *
 * @author CY 2020-02-03
 */
@Keep
@RestrictTo(RestrictTo.Scope.LIBRARY)
class CameraFragment : Fragment(), View.OnClickListener, ICameraSource.Status {

    private lateinit var mControlView: CaptureButton
    private var mPreviewResult: ImageView? = null
    private lateinit var mSelectImage: ImageView
    private var mOverlay: ImageView? = null
    private var mProgressView: View? = null

    /**
     * 当前状态
     */
    private var currentStatus = STATE_CAMERA

    private val sourceLiveData: MutableLiveData<String> = MutableLiveData<String>("en")
    private val targetLiveData: MutableLiveData<String> = MutableLiveData<String>("en")

    private val translateViewModel: ImageTranslate by viewModels {
        object : ViewModelProvider.Factory {
            override fun <T : ViewModel?> create(modelClass: Class<T>): T {
                return ImageTranslate(requireContext()) as T
            }

        }
    }

    private lateinit var cameraBroadcastReceiver: CameraBroadcastReceiver

    private var singleClickTime: Long = 0

    //private var ivToCamera: View? = null
    private var tvSourceLanguage: TextView? = null
    private var tvTargetLanguage: TextView? = null
    private var flashLight: CheckBox? = null
    private lateinit var mCameraSource: ICameraSource

    companion object {

        /*Camera state*/
        const val STATE_CAMERA = 100

        const val STATE_SELECT_IMAGE = 300

        const val STATE_RESULT_FROM_CAMERA_SUCCESS = 400
        const val STATE_RESULT_FROM_GALLERY_SUCCESS = 401

        const val STATE_RESULT_FROM_CAMERA_FAIL = 402
        const val STATE_RESULT_FROM_GALLERY_FAIL = 403

        const val STATE_TRANSLATING_CAMERA = 500
        const val STATE_TRANSLATING_GALLERY = 501


        const val REQUEST_CODE_SELECT_IMAGE = 1000
        const val REQUEST_CODE_SELECT_SOURCE_LANGUAGE = 1100
        const val REQUEST_CODE_SELECT_TARGET_LANGUAGE = 1200

        private const val PREFER_KEY_TARGET_LANG = "key_target_lang"

    }

    override fun onCreateView(
        inflater: LayoutInflater, container: ViewGroup?,
        savedInstanceState: Bundle?
    ): View? {
        return inflater.inflate(R.layout.fragment_camera, container, false)
    }

    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        mCameraSource = CameraCompatSource()
        //translateViewModel = ImageTranslate(requireContext())
        //mainExecutor = ContextCompat.getMainExecutor(requireContext())
        initPreviewView(view)
        initView(view)
        cameraBroadcastReceiver = CameraBroadcastReceiver(this)
        registerBroadcast()
        enterCameraState()
        readLanguage(view)
        handleIntent()
        mCameraSource.setStatusCallback(this)
    }

    override fun cameraInitializeFail(exception: Exception) {
        showErrorAlert(requireContext(), exception)
    }

    override fun onPicturetaked(data: ByteArray, metadata: FrameMetadata) {
        if (!isDetached && activity?.isDestroyed == false) {
            onImageSaved(data, metadata)
        }
    }

    override fun onPicturetakeFail(exception: java.lang.Exception) {
        showErrorAlert(requireActivity(), exception)
    }

    override fun onStatusChange(state: Int) {
        when (state) {
            ICameraSource.TORCH_OFF -> {
                flashLight?.isChecked = false
            }
            ICameraSource.TORCH_ON -> {
                flashLight?.isChecked = true
            }
        }
    }

    /**
     * Handle Intent
     */
    private fun handleIntent() {
        val intent = requireActivity().intent
        when (intent?.action) {
            Intent.ACTION_SEND -> {
                val uri = intent.getParcelableExtra<Uri?>(Intent.EXTRA_STREAM)

                if (uri != null) {
                    setImageFromGallery(uri)
                }
            }
        }
    }

    override fun onPause() {
        super.onPause()
        mCameraSource.torch(false)
    }

    override fun onDestroyView() {
        super.onDestroyView()
        unregisterBroadcast()
        translateViewModel.cleanUp()
    }

    private fun enterCameraState() {
        if (screenShotMode) {
            activity?.finish()
            return
        }
        Glide.with(context ?: return).clear(mPreviewResult ?: return)
        Glide.with(context ?: return).clear(mOverlay ?: return)

        currentStatus =
            STATE_CAMERA
        mControlView.visibility = View.VISIBLE
        mControlView.setState(mControlView.cameraState)
        mControlView.revertAnim()
        mControlView.isClickable = true

        mOverlay?.visibility = View.GONE
        //mControlView.setBackgroundResource(R.drawable.ic_camera_take)
        if (mPreviewResult?.visibility != View.GONE) {
            lifecycleScope.launch(Dispatchers.Main) {
                delay(500)
                mPreviewResult?.visibility = View.INVISIBLE
            }
        }

        flashLight?.visibility = View.VISIBLE
        mSelectImage.visibility = View.VISIBLE

        hideProgressBar()

        tvSourceLanguage?.isClickable = true
        tvTargetLanguage?.isClickable = true

        tvSourceLanguage?.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg
            )
        )
        tvTargetLanguage?.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg
            )
        )
        if (iv_exchange?.isEnabled == true) {
            iv_exchange?.alpha = 1.0f
            iv_exchange?.isClickable = true
        }
        mCameraSource.torch(mPreTorchState)
    }

    private fun enterTranslateState() {
        tvSourceLanguage?.isClickable = false
        tvTargetLanguage?.isClickable = false
        //mControlView.setState(mControlView.cameraState)
        //mControlView.setState(mControlView.cropState)
        mControlView.startAnim()
        mControlView.isClickable = false
        language_selector?.background?.alpha = 0x66
        language_selector?.isClickable = false
        tvSourceLanguage?.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg2
            )
        )
        tvTargetLanguage?.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg2
            )
        )
        iv_exchange?.alpha = 0.5f
        iv_exchange?.isClickable = false
        mOverlay?.visibility = View.INVISIBLE
        flashLight?.visibility = View.INVISIBLE
        mSelectImage.visibility = View.INVISIBLE
    }

    private fun enterResultState() {
        //currentStatus = STATE_RESULT_FROM_CAMERA
        hideProgressBar()
        tvSourceLanguage?.isClickable = true
        tvTargetLanguage?.isClickable = true
        mControlView.revertAnim()
        mControlView.setState(mControlView.resultState)
        mControlView.isClickable = true
        mControlView.visibility = View.VISIBLE
        language_selector?.background?.alpha = 0xff
        language_selector?.isClickable = true
        tvSourceLanguage?.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg
            )
        )
        tvTargetLanguage?.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg
            )
        )
        if (iv_exchange?.isEnabled == true) {
            iv_exchange?.alpha = 1.0f
            iv_exchange?.isClickable = false
        }
        flashLight?.visibility = View.INVISIBLE
        mSelectImage.visibility = View.INVISIBLE
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun initView(view: View) {
        mControlView = view.findViewById(R.id.btn_control)
        mControlView.setOnClickListener(this)
        mPreviewResult = view.findViewById(R.id.iv_preview)
        mOverlay = view.findViewById(R.id.overlay)
        mSelectImage = view.findViewById(R.id.iv_select_image)
        mSelectImage.setOnClickListener(this)
        mProgressView = view.findViewById(R.id.progress)
        iv_exchange?.setOnClickListener(this)
        view.findViewById<View>(R.id.language_selector).setOnClickListener(this)
        val finishView = view.findViewById<View>(R.id.iv_finish)
        finishView.setOnClickListener(this)
        flashLight = view.findViewById(R.id.torch_selector)
        flashLight?.setOnClickListener(this)
        tvSourceLanguage = view.findViewById(R.id.tv_source_language)
        tvTargetLanguage = view.findViewById(R.id.tv_target_language)
        tvSourceLanguage?.setOnClickListener(this)
        tvTargetLanguage?.setOnClickListener(this)
        sourceLiveData.observe(viewLifecycleOwner, Observer { language ->
            tvSourceLanguage?.text = if (language == LANG.AUTO) {
                getString(R.string.auto)
            } else {
                Locale.forLanguageTag(language).displayLanguage
            }
            //exchangeState()
        })
        targetLiveData.observe(viewLifecycleOwner, Observer { language ->
            tvTargetLanguage?.text = Locale.forLanguageTag(language).displayLanguage
            exchangeState()
        })

        mPreviewResult?.setOnTouchListener { v, event ->
            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
            ) {
                when(event.action) {
                    MotionEvent.ACTION_DOWN -> {
                        if (mOverlay?.visibility != View.GONE) {
                            mOverlay?.visibility = View.GONE
                        }
                    }
                    MotionEvent.ACTION_UP-> {
                        if (mOverlay?.visibility != View.VISIBLE) {
                            mOverlay?.visibility = View.VISIBLE
                        }
                    }
                }
            }
            return@setOnTouchListener true
        }
    }

    private fun readLanguage(view: View) {
        val prefer = context?.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE) ?: return
        val defaultTargetLang = LANG.AUTO

        val sourceLang =
            prefer.getString(PREFER_KEY_SOURCE_LANG, defaultTargetLang) ?: defaultTargetLang
        val targetLang =
            prefer.getString(PREFER_KEY_TARGET_LANG, Locale.getDefault().language)
                ?: Locale.getDefault().language
        sourceLiveData.value = sourceLang
        targetLiveData.value = targetLang
    }

    /***************************LANGUAGE START*********************************/

    /**
     * Picker for Language
     */
    @SuppressLint("InflateParams")
    private fun languagePicker(context: Context, anchorView: View) {
        val bottomSheetDialog = BottomSheetDialog(requireActivity())

        val contentView =
            LayoutInflater.from(context).inflate(R.layout.layout_language_picker, null)

        //val bmp: Bitmap = BitmapFactory.decodeResource(resources, R.drawable.picker_bg)

        bottomSheetDialog.setContentView(contentView)
        bottomSheetDialog.window?.findViewById<View>(R.id.design_bottom_sheet)
            ?.setBackgroundResource(android.R.color.transparent)

        val sourcePicker = contentView.findViewById<WheelView<CustomLocale>>(R.id.np_source_picker)
        val targetPicker = contentView.findViewById<WheelView<CustomLocale>>(R.id.np_target_picker)
        val submit = contentView.findViewById<View>(R.id.submit_language)

        /* ***********Source Picker*************/

        val supportSourceLanguages = if (OcrDispatcher.getGoogleOcrEnable()) {
                getLocaleListCompat(firebaseSourceSupportLanguages, googleVisionLanguages)
        } else {
                getLocaleListCompat(firebaseSourceSupportLanguages)
        }

        val supportSourceLanguageSize = supportSourceLanguages.size() + 1
        val sourceValues = Array(supportSourceLanguageSize) {
            if (it == 0) {
                CustomLocale(
                    LANG.AUTO,
                    getString(R.string.auto)
                )
            } else {
                CustomLocale(
                    supportSourceLanguages[it - 1].language,
                    supportSourceLanguages[it - 1].displayLanguage
                )
            }
        }

        val sourceLanguageTag = sourceLiveData.value
        val defaultSourceIndex = if (sourceLanguageTag != null) {
            var index = 0
            repeat(sourceValues.size) {
                if (sourceValues[it].languageTag == sourceLanguageTag) {
                    index = it
                    return@repeat
                }
            }
            index
        } else 0

        sourcePicker.setData(sourceValues.toList())
        sourcePicker.setSelectedItemPosition(defaultSourceIndex)

        /* ***********Target Picker*************/

        val supportTargetLanguages = getLocaleListCompat(targetSupportLanguages)
        val supportTargetLanguageSize = supportTargetLanguages.size()
        val targetValues = Array(supportTargetLanguageSize) {
            CustomLocale(
                supportTargetLanguages[it].language,
                supportTargetLanguages[it].displayLanguage
            )
        }
        //println("supportTargetLanguageSize=$supportTargetLanguageSize")
        val targetLanguageTag = targetLiveData.value
        val defaultTargetIndex = if (targetLanguageTag != null) {
            var index = 0
            for (i in 0..supportTargetLanguageSize) {
                val locale = supportTargetLanguages.get(i) ?: continue
                val tempTag = locale.language
                if (tempTag == targetLanguageTag) {
                    index = i
                    break
                }
            }

            index

        } else 0

        targetPicker.setData(targetValues.toList())
        targetPicker.setSelectedItemPosition(defaultTargetIndex)

        var sourceLocale = sourceValues[sourcePicker.getSelectedItemPosition()]
        val sourceSelectedListener = object : WheelView.OnItemSelectedListener<CustomLocale> {
            override fun onItemSelected(
                wheelView: WheelView<CustomLocale>,
                data: CustomLocale,
                position: Int
            ) {
                sourceLocale = wheelView.getItemData(position)
            }

        }

        var targetLocale = targetValues[targetPicker.getSelectedItemPosition()]
        val targetSelectedListener = object : WheelView.OnItemSelectedListener<CustomLocale> {
            override fun onItemSelected(
                wheelView: WheelView<CustomLocale>,
                data: CustomLocale,
                position: Int
            ) {
                targetLocale = wheelView.getItemData(position)
            }

        }

        sourcePicker.setOnItemSelectedListener(sourceSelectedListener)
        targetPicker.setOnItemSelectedListener(targetSelectedListener)

        submit.setOnClickListener {
            //readLanguage(submit)
            saveLanguage(context, PREFER_KEY_SOURCE_LANG, sourceLocale.languageTag)
            saveLanguage(
                context,
                PREFER_KEY_TARGET_LANG, targetLocale.languageTag
            )
            sourceLiveData.value = sourceLocale.languageTag
            targetLiveData.value = targetLocale.languageTag
            if (bottomSheetDialog.isShowing) {
                bottomSheetDialog.dismiss()
            }

            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                if (!context.isNetworkConnected()) {
                    Snackbar.make(anchorView, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return@setOnClickListener
                }

                mPreviewResult?.drawable?.toBitmap()?.let {
                    showProgressBar()
                    enterTranslateState()
                    if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS) {
                        currentStatus =
                            STATE_TRANSLATING_CAMERA
                    } else if (currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS) {
                        currentStatus =
                            STATE_TRANSLATING_GALLERY
                    }
                    translateViewModel.changeLanguage(
                        sourceLiveData.value ?: return@let,
                        targetLiveData.value ?: return@let
                    )
                        .observe(viewLifecycleOwner, Observer { result ->
                            if (result.isSuccess) {
                                recognitionSuccess(result.getOrNull())
                            } else {
                                recognitionFailure(result.exceptionOrNull())
                            }
                            enterResultState()
                        })
                }
            }

            changeLanguageEvent(
                sourceLiveData.value ?: return@setOnClickListener,
                targetLiveData.value ?: return@setOnClickListener
            )
        }

        bottomSheetDialog.setOnShowListener {
            language_selector?.visibility = View.INVISIBLE
        }
        bottomSheetDialog.setOnDismissListener {
            language_selector?.visibility = View.VISIBLE
        }

        targetPicker.setVibratorEnable(true)
        sourcePicker.setVibratorEnable(true)

        bottomSheetDialog.show()
    }

    /**
     * Exchange Language
     */
    private fun exchangeLanguage(context: Context) {
        val sourceLanguageTag = targetLiveData.value ?: return
        val targetLanguageTag = sourceLiveData.value ?: return
        saveLanguage(context, PREFER_KEY_SOURCE_LANG, sourceLanguageTag)
        saveLanguage(
            context,
            PREFER_KEY_TARGET_LANG, targetLanguageTag
        )
        sourceLiveData.value = sourceLanguageTag
        targetLiveData.value = targetLanguageTag
    }

    /***************************LANGUAGE END*********************************/

    /***************************CAMERA START*********************************/

    private fun initPreviewView(view: View) {
        val camera = view.findViewById<CameraView>(R.id.preview_container)
        mCameraSource.initialize(requireContext(), camera, this)
        //mPreviewView = view.findViewById(R.id.preview)
    }

    /**
     * Take Photo
     */
    private fun takePicture(context: Context) {
        mCameraSource.takePicture()
    }

    private var mMetadata: FrameMetadata? = null

    private fun doOcrTranslate(img: Any) {

        Glide.with(requireContext())
            .asBitmap()
            .load(img)
            .listener(object : RequestListener<Bitmap> {
                override fun onResourceReady(
                    resource: Bitmap,
                    model: Any?,
                    target: Target<Bitmap>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    //mCameraSource.pause()
                    mPreviewResult?.visibility = View.VISIBLE
                    val sourceLanguageTag = sourceLiveData.value ?: "en"
                    val targetLanguageTag = targetLiveData.value ?: "en"
                    //val sourceLanguage = localeToGoogleLanguage[sourceLanguageTag] ?: "en"
                    //val targetLanguage = localeToGoogleLanguage[targetLanguageTag] ?: "en"
                    showProgressBar()
                    //enterTranslateState()
                    if (mMetadata == null) {
                        mMetadata = FrameMetadata.Builder()
                            .setWidth(resource.width)
                            .setHeight(resource.height)
                            .setCameraFacing(Facing.BACK)
                            .setPictureFormat(PictureFormat.JPEG)
                            .setRotation(0)
                            .build()
                    }

                    translateImage(
                        resource,
                        sourceLanguageTag,
                        targetLanguageTag
                    ).observe(viewLifecycleOwner, Observer {
                        if (it.isSuccess) {
                            recognitionSuccess(it.getOrNull())
                        } else {
                            recognitionFailure(it.exceptionOrNull())
                        }
                        enterResultState()
                    })
                    logEvent(
                        Trans_start_translate,
                        hashMapOf(
                            language to "$sourceLanguageTag-$targetLanguageTag",
                            moduleType to modulePic
                        )
                    )
                    return false
                }

                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Bitmap>?,
                    isFirstResource: Boolean
                ): Boolean {
                    fail_toast?.visibility = View.VISIBLE
                    fail_toast_text?.text = getString(R.string.load_image_fail)
                    fail_toast?.postDelayed({ fail_toast?.visibility = View.GONE }, 2000)
                    enterCameraState()
                    return false
                }

            })
            .into(mPreviewResult ?: return)
    }

    private var mPreTorchState: Boolean = false

    /**
     * Take Photo Success
     */
    private fun onImageSaved(data: ByteArray, metadata: FrameMetadata) {
        mMetadata = metadata
        doOcrTranslate(data)
        mPreTorchState = flashLight?.isChecked ?: false
        mCameraSource.torch(switch = false)
    }

    private fun showErrorAlert(context: Context, exc: java.lang.Exception) {
        val onClickListener = DialogInterface.OnClickListener { dialog, which ->
            activity?.finish()
        }

        AlertDialog.Builder(context)
            .setTitle(android.R.string.dialog_alert_title)
            .setMessage(R.string.camera_unavailable)
            .setPositiveButton(android.R.string.ok, onClickListener)
            .setCancelable(false)
            .create()
            .show()
    }

    /***************************CAMERA END*********************************/

    /**
     * 翻译图片
     */
    private fun translateImage(
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<Bitmap>> {
        Log.d("cjslog", "source:${sourceLanguage} ${targetLanguage}")
        return translateViewModel.translate(
            bitmap,
            mMetadata,
            sourceLanguage,
            targetLanguage
        )
    }

    private fun recognitionFailure(throwable: Throwable? = null) {
        if (currentStatus == STATE_TRANSLATING_CAMERA) {
            currentStatus =
                STATE_RESULT_FROM_CAMERA_FAIL
        } else if (currentStatus == STATE_TRANSLATING_GALLERY) {
            currentStatus =
                STATE_RESULT_FROM_GALLERY_FAIL
        }
        showTranslateError(throwable)
    }

    /**
     * 识别翻译成功
     */
    private fun recognitionSuccess(newBitmap: Bitmap?) {
        //if (isDetached) return
        if (currentStatus == STATE_TRANSLATING_GALLERY) {
            currentStatus =
                STATE_RESULT_FROM_GALLERY_SUCCESS
            logEvent(PT__gallery_click_success)
            logEvent(
                Trans_translate_success,
                hashMapOf(
                    language to ("${sourceLiveData.value ?: return}-${targetLiveData.value ?: return}"),
                    moduleType to modulePic
                )
            )
        } else if (currentStatus == STATE_TRANSLATING_CAMERA) {
            currentStatus =
                STATE_RESULT_FROM_CAMERA_SUCCESS
            logEvent(PT__camera_click_success)
            logEvent(
                Trans_translate_success,
                hashMapOf(
                    language to ("${sourceLiveData.value ?: return}-${targetLiveData.value ?: return}"),
                    moduleType to modulePic
                )
            )
        } else if (currentStatus == STATE_RESULT_FROM_GALLERY_FAIL) {
            currentStatus =
                STATE_RESULT_FROM_GALLERY_SUCCESS
        } else if (currentStatus == STATE_RESULT_FROM_CAMERA_FAIL) {
            currentStatus =
                STATE_RESULT_FROM_CAMERA_SUCCESS
        }

        mOverlay?.let {
            Glide.with(context ?: return@let)
                .load(newBitmap)
                .listener(object : RequestListener<Drawable> {
                    override fun onResourceReady(
                        resource: Drawable?,
                        model: Any?,
                        target: Target<Drawable>?,
                        dataSource: DataSource?,
                        isFirstResource: Boolean
                    ): Boolean {
                        it.visibility = View.VISIBLE
                        return false
                    }

                    override fun onLoadFailed(
                        e: GlideException?,
                        model: Any?,
                        target: Target<Drawable>?,
                        isFirstResource: Boolean
                    ): Boolean {
                        return false
                    }

                })
                .into(it)
        }
        //mPreviewResult?.setImageBitmap(newBitmap)

    }


    /**
     * Show Progress
     */
    private fun showProgressBar() {
        mProgressView?.visibility = View.VISIBLE

    }

    /**
     * Hide Progress
     */
    private fun hideProgressBar() {
        mProgressView?.visibility = View.GONE
    }

    private fun showTranslateError(throwable: Throwable? = null) {
        val v = view ?: return
        val resId = when (throwable) {
            is FirebaseMLException -> {
                R.string.waiting_recognition_model_download
            }
            is IOException -> {
                R.string.network_unavailable
            }
            else -> {
                R.string.no_content_identified
            }
        }

        fail_toast?.visibility = View.VISIBLE
        fail_toast_text?.text = throwable?.message ?: ""
        fail_toast?.postDelayed({ fail_toast?.visibility = View.GONE }, 2000)

        translationFailEvent(
            currentStatus,
            throwable
        )
        logEvent(
            Trans_translate_failure,
            hashMapOf(
                language to ("${sourceLiveData.value ?: return}-${targetLiveData.value ?: return}"),
                moduleType to modulePic,
                errorMessage to (throwable?.message ?: "unknown error")
            )
        )
    }

    /***************************CROPPER WITH TRANSLATE END*********************************/

    /***************************SELECT START*********************************/
    /**
     * 设置图片
     */
    private fun setImageFromUri(uri: Uri) {
        currentStatus =
            STATE_TRANSLATING_GALLERY
        enterTranslateState()
        doOcrTranslate(uri)
    }

    fun setImageFromShot(uri: Uri) {
        setImageFromUri(uri)
        logEvent(PT_screen_shot_iden_click)
    }

    fun setImageFromGallery(uri: Uri) {
        setImageFromUri(uri)
        logEvent(PT_gallery_iden_click)
    }

    private var screenShotMode = false

    fun setScreenShotMode(mode: Boolean) {
        screenShotMode = mode
    }

    override fun onResume() {
        super.onResume()
        singleClickTime = System.currentTimeMillis()
    }

    /***************************SELECT END*********************************/

    override fun onClick(v: View) {

        val currentTime = System.currentTimeMillis()
        if (currentTime - singleClickTime < 500) {
            singleClickTime = currentTime
            return
        }

        when (v.id) {
            R.id.btn_control -> {

                if (!v.context.isNetworkConnected()) {
                    Snackbar.make(v, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return
                }

                when (currentStatus) {
                    STATE_CAMERA -> {
                        currentStatus =
                            STATE_TRANSLATING_CAMERA
                        enterTranslateState()
                        takePicture(v.context)
                        logEvent(PT_camera_click)
                    }
                    STATE_RESULT_FROM_CAMERA_FAIL,
                    STATE_RESULT_FROM_CAMERA_SUCCESS,
                    STATE_RESULT_FROM_GALLERY_SUCCESS,
                    STATE_RESULT_FROM_GALLERY_FAIL -> {
                        enterCameraState()
                        logEvent(PT_exit)
                    }
                }
            }
            R.id.iv_select_image -> {

                try {
                    val intent = Intent(Intent.ACTION_PICK)

                    intent.type = "image/*"

                    startActivityForResult(
                        intent,
                        REQUEST_CODE_SELECT_IMAGE
                    )
                    logEvent(PT_gallery_click)
                } catch (e: ActivityNotFoundException) {
                    fail_toast?.visibility = View.VISIBLE
                    fail_toast_text?.text = context?.getString(R.string.gallery_not_found)
                    fail_toast?.postDelayed({ fail_toast?.visibility = View.GONE }, 2000)
                }
            }
            R.id.torch_selector -> {
                val isChecked = (v as? Checkable)?.isChecked
                mCameraSource.torch(isChecked == true)
                //mCamera?.cameraControl?.enableTorch(isChecked == true)
                logEvent(PT_flash_click)
            }
            R.id.iv_finish -> {
                //requireActivity().finish()
                if (currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                ) {
                    enterCameraState()
                } else {
                    activity?.finish()
                }
            }
            R.id.tv_source_language, R.id.tv_target_language, R.id.language_selector -> {
                /*startActivityForSelectLanguage(
                    v.context,
                    REQUEST_CODE_SELECT_SOURCE_LANGUAGE
                )*/
                languagePicker(v.context, v)
                pickerClickedEvent(state = currentStatus)
            }
            R.id.iv_exchange -> {
                exchangeLanguage(v.context)
                logEvent(PT_lan_click_switch)
            }

        }
    }

    private fun exchangeState() {
        val targetLanguageTag = targetLiveData.value ?: return
        val sourceLanguageTag = sourceLiveData.value ?: return
        iv_exchange?.isEnabled = if (sourceLanguageTag == LANG.AUTO) {
            false
        } else {
            firebaseSourceSupportLanguages.contains(targetLanguageTag)
        }
        iv_exchange?.alpha = if (iv_exchange?.isEnabled == true) {
            1.0f
        } else {
            0.5f
        }
    }

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)

        when (requestCode) {
            REQUEST_CODE_SELECT_IMAGE -> {
                val uri = data?.data
                //println("uri===$uri")

                if (uri != null) {
                    setImageFromGallery(uri)
                }

            }
            REQUEST_CODE_SELECT_SOURCE_LANGUAGE -> {
                val languageTag = data?.getStringExtra("languageTag") ?: return
                saveLanguage(context ?: return, PREFER_KEY_SOURCE_LANG, languageTag)
            }
            REQUEST_CODE_SELECT_TARGET_LANGUAGE -> {
                val languageTag = data?.getStringExtra("languageTag") ?: return
                saveLanguage(
                    context ?: return,
                    PREFER_KEY_TARGET_LANG, languageTag
                )
            }
        }
    }

    private fun registerBroadcast() {
        LocalBroadcastManager.getInstance(requireContext())
            .registerReceiver(cameraBroadcastReceiver, IntentFilter(KEY_EVENT_ACTION))
    }

    private fun unregisterBroadcast() {
        LocalBroadcastManager.getInstance(requireContext())
            .unregisterReceiver(cameraBroadcastReceiver)
    }

    private class CameraBroadcastReceiver(cameraFragment: CameraFragment) : BroadcastReceiver() {
        private val weakReference = WeakReference(cameraFragment)
        override fun onReceive(context: Context, intent: Intent) {
            val fragment = weakReference.get() ?: return
            when (intent.action) {
                KEY_EVENT_ACTION -> {

                    when (intent.getIntExtra(KEY_EVENT_EXTRA, KeyEvent.KEYCODE_BACK)) {
                        KeyEvent.KEYCODE_BACK -> {
                            if (fragment.currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
                                || fragment.currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                                || fragment.currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                                || fragment.currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                            ) {
                                fragment.enterCameraState()
                            } else {
                                fragment.activity?.finish()
                            }
                        }
                    }
                }
            }
        }
    }
}