package com.talpa.translate.ocr

import android.content.Context
import android.graphics.Bitmap
import android.graphics.Rect
import android.util.Log
import com.google.api.client.http.javanet.NetHttpTransport
import com.google.api.client.json.JsonFactory
import com.google.api.client.json.gson.GsonFactory
import com.google.api.services.vision.v1.Vision
import com.google.api.services.vision.v1.VisionRequest
import com.google.api.services.vision.v1.VisionRequestInitializer
import com.google.api.services.vision.v1.model.*
import com.photo.translation.BuildConfig
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.TYPE_GOOGLE_VISION
import com.talpa.translate.base.utils.getSignature
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.result.Block
import com.talpa.translate.ocr.result.OcrResult
import java.io.IOException

/**
 * Create by chenjunsheng on 2020/7/29
 */
class GoogleRecognizer(var context: Context) : Recognizer {

    companion object {
        const val ANDROID_PACKAGE_HEADER = "X-Android-Package"
        const val ANDROID_CERT_HEADER = "X-Android-Cert"
        const val TAG = "Recognizer"
        const val TEST_PACKAGE_NAME = "com.talpa.translate"
        const val TEST_SHA1 = "ebd663baf5c6e7e4139675407b8216914bca417a"
    }

    private lateinit var mTransParams: TransParams
    private var mHistory: OcrResult? = null

    override fun setup(transParams: TransParams) {
        mTransParams = transParams
        Log.d("cjslog", "created Cloud Vision request object, sending request")
    }

    override suspend fun doOcr(bitmap: Bitmap): OcrResult {
        Log.d(TAG, "start google ocr")

        val httpTransport = NetHttpTransport()
        val jsonFactory: JsonFactory = GsonFactory.getDefaultInstance()
        val requestInitializer: VisionRequestInitializer =
            object : VisionRequestInitializer(BuildConfig.CLOUD_VISION_API_KEY) {
                /**
                 * We override this so we can inject important identifying fields into the HTTP
                 * headers. This enables use of a restricted cloud platform API key.
                 */
                @Throws(IOException::class)
                override fun initializeVisionRequest(visionRequest: VisionRequest<*>) {
                    super.initializeVisionRequest(visionRequest)
                    if (BuildConfig.DEBUG) {
                        val packageName: String = TEST_PACKAGE_NAME
                        visionRequest.requestHeaders[ANDROID_PACKAGE_HEADER] = packageName
                        //val sig: String? = getSignature(context.packageManager, packageName)
                        visionRequest.requestHeaders[ANDROID_CERT_HEADER] = TEST_SHA1
                    } else {
                        val packageName: String = context.packageName
                        visionRequest.requestHeaders[ANDROID_PACKAGE_HEADER] = packageName
                        val sig: String? = getSignature(context.packageManager, packageName)
                        visionRequest.requestHeaders[ANDROID_CERT_HEADER] = sig
                    }
                }
            }

        val builder = Vision.Builder(httpTransport, jsonFactory, null)
        builder.setVisionRequestInitializer(requestInitializer)
        val vision = builder.build()
        val batchAnnotateImagesRequest = BatchAnnotateImagesRequest().apply {
            requests = arrayListOf(AnnotateImageRequest()
                .also
                { imageRequest ->

                    // Add the image
                    val base64EncodedImage = Image()
                    // Convert the bitmap to a JPEG
                    // Just in case it's a format that Android understands but Cloud Vision
                    base64EncodedImage.content = PhotoAnalyzer.getBase64String(bitmap)
                    imageRequest.image = base64EncodedImage

                    // add the features we want
                    imageRequest.features = arrayListOf(Feature().also { feature ->
                        feature.type = "TEXT_DETECTION"
                    })
                })
        }


        val annotateRequest =
            vision.images().annotate(batchAnnotateImagesRequest)
        // Due to a bug: requests to Vision API containing large images fail when GZipped.
        annotateRequest.disableGZipContent = true

        val response: BatchAnnotateImagesResponse = annotateRequest.execute()

        val blocks = arrayListOf<Block>()
        Log.d("cjslog", "response:${response.size}")
        response.responses?.forEach { response ->
            val wholeStr = response.fullTextAnnotation?.text ?: return@forEach
            val sliceStr = wholeStr.split("\n")
            var index = 1
            sliceStr.forEach {
                val str = it.replace("\\s".toRegex(), "")
                if (str.isEmpty()) {
                    return@forEach
                }
                var strLen = 0
                val blockList = arrayListOf<EntityAnnotation>()
                kotlin.run {
                    response.textAnnotations.drop(index).forEach { entity ->
                        strLen += entity.description?.length ?: 0
                        if (strLen > str.length) {
                            strLen = 0
                            return@run
                        }
                        blockList.add(entity)
                        index++
                    }
                }

                //val blockList = response.textAnnotations.slice(index until index + it.length)

                var rect = Rect()
                blockList.forEachIndexed { index, entityAnnotation ->
                    val x0 = entityAnnotation.boundingPoly?.vertices?.get(0)?.x ?: 0
                    val y0 = entityAnnotation.boundingPoly?.vertices?.get(0)?.y ?: 0
                    val x1 = entityAnnotation.boundingPoly?.vertices?.get(2)?.x ?: 0
                    val y1 = entityAnnotation.boundingPoly?.vertices?.get(2)?.y ?: 0
                    if (index == 0) {
                        rect.set(x0, y0, x1, y1)
                    } else {
                        rect.union(x0, y0, x1, y1)
                    }
                }
                val metadata = mTransParams.metaData
                if (rect.right >= metadata.width) {
                    rect.right = metadata.width - 1
                }
                if (rect.bottom >= metadata.height) {
                    rect.bottom = metadata.height - 1
                }
                if (rect.left <= 0) {
                    rect.left = 1
                }
                if (rect.top <= 0) {
                    rect.top = 1
                }
                blockList.clear()
                if (rect.width() <= 0 || rect.height() <= 0) {
                    return@forEach
                }
                val block = Block(str, rect)
                blocks.add(block)
            }
            /*response.fullTextAnnotation?.pages?.forEach { page ->
                page.blocks?.forEach { block ->
                    val stringBuilder = StringBuilder()
                    block.paragraphs.forEach {
                        it.words.forEach { word ->
                            word.symbols.forEach { symbol ->
                                stringBuilder.append(symbol.text)
                            }
                            //stringBuilder.append(" ")
                        }
                    }
                    val text = stringBuilder.toString() //2
                    var left = block.boundingBox.vertices[0].x ?: return@forEach
                    var top = block.boundingBox.vertices[0].y ?: return@forEach
                    var right = block.boundingBox.vertices[2].x ?: return@forEach
                    var bottom = block.boundingBox.vertices[2].y ?: return@forEach

                    val metadata = mDataSource.getMetaData()
                    if (right >= metadata.width) {
                        right = metadata.width - 1
                    }
                    if (bottom >= metadata.height) {
                        bottom = metadata.height - 1
                    }
                    if (left <= 0) {
                        left = 1
                    }
                    if (top <= 0) {
                        top = 1
                    }
                    val rect = Rect(left, top, right, bottom)
                    if (rect.width() > 0 && rect.height() > 0) {
                        val block1 = Block(text, rect)
                        blocks.add(block1)
                    }
                }
            }*/
        }
        Log.d("cjslog", "google ocr finish")
        return OcrResult(blocks).also { mHistory = it }

    }

    override fun getFrameMetadata(): FrameMetadata? {
        return mTransParams.metaData
    }

    override fun getHistory(): OcrResult? {
        return mHistory
    }

    override fun getType(): Int {
        return TYPE_GOOGLE_VISION
    }
}