package com.talpa.translate.ocr

import android.content.Context
import android.graphics.Bitmap
import android.os.Build
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.ViewModel
import androidx.lifecycle.asLiveData
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.factory.TranslatorFactory
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.datasource.TransParams
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.render.Render
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.onStart


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */
class ImageTranslate constructor(private val context: Context) : ViewModel() {

    companion object {
        const val TAG = "ocr_translate"
    }

    private lateinit var mTranslator: Recognizer
    private val mRender: Render

    fun cleanUp() {
        mRender.release()
    }

    init {
        val analyzer: IAnalyzer
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            analyzer = PhotoAnalyzer(context)
        } else {
            analyzer = PhotoAnalyzerCompat(context)
        }
        mRender = TranslatorFactory.getRender(context, analyzer)
    }

    /**
     * 翻译
     */
    fun translate(
        bitmap: Bitmap,
        metadata: FrameMetadata?,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<Bitmap>> {
        return flow {
            val ocrResult = mTranslator.doOcr(mRender.getOcrBitmap()) //Step1:执行OCR
            val texts = arrayListOf<String>()
            ocrResult.blocks.forEach { texts.add(it.text) }
            if (texts.isEmpty()) { throw NoContentException(context.getString(R.string.no_trans_for_ocr)) }
            val transResult = HiTranslator.getInstance(context)
                .postTranslate(from = sourceLanguage, to = targetLanguage, texts = texts)  //Step2:OCR结果提交翻译
            val textsResponse = transResult.result?.texts ?: throw NoContentException(context.getString(R.string.translate_fail))
            if (textsResponse.isEmpty()) throw NoContentException(context.getString(R.string.no_trans_for_ocr))
            //Step3:结果渲染
            val result = mRender.renderOverlay(mTranslator, textsResponse) ?: throw NoContentException(context.getString(R.string.generate_bitmap_fail))
            emit(Result.success(result))
        }.onStart {
            mTranslator = OcrDispatcher.dispatchOcrTranslator(context, sourceLanguage)
            mRender.initialize(bitmap)
            mTranslator.setup(TransParams(checkNotNull(metadata), sourceLanguage))
        }.catch {
            Log.d("cjslog", "translate error", it)
            emit(Result.failure(it))
        }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun changeLanguage(sourceLanguage: String, targetLanguage: String): LiveData<Result<Bitmap>> {
        return flow {
            val history = mTranslator.getHistory() ?: throw NoContentException(
                context.getString(R.string.no_trans_for_ocr)
            )

            val texts = arrayListOf<String>()
            history.blocks.forEach { block ->
                texts.add(block.text)
            }
            val transResult = HiTranslator.getInstance(context)
                .postTranslate(from = sourceLanguage, to = targetLanguage, texts = texts)

            //Log.d("cjslog", "trasn result change:${transResult}")

            val textsResponse = transResult.result?.texts ?: throw NoContentException(
                context.getString(R.string.translate_fail)
            )
            if (textsResponse.isEmpty()) throw NoContentException(
                context.getString(R.string.no_trans_for_ocr)
            )
            val result =
                mRender.renderOverlay(mTranslator, textsResponse) ?: throw NoContentException(
                    context.getString(R.string.generate_bitmap_fail)
                )

            emit(Result.success(result))
        }.catch { emit(Result.failure(it)) }.flowOn(Dispatchers.IO)
            .asLiveData()
    }


}