package com.talpa.translate.camera


import android.annotation.SuppressLint
import android.content.ActivityNotFoundException
import android.content.Context
import android.content.DialogInterface
import android.content.Intent
import android.graphics.Bitmap
import android.graphics.Typeface
import android.graphics.drawable.Drawable
import android.net.Uri
import android.os.Bundle
import android.text.TextUtils
import android.view.LayoutInflater
import android.view.MotionEvent
import android.view.View
import android.widget.CheckBox
import android.widget.Checkable
import android.widget.Toast
import androidx.activity.OnBackPressedCallback
import androidx.annotation.RestrictTo
import androidx.appcompat.app.AlertDialog
import androidx.core.content.ContextCompat
import androidx.core.graphics.drawable.toBitmap
import androidx.core.os.LocaleListCompat
import androidx.core.view.forEachIndexed
import androidx.fragment.app.Fragment
import androidx.fragment.app.viewModels
import androidx.lifecycle.*
import androidx.lifecycle.Observer
import androidx.viewpager.widget.ViewPager
import com.bumptech.glide.Glide
import com.bumptech.glide.load.DataSource
import com.bumptech.glide.load.engine.GlideException
import com.bumptech.glide.request.RequestListener
import com.bumptech.glide.request.target.Target
import com.fasterxml.jackson.databind.ObjectMapper
import com.google.android.material.bottomsheet.BottomSheetDialog
import com.google.android.material.snackbar.Snackbar
import com.gyf.immersionbar.BarHide
import com.gyf.immersionbar.ImmersionBar
import com.gyf.immersionbar.ktx.immersionBar
import com.photo.translation.R
import com.photo.translation.databinding.FragmentCameraBinding
import com.talpa.tengine.lang.LANG
import com.talpa.translate.base.adapter.Feature
import com.talpa.translate.base.adapter.FeatureAdapter
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.state.PhotoMLState
import com.talpa.translate.base.state.PhotoMLState.Companion.TYPE_OBJECT
import com.talpa.translate.base.state.PhotoMLState.Companion.TYPE_TEXT
import com.talpa.translate.base.utils.*
import com.talpa.translate.base.view.wheel.WheelView
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.camera.view.controls.PictureFormat
import com.talpa.translate.camera.view.gesture.ViewPageGestureWrapper
import com.talpa.translate.camera.view.preview.ViewTouchListener
import com.talpa.translate.datasource.CameraCompatSource
import com.talpa.translate.datasource.ICameraSource
import com.talpa.translate.imglabel.ImageLabelData
import com.talpa.translate.ocr.CombineViewModel
import com.talpa.translate.ocr.ImageTranslate
import com.talpa.translate.ocr.OcrDispatcher
import com.talpa.translate.ocr.Utils
import com.talpa.translate.ocr.datasource.CompleteResult
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.ocr.result.CompleteTransfer
import com.talpa.translate.ocr.result.ContrastActivity
import com.talpa.translate.repository.box.collins.SenseNew
import com.talpa.translate.repository.box.translate.TranslateHistory1
import com.tapla.textspeech.TextSpeech
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import java.util.*


/**
 * Camera Preview
 *
 * @author CY 2020-02-03
 */

const val COMPLETE_RESULT = "complete_result"

@RestrictTo(RestrictTo.Scope.LIBRARY)
class CameraFragment : Fragment(R.layout.fragment_camera), View.OnClickListener,
    ICameraSource.Status, ViewTouchListener {

    internal lateinit var binding: FragmentCameraBinding

    /**
     * 当前状态
     */
    internal var currentStatus = STATE_CAMERA

    internal var mCompleteResult: CompleteResult? = null
    internal var mItemRecognizeResult: ImageLabelData? = null
    private var mViewPageGestureWrapper: ViewPageGestureWrapper? = null

    internal val translateViewModel: ImageTranslate by lazy {
        ViewModelProvider(
            this,
            ViewModelProvider.AndroidViewModelFactory.getInstance(activity?.application!!)
        ).get(ImageTranslate::class.java)
    }

    internal lateinit var mCameraSource: CameraCompatSource

    companion object {

        /*Camera state*/
        const val STATE_CAMERA = 100

        const val STATE_SELECT_IMAGE = 300

        const val STATE_RESULT_FROM_CAMERA_SUCCESS = 400
        const val STATE_RESULT_FROM_GALLERY_SUCCESS = 401

        const val STATE_RESULT_FROM_CAMERA_FAIL = 402
        const val STATE_RESULT_FROM_GALLERY_FAIL = 403

        const val STATE_TRANSLATING_CAMERA = 500
        const val STATE_TRANSLATING_GALLERY = 501

        const val REQUEST_CODE_SELECT_IMAGE = 1000
        const val REQUEST_CODE_SELECT_SOURCE_LANGUAGE = 1100
        const val REQUEST_CODE_SELECT_TARGET_LANGUAGE = 1200

        const val EXTRA_DATA = "data"
        const val ACTION_FOR_DICTIONARY_TRANS =
            "com.talpa.translate.ACTION_FOR_DICTIONARY_TRANS_DETAIL"
    }

    internal var mState: PhotoMLState = PhotoTranslateState()
        set(value) {
            field.onStateChange(false)
            value.onStateChange(true)
            field = value
        }
    internal val photoTranslateState = PhotoTranslateState()
    internal val itemRecognizeState = ItemRecognizeState()
    internal var mFeatureAdapter: FeatureAdapter? = null

    abstract inner class AbstractState : PhotoMLState {
        override fun enterCameraState() {
            if (screenShotMode) {
                activity?.finish()
                return
            }
            mCompleteResult = null
            mItemRecognizeResult = null

            currentStatus = STATE_CAMERA
            binding.btnControl.visibility = View.VISIBLE
            binding.btnControl.setState(binding.btnControl.cameraState)
            binding.btnControl.revertAnim()
            binding.btnControl.isClickable = true

            binding.overlay.visibility = View.GONE
            binding.ivPreview.visibility = View.INVISIBLE

            binding.torchSelector.setOnClickListener(this@CameraFragment)

            binding.torchSelector.setButtonDrawable(R.drawable.selector_torch)
            binding.ivSelectImage.setImageResource(R.drawable.icon_gallery)

            hideProgressBar()

            binding.tvSourceLanguage.isClickable = true
            binding.tvTargetLanguage.isClickable = true
            binding.tabLayout.isEnabled = true

            binding.tvSourceLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            binding.tvTargetLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            if (binding.ivExchange.isEnabled) {
                binding.ivExchange.alpha = 1.0f
                binding.ivExchange.isClickable = true
            }
            mCameraSource.torch(mPreTorchState)
            binding.ivPreview.attacher.resetMatrix()
            binding.overlay.attacher.resetMatrix()
        }

        protected open fun enterResultState(isSuccess: Boolean) {
            hideProgressBar()
            binding.tabLayout.isEnabled = false
            binding.tvSourceLanguage.isClickable = true
            binding.tvTargetLanguage.isClickable = true
            binding.btnControl.revertAnim()
            binding.btnControl.setState(binding.btnControl.resultState)
            binding.btnControl.isClickable = true
            binding.btnControl.visibility = View.VISIBLE
            binding.languageSelector.background?.alpha = 0xff
            binding.languageSelector.isClickable = true
            binding.tvSourceLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            binding.tvTargetLanguage.setTextColor(
                ContextCompat.getColor(
                    requireContext(),
                    R.color.camera_tool_bg
                )
            )
            if (binding.ivExchange.isEnabled) {
                binding.ivExchange.alpha = 1.0f
                binding.ivExchange.isClickable = false
            }
        }

    }

    inner class PhotoTranslateState : AbstractState() {

        override fun onStateChange(focus: Boolean) {
            if (focus) {
                translateViewModel.sourceLiveData.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvSourceLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
                translateViewModel.targetLiveData.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvTargetLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
            } else {
                translateViewModel.sourceLiveData.removeObservers(this@CameraFragment)
                translateViewModel.targetLiveData.removeObservers(this@CameraFragment)
            }
        }

        override fun getType(): Int {
            return TYPE_TEXT
        }

        override fun onPictureTaken(
            bitmap: Bitmap,
            sourceLanguage: String,
            targetLanguage: String
        ) {
            translateImage(
                bitmap,
                sourceLanguage,
                targetLanguage
            )?.observe(viewLifecycleOwner, Observer {
                if (it.isSuccess) {
                    mCompleteResult = it.getOrNull()
                    val completeBitmap = mCompleteResult?.renderResult
                    recognitionSuccess(completeBitmap)
                } else {
                    recognitionFailure(it.exceptionOrNull())
                }
                enterResultState(it.isSuccess)
            })
            logEvent(
                Trans_start_translate,
                hashMapOf(
                    language to "$sourceLanguage-$targetLanguage",
                    moduleType to modulePic
                )
            )
        }

        override fun enterCameraState() {
            super.enterCameraState()
            binding.torchSelector.visibility = View.VISIBLE
            binding.ivSelectImage.visibility = View.VISIBLE
            binding.ivSelectImage.setOnClickListener(this@CameraFragment)
            binding.include.root.visibility = View.GONE
            readLanguage()

            /*binding.overlay.setOnScaleChangeListener { scaleFactor, focusX, focusY ->
                binding.ivPreview.setScale(binding.overlay.scale, focusX, focusY, false)
            }
            binding.ivPreview.setOnScaleChangeListener { scaleFactor, focusX, focusY ->
                binding.overlay.setScale(binding.ivPreview.scale, focusX, focusY, false)
            }*/
            binding.overlay.setOnMatrixChangeListener {
                if (binding.ivPreview.displayRect != it) {
                    binding.ivPreview.setDisplayMatrix(binding.overlay.imageMatrix)
                }
            }
            binding.ivPreview.setOnMatrixChangeListener {
                if (binding.overlay.displayRect != it) {
                    binding.overlay.setDisplayMatrix(binding.ivPreview.imageMatrix)
                }
            }
        }

        override fun getSupportSourceLanguage(): LocaleListCompat {
            return if (OcrDispatcher.getGoogleOcrEnable()) {
                getLocaleListCompat(firebaseSourceSupportLanguages, googleVisionLanguages)
            } else {
                getLocaleListCompat(firebaseSourceSupportLanguages)
            }
        }

        override fun getSupportTargetLanguage(): LocaleListCompat {
            return getLocaleListCompat(targetSupportLanguages)
        }

        override fun changeLanguageAfter(sourceLocale: CustomLocale, targetLocale: CustomLocale) {
            val context = context ?: return
            saveLanguage(context, PREFER_KEY_SOURCE_LANG, sourceLocale.languageTag)
            saveLanguage(
                context,
                PREFER_KEY_TARGET_LANG, targetLocale.languageTag
            )
            translateViewModel.getSourceData(mState).value = sourceLocale.languageTag
            translateViewModel.getTargetData(mState).value = targetLocale.languageTag

            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                binding.ivPreview.drawable?.toBitmap()?.let {
                    showProgressBar()
                    enterTranslateState()
                    if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS) {
                        currentStatus =
                            STATE_TRANSLATING_CAMERA
                    } else if (currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS) {
                        currentStatus =
                            STATE_TRANSLATING_GALLERY
                    }
                    mCompleteResult?.let { result ->
                        if (result.sourceLanguage != sourceLocale.languageTag) {
                            translateImage(
                                it,
                                sourceLocale.languageTag,
                                targetLocale.languageTag
                            )?.observe(viewLifecycleOwner, Observer {
                                if (it.isSuccess) {
                                    mCompleteResult = it.getOrNull()
                                    val completeBitmap = mCompleteResult?.renderResult
                                    recognitionSuccess(completeBitmap)
                                } else {
                                    recognitionFailure(it.exceptionOrNull())
                                }
                                enterResultState(it.isSuccess)
                            })
                        } else {
                            translateViewModel.changeTargetLanguage(
                                it,
                                translateViewModel.getSourceData(mState).value ?: return@let,
                                translateViewModel.getTargetData(mState).value ?: return@let
                            )
                                .observe(viewLifecycleOwner, Observer { result ->
                                    if (result.isSuccess) {
                                        mCompleteResult = result.getOrNull()
                                        recognitionSuccess(mCompleteResult?.renderResult)
                                    } else {
                                        recognitionFailure(result.exceptionOrNull())
                                    }
                                    enterResultState(result.isSuccess)
                                })
                        }
                    } ?: kotlin.run {
                        recognitionFailure(NoContentException(context?.getString(R.string.no_content_identified)))
                        enterResultState(false)
                    }

                }
            }
        }

        override fun enterResultState(isSuccess: Boolean) {
            //currentStatus = STATE_RESULT_FROM_CAMERA
            super.enterResultState(isSuccess)
            binding.ivPreview.setOnViewTapListener { view, x, y ->
                if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                ) {
                    binding.overlay.visibility = View.VISIBLE
                }
            }
            binding.overlay.setOnViewTapListener { view, x, y ->
                if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                ) {
                    binding.overlay.visibility = View.GONE
                }
            }

            if (isSuccess) {
                binding.torchSelector.visibility = View.VISIBLE
                binding.torchSelector.setButtonDrawable(R.drawable.ic_contrast)
                binding.torchSelector.setOnClickListener {
                    val bundle = Bundle()
                    bundle.putBinder(COMPLETE_RESULT, CompleteTransfer(mCompleteResult))
                    //bundle.putBinder()
                    //findNavController().navigate(R.id.contrastFragment)
                    val intent =
                        Intent(context ?: return@setOnClickListener, ContrastActivity::class.java)
                    intent.putExtras(bundle)
                    startActivity(intent)
                    logEvent(PT_camera_contrast)
                }
                binding.ivSelectImage.setImageResource(R.drawable.ic_save_img)
                binding.ivSelectImage.visibility = View.VISIBLE
                binding.ivSelectImage.setOnClickListener {
                    saveToGallery(binding.overlay.visibility == View.VISIBLE)
                }
            }
            binding.include.dict.isClickable = true
            binding.include.sound.isClickable = true
        }
    }

    inner class ItemRecognizeState : AbstractState() {

        override fun onStateChange(focus: Boolean) {
            if (focus) {
                translateViewModel.objectSource.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvSourceLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
                translateViewModel.objectTarget.observe(
                    this@CameraFragment,
                    Observer { language ->
                        binding.tvTargetLanguage.text =
                            Locale.forLanguageTag(language).displayLanguage
                        //exchangeState()
                    })
                logEvent(LO_Identify_objects)
            } else {
                translateViewModel.objectSource.removeObservers(this@CameraFragment)
                translateViewModel.objectTarget.removeObservers(this@CameraFragment)
            }
        }

        override fun getType(): Int {
            return TYPE_OBJECT
        }

        override fun onPictureTaken(
            bitmap: Bitmap,
            sourceLanguage: String,
            targetLanguage: String
        ) {
            translateViewModel.itemRecognize(
                bitmap,
                sourceLanguage = sourceLanguage,
                metadata = mMetadata ?: return,
                targetLanguage = targetLanguage
            ).observe(this@CameraFragment, Observer {
                if (it.isSuccess) {
                    mItemRecognizeResult = it.getOrNull()
                    if (mItemRecognizeResult != null && mItemRecognizeResult!!.source.isNotEmpty() && mItemRecognizeResult!!.target.isNotEmpty()) {
                        labelSuccess(mItemRecognizeResult!!)
                    } else {
                        recognitionFailure(it.exceptionOrNull())
                    }

                } else {
                    recognitionFailure(it.exceptionOrNull())
                }
                enterResultState(it.isSuccess)
            })
        }

        override fun enterResultState(isSuccess: Boolean) {
            super.enterResultState(isSuccess)
            binding.ivPreview.setOnViewTapListener(null)
            binding.overlay.setOnViewTapListener(null)
        }

        private fun showDict(): Boolean {
            return translateViewModel.getTargetData(mState).value == "en"
        }

        private fun labelSuccess(labelData: ImageLabelData) {
            binding.include.root.visibility = View.VISIBLE
            binding.include.word.text = labelData.target[0].data?.translated
            binding.include.dict.tag = labelData.target[0]
            binding.include.dict.visibility = if (showDict()) {
                View.VISIBLE
            } else {
                View.GONE
            }
            val pronunciation = labelData.target[0].data?.pronunciation
            if (TextUtils.isEmpty(pronunciation) || showDict().not()) {
                binding.include.prounce.visibility = View.GONE
            } else {
                binding.include.prounce.visibility = View.VISIBLE
                binding.include.prounce.text = pronunciation
            }
            binding.include.replacement.removeAllViews()
            labelData.source.forEachIndexed { index, senseNew ->
                val item = LayoutInflater.from(context ?: return@forEachIndexed)
                    .inflate(
                        R.layout.image_label_tab_item,
                        binding.include.replacement,
                        false
                    ) as CheckBox
                item.text = senseNew.data?.translated
                item.tag = labelData.target[index]
                if (index == 0) {
                    item.typeface = Typeface.DEFAULT_BOLD
                }
                item.setOnClickListener {
                    val checkBox = it as CheckBox
                    binding.include.replacement.forEachIndexed { index, view ->
                        val checkedView: CheckBox = view as CheckBox
                        if (it != checkedView) {
                            if (TextSpeech.isSpeaking()) {
                                TextSpeech.stopSpeak()
                            }
                            if (checkedView.isChecked) {
                                checkedView.isChecked = false
                            }
                            checkedView.typeface = Typeface.SANS_SERIF
                        }
                    }
                    checkBox.typeface = Typeface.DEFAULT_BOLD
                    //checkBox.isChecked = true
                    val targetSense = checkBox.tag as SenseNew
                    binding.include.dict.tag = targetSense
                    binding.include.word.text = targetSense.data?.translated
                    val pronunciation = targetSense.data?.pronunciation
                    if (TextUtils.isEmpty(pronunciation) || showDict().not()) {
                        binding.include.prounce.visibility = View.GONE
                    } else {
                        binding.include.prounce.visibility = View.VISIBLE
                        binding.include.prounce.text = pronunciation
                    }

                    if (showDict()) {
                        val sb = StringBuilder()
                        val defSize = targetSense.data?.definition?.size ?: 0
                        targetSense.data?.definition?.forEachIndexed { index, definition ->
                            sb.append(definition.speechPart)
                            sb.append(".")
                            sb.append(definition.source)
                            if (index < defSize - 1) {
                                sb.append("\n")
                            }
                        }

                        binding.include.explainTv.text = sb.toString()
                        binding.include.explainTv.visibility = View.VISIBLE
                    } else {
                        binding.include.explainTv.visibility = View.GONE
                    }
                    binding.include.dict.visibility = if (showDict()) {
                        View.VISIBLE
                    } else {
                        View.GONE
                    }
                    logEvent(LO_to_action, hashMapOf("type" to "word"))
                }
                binding.include.replacement.addView(item)
            }
            (binding.include.replacement.getChildAt(0) as CheckBox).isChecked = true
            if (TextSpeech.isLanguageAvailable(
                    Locale.forLanguageTag(
                        translateViewModel.getTargetData(
                            mState
                        ).value ?: return
                    )
                )
            ) {
                binding.include.sound.visibility = View.VISIBLE
            } else {
                binding.include.sound.visibility = View.INVISIBLE
            }

            binding.include.dict.setOnClickListener {
                if (TextSpeech.isSpeaking()) {
                    TextSpeech.stopSpeak()
                }
                val intent = Intent()
                intent.action = ACTION_FOR_DICTIONARY_TRANS
                val senseNew = it.tag as SenseNew
                val mapper = ObjectMapper()
                intent.putExtra(
                    EXTRA_DATA, TranslateHistory1(
                        text = senseNew.data?.word ?: return@setOnClickListener,
                        sourceLanguageTag = translateViewModel.getSourceData(mState).value
                            ?: return@setOnClickListener,
                        targetLanguageTag = translateViewModel.getTargetData(mState).value
                            ?: return@setOnClickListener,
                        millis = System.currentTimeMillis(),
                        star = false,
                        scene = 0,
                        sceneJson = if (showDict()) {
                            mapper.writeValueAsString(senseNew)
                        } else {
                            ""
                        },
                        history = false,
                        translation = senseNew.data?.translated ?: return@setOnClickListener
                    )
                )
                startActivity(intent)
                logEvent(LO_to_action, hashMapOf("type" to "dict"))
            }

            binding.include.sound.setOnClickListener {
                if (TextSpeech.isSpeaking()) {
                    TextSpeech.stopSpeak()
                }
                TextSpeech.speak(
                    binding.include.word.text.toString(),
                    Locale.forLanguageTag(
                        translateViewModel.getTargetData(mState).value ?: return@setOnClickListener
                    )
                )
                logEvent(LO_to_action, hashMapOf("type" to "speak"))
            }

            if (showDict()) {
                val sb = StringBuilder()
                val defSize = labelData.target[0].data?.definition?.size ?: 0
                labelData.target[0].data?.definition?.forEachIndexed { index, definition ->
                    sb.append(definition.speechPart)
                    sb.append(".")
                    sb.append(definition.source)
                    if (index < defSize - 1) {
                        sb.append("\n")
                    }
                }
                binding.include.explainTv.setText(sb.toString())
                binding.include.explainTv.visibility = View.VISIBLE
            } else {
                binding.include.explainTv.visibility = View.GONE
            }
            binding.include.labelContainer.post {
                binding.include.labelContainer.scrollTo(0, 0)
            }

            if (currentStatus == STATE_TRANSLATING_CAMERA) {
                currentStatus =
                    STATE_RESULT_FROM_CAMERA_SUCCESS
            }
            logEvent(
                LO_camera_click_success,
                hashMapOf(
                    source_target_language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                        translateViewModel.getTargetData(mState).value ?: return
                    }")
                )
            )
        }

        override fun enterCameraState() {
            super.enterCameraState()
            binding.torchSelector.visibility = View.INVISIBLE
            binding.ivSelectImage.visibility = View.INVISIBLE
            mCameraSource.torch(false)
            mPreTorchState = false
            readLanguageLabel()
            binding.include.root.visibility = View.GONE
            translateViewModel.objectSource.observe(viewLifecycleOwner, Observer { language ->
                binding.tvSourceLanguage.text = Locale.forLanguageTag(language).displayLanguage
            })
            translateViewModel.objectTarget.observe(viewLifecycleOwner, Observer { language ->
                binding.tvTargetLanguage.text = Locale.forLanguageTag(language).displayLanguage
                //exchangeState()
            })
        }

        override fun getSupportSourceLanguage(): LocaleListCompat {
            return getLocaleListCompat(targetSupportLanguages)
        }

        override fun getSupportTargetLanguage(): LocaleListCompat {
            return if (OcrDispatcher.getGoogleOcrEnable()) {
                getLocaleListCompat(firebaseSourceSupportLanguages, googleVisionLanguages)
            } else {
                getLocaleListCompat(firebaseSourceSupportLanguages)
            }
        }

        override fun changeLanguageAfter(sourceLocale: CustomLocale, targetLocale: CustomLocale) {
            val context = context ?: return
            translateViewModel.getSourceData(mState).value = sourceLocale.languageTag
            translateViewModel.getTargetData(mState).value = targetLocale.languageTag
            saveLanguage(context, PREFER_KEY_SOURCE_LANG_LABEL, sourceLocale.languageTag)
            saveLanguage(
                context,
                PREFER_KEY_TARGET_LANG_LABEL, targetLocale.languageTag
            )
            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS) {
                    currentStatus =
                        STATE_TRANSLATING_CAMERA
                } else if (currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS) {
                    currentStatus =
                        STATE_TRANSLATING_GALLERY
                }
                mItemRecognizeResult?.let {
                    showProgressBar()
                    enterTranslateState()
                    translateViewModel.imageLabelChangeLanguage(
                        it,
                        sourceLocale.languageTag,
                        targetLocale.languageTag
                    ).observe(this@CameraFragment, Observer {
                        if (it.isSuccess) {
                            mItemRecognizeResult = it.getOrNull()
                            if (mItemRecognizeResult != null && mItemRecognizeResult!!.source.isNotEmpty() && mItemRecognizeResult!!.target.isNotEmpty()) {
                                labelSuccess(mItemRecognizeResult!!)
                            } else {
                                recognitionFailure(it.exceptionOrNull())
                            }

                        } else {
                            recognitionFailure(it.exceptionOrNull())
                        }
                        enterResultState(it.isSuccess)
                    })
                }
            }
        }
    }


    override fun onViewCreated(view: View, savedInstanceState: Bundle?) {
        super.onViewCreated(view, savedInstanceState)
        //translateViewModel = ImageTranslate(requireContext())
        immersionBar {
            hideBar(BarHide.FLAG_HIDE_BAR)
            fullScreen(true)
        }
        mState = photoTranslateState
        binding = FragmentCameraBinding.bind(view)
        mViewPageGestureWrapper = ViewPageGestureWrapper(requireContext())
            .also { it.setViewPager(binding.feature) }
        mCameraSource = CameraCompatSource()
        //translateViewModel = ImageTranslate(requireContext())
        //mainExecutor = ContextCompat.getMainExecutor(requireContext())
        initPreviewView(view)
        initView(view)
        mState.enterCameraState()
        handleIntent()
        mCameraSource.setStatusCallback(this)
        requireActivity().onBackPressedDispatcher.addCallback(
            viewLifecycleOwner,
            object : OnBackPressedCallback(true) {
                override fun handleOnBackPressed() {
                    if (TextSpeech.isSpeaking()) {
                        TextSpeech.stopSpeak()
                    }
                    if (currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
                        || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                        || currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                        || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                    ) {
                        mState.enterCameraState()
                    } else {
                        activity?.finish()
                    }
                }

            })
    }

    override fun cameraInitializeFail(exception: Exception) {
        showErrorAlert(requireContext(), exception)
    }

    override fun onPicturetaked(data: ByteArray, metadata: FrameMetadata) {
        if (!isDetached && activity?.isDestroyed == false) {
            onImageSaved(data, metadata)
        }
    }

    override fun onPicturetakeFail(exception: java.lang.Exception) {
        showErrorAlert(requireActivity(), exception)
    }

    override fun onStatusChange(state: Int) {
        when (state) {
            ICameraSource.TORCH_OFF -> {
                binding.torchSelector.isChecked = false
            }
            ICameraSource.TORCH_ON -> {
                binding.torchSelector.isChecked = true
            }
        }
    }

    /**
     * Handle Intent
     */
    private fun handleIntent() {
        val intent = requireActivity().intent
        when (intent?.action) {
            Intent.ACTION_SEND -> {
                val uri = intent.getParcelableExtra<Uri?>(Intent.EXTRA_STREAM)

                if (uri != null) {
                    setImageFromGallery(uri)
                }
            }
        }
    }

    override fun onPause() {
        super.onPause()
        mCameraSource.torch(false)
    }

    internal fun enterTranslateState() {
        binding.tvSourceLanguage.isClickable = false
        binding.tvTargetLanguage.isClickable = false
        //mControlView.setState(mControlView.cameraState)
        //mControlView.setState(mControlView.cropState)
        binding.btnControl.startAnim()
        binding.btnControl.isClickable = false
        binding.languageSelector.background?.alpha = 0x66
        binding.languageSelector.isClickable = false
        binding.tvSourceLanguage.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg2
            )
        )
        binding.tvTargetLanguage.setTextColor(
            ContextCompat.getColor(
                requireContext(),
                R.color.camera_tool_bg2
            )
        )
        binding.ivExchange.alpha = 0.5f
        binding.ivExchange.isClickable = false
        binding.overlay.visibility = View.INVISIBLE
        binding.torchSelector.visibility = View.INVISIBLE
        binding.ivSelectImage.visibility = View.INVISIBLE
        binding.include.dict.isClickable = false
        binding.include.sound.isClickable = false
        binding.tabLayout.isEnabled = false
    }

    @SuppressLint("ClickableViewAccessibility")
    private fun initView(view: View) {

        mFeatureAdapter = FeatureAdapter(
            context ?: return,
            arrayListOf(
                Feature.PHOTO_TRANSLATE,
                Feature.ITEM_RECOGNIZE
            )
        )
        binding.feature.adapter = mFeatureAdapter
        binding.feature.addOnPageChangeListener(object : ViewPager.OnPageChangeListener {
            override fun onPageScrolled(
                position: Int,
                positionOffset: Float,
                positionOffsetPixels: Int
            ) {

            }

            override fun onPageSelected(position: Int) {
                when (mFeatureAdapter!!.getItem(position)) {
                    Feature.PHOTO_TRANSLATE -> {
                        mState = photoTranslateState
                    }
                    else -> {
                        mState = itemRecognizeState
                    }
                }
                mState.enterCameraState()
            }

            override fun onPageScrollStateChanged(state: Int) {

            }

        })

        binding.tabLayout.setViewPager(binding.feature)

        binding.btnControl.setOnClickListener(this)

        //binding.ivExchange.setOnClickListener(this)
        binding.languageSelector.setOnClickListener(this)
        binding.ivFinish.setOnClickListener(this)
        binding.tvSourceLanguage.setOnClickListener(this)
        binding.tvTargetLanguage.setOnClickListener(this)
        binding.progress.setOnTouchListener { v, event -> true }
    }

    internal fun readLanguage() {
        val defaultSourceLan = if (Locale.getDefault().language == Locale.ENGLISH.language) {
            LANG.HI
        } else {
            Locale.ENGLISH.language
        }
        val prefer = context?.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE) ?: return
        //val defaultTargetLang = PhotoTranslateController.targetLanguage

        var sourceLang =
            prefer.getString(PREFER_KEY_SOURCE_LANG, defaultSourceLan)
                ?: defaultSourceLan
        if (!ENG_SEQUENCE.contains(sourceLang)) {
            sourceLang = Locale.ENGLISH.language
        }
        val targetLang =
            prefer.getString(PREFER_KEY_TARGET_LANG, Locale.getDefault().language)
                ?: Locale.getDefault().language
        translateViewModel.getSourceData(mState).value = sourceLang
        translateViewModel.getTargetData(mState).value = targetLang
    }

    internal fun readLanguageLabel() {
        val prefer = context?.getSharedPreferences(PREFER_NAME, Context.MODE_PRIVATE) ?: return
        //val defaultTargetLang = PhotoTranslateController.targetLanguage

        var sourceLang =
            prefer.getString(PREFER_KEY_SOURCE_LANG_LABEL, translateViewModel.targetLiveData.value)
                ?: translateViewModel.targetLiveData.value
        if (!ENG_SEQUENCE.contains(sourceLang)) {
            sourceLang = Locale.ENGLISH.language
        }
        val targetLang =
            prefer.getString(PREFER_KEY_TARGET_LANG_LABEL, translateViewModel.sourceLiveData.value)
                ?: translateViewModel.sourceLiveData.value
        translateViewModel.getSourceData(mState).value = sourceLang
        translateViewModel.getTargetData(mState).value = targetLang
    }

    /***************************LANGUAGE START*********************************/

    /**
     * Picker for Language
     */
    @SuppressLint("InflateParams")
    private fun languagePicker(context: Context, anchorView: View) {
        val bottomSheetDialog = BottomSheetDialog(requireActivity())

        val contentView =
            LayoutInflater.from(context).inflate(R.layout.layout_language_picker, null)

        //val bmp: Bitmap = BitmapFactory.decodeResource(resources, R.drawable.picker_bg)

        bottomSheetDialog.setContentView(contentView)
        bottomSheetDialog.window?.findViewById<View>(R.id.design_bottom_sheet)
            ?.setBackgroundResource(android.R.color.transparent)

        val sourcePicker = contentView.findViewById<WheelView<CustomLocale>>(R.id.np_source_picker)
        val targetPicker = contentView.findViewById<WheelView<CustomLocale>>(R.id.np_target_picker)
        val submit = contentView.findViewById<View>(R.id.submit_language)

        /* ***********Source Picker*************/

        val supportSourceLanguages = mState.getSupportSourceLanguage()

        val supportSourceLanguageSize = supportSourceLanguages.size()
        val sourceValues = Array(supportSourceLanguageSize) {
            CustomLocale(
                supportSourceLanguages[it].language,
                supportSourceLanguages[it].displayLanguage
            )
        }

        val sourceLanguageTag = translateViewModel.getSourceData(mState).value
        val defaultSourceIndex = if (sourceLanguageTag != null) {
            var index = 0
            repeat(sourceValues.size) {
                if (sourceValues[it].languageTag == sourceLanguageTag) {
                    index = it
                    return@repeat
                }
            }
            index
        } else 0

        sourcePicker.setData(sourceValues.toList())
        sourcePicker.setSelectedItemPosition(defaultSourceIndex)

        /* ***********Target Picker*************/

        val supportTargetLanguages = mState.getSupportTargetLanguage()
        val supportTargetLanguageSize = supportTargetLanguages.size()
        val targetValues = Array(supportTargetLanguageSize) {
            CustomLocale(
                supportTargetLanguages[it].language,
                supportTargetLanguages[it].displayLanguage
            )
        }
        //println("supportTargetLanguageSize=$supportTargetLanguageSize")
        val targetLanguageTag = translateViewModel.getTargetData(mState).value
        val defaultTargetIndex = if (targetLanguageTag != null) {
            var index = 0
            for (i in 0..supportTargetLanguageSize) {
                val locale = supportTargetLanguages.get(i) ?: continue
                val tempTag = locale.language
                if (tempTag == targetLanguageTag) {
                    index = i
                    break
                }
            }

            index

        } else 0

        targetPicker.setData(targetValues.toList())
        targetPicker.setSelectedItemPosition(defaultTargetIndex)

        var sourceLocale = sourceValues[sourcePicker.getSelectedItemPosition()]
        val sourceSelectedListener = object : WheelView.OnItemSelectedListener<CustomLocale> {
            override fun onItemSelected(
                wheelView: WheelView<CustomLocale>,
                data: CustomLocale,
                position: Int
            ) {
                sourceLocale = wheelView.getItemData(position)
            }

        }

        var targetLocale = targetValues[targetPicker.getSelectedItemPosition()]
        val targetSelectedListener = object : WheelView.OnItemSelectedListener<CustomLocale> {
            override fun onItemSelected(
                wheelView: WheelView<CustomLocale>,
                data: CustomLocale,
                position: Int
            ) {
                targetLocale = wheelView.getItemData(position)
            }

        }

        sourcePicker.setOnItemSelectedListener(sourceSelectedListener)
        targetPicker.setOnItemSelectedListener(targetSelectedListener)

        submit.setOnClickListener {
            //readLanguage(submit)

            if (bottomSheetDialog.isShowing) {
                bottomSheetDialog.dismiss()
            }

            if (currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                || currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
            ) {
                if (!context.isNetworkConnected()) {
                    Snackbar.make(anchorView, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return@setOnClickListener
                }

            }

            mState.changeLanguageAfter(sourceLocale, targetLocale)

            when (mState.getType()) {
                TYPE_TEXT -> {
                    logEvent(
                        PT_lan_click_success,
                        source_target_language,
                        "${translateViewModel.getSourceData(mState).value ?: return@setOnClickListener}&${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return@setOnClickListener
                        }"
                    )
                }
                TYPE_OBJECT -> {
                    logEvent(
                        LO_lan_click_success,
                        source_target_language,
                        "${translateViewModel.getSourceData(mState).value ?: return@setOnClickListener}&${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return@setOnClickListener
                        }"
                    )
                }
            }

        }

        bottomSheetDialog.setOnShowListener {
            binding.languageSelector.visibility = View.INVISIBLE
        }
        bottomSheetDialog.setOnDismissListener {
            activity ?: return@setOnDismissListener
            binding.languageSelector.visibility = View.VISIBLE
            ImmersionBar.with(this@CameraFragment)
                .fullScreen(true)
                .init()
        }

        targetPicker.setVibratorEnable(true)
        sourcePicker.setVibratorEnable(true)

        bottomSheetDialog.show()
    }

    /**
     * Exchange Language
     */
    private fun exchangeLanguage(context: Context) {
        val sourceLanguageTag = translateViewModel.getTargetData(mState).value ?: return
        val targetLanguageTag = translateViewModel.getSourceData(mState).value ?: return
        saveLanguage(context, PREFER_KEY_SOURCE_LANG, sourceLanguageTag)
        saveLanguage(
            context,
            PREFER_KEY_TARGET_LANG, targetLanguageTag
        )
        translateViewModel.getSourceData(mState).value = sourceLanguageTag
        translateViewModel.getTargetData(mState).value = targetLanguageTag
    }

    /***************************LANGUAGE END*********************************/

    /***************************CAMERA START*********************************/

    private fun initPreviewView(view: View) {
        mCameraSource.initialize(requireContext(), binding.previewContainer, this)
        mCameraSource.setViewTouchListener(this)
        //mPreviewView = view.findViewById(R.id.preview)
    }

    /**
     * Take Photo
     */
    private fun takePicture(context: Context) {
        mCameraSource.takePicture()
    }

    internal var mMetadata: FrameMetadata? = null
    internal var mPreviewBitmap: Bitmap? = null

    private fun doOcrTranslate(img: Any) {

        Glide.with(requireContext())
            .asBitmap()
            .load(img)
            .listener(object : RequestListener<Bitmap> {
                override fun onResourceReady(
                    resource: Bitmap,
                    model: Any?,
                    target: Target<Bitmap>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    //mCameraSource.pause()
                    mPreviewBitmap = resource
                    binding.ivPreview.visibility = View.VISIBLE
                    val sourceLanguageTag = translateViewModel.getSourceData(mState).value ?: "en"
                    val targetLanguageTag = translateViewModel.getTargetData(mState).value ?: "en"
                    //val sourceLanguage = localeToGoogleLanguage[sourceLanguageTag] ?: "en"
                    //val targetLanguage = localeToGoogleLanguage[targetLanguageTag] ?: "en"
                    showProgressBar()
                    //enterTranslateState()
                    if (mMetadata == null) {
                        mMetadata = FrameMetadata.Builder()
                            .setWidth(resource.width)
                            .setHeight(resource.height)
                            .setCameraFacing(Facing.BACK)
                            .setPictureFormat(PictureFormat.JPEG)
                            .setRotation(0)
                            .build()
                    }

                    mState.onPictureTaken(resource, sourceLanguageTag, targetLanguageTag)
                    return false
                }

                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Bitmap>?,
                    isFirstResource: Boolean
                ): Boolean {
                    binding.failToast.visibility = View.VISIBLE
                    binding.failToastText.text = getString(R.string.load_image_fail)
                    binding.failToast.postDelayed(
                        { binding.failToast.visibility = View.GONE },
                        2000
                    )
                    mState.enterCameraState()
                    return false
                }

            })
            .into(binding.ivPreview)
    }

    internal var mPreTorchState: Boolean = false

    /**
     * Take Photo Success
     */
    private fun onImageSaved(data: ByteArray, metadata: FrameMetadata) {
        mMetadata = metadata
        doOcrTranslate(data)
        mPreTorchState = binding.torchSelector.isChecked
        mCameraSource.torch(switch = false)
    }

    private fun showErrorAlert(context: Context, exc: java.lang.Exception) {
        val onClickListener = DialogInterface.OnClickListener { dialog, which ->
            activity?.finish()
        }

        AlertDialog.Builder(context)
            .setTitle(android.R.string.dialog_alert_title)
            .setMessage(R.string.camera_unavailable)
            .setPositiveButton(android.R.string.ok, onClickListener)
            .setCancelable(false)
            .create()
            .show()
    }

    /***************************CAMERA END*********************************/

    /**
     * 翻译图片
     */
    internal fun translateImage(
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>>? {
        return translateViewModel.translate(
            bitmap,
            mMetadata ?: return null,
            sourceLanguage,
            targetLanguage
        )
    }

    internal fun recognitionFailure(throwable: Throwable? = null) {
        if (currentStatus == STATE_TRANSLATING_CAMERA) {
            currentStatus =
                STATE_RESULT_FROM_CAMERA_FAIL
        } else if (currentStatus == STATE_TRANSLATING_GALLERY) {
            currentStatus =
                STATE_RESULT_FROM_GALLERY_FAIL
        }
        showTranslateError(throwable)
        when (mState.getType()) {
            TYPE_TEXT -> {
                translationFailEvent(
                    PT__camera_click_fail,
                    currentStatus,
                    throwable
                )

            }
            TYPE_OBJECT -> {
                translationFailEvent(LO__camera_click_fail, currentStatus, throwable)
            }
        }

        logEvent(
            Trans_translate_failure,
            hashMapOf(
                language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                    translateViewModel.getTargetData(
                        mState
                    ).value ?: return
                }"),
                moduleType to modulePic,
                errorMessage to (throwable?.message ?: "unknown error")
            )
        )

    }

    /**
     * 识别翻译成功
     */
    internal fun recognitionSuccess(newBitmap: Bitmap?) {
        //if (isDetached) return
        when (currentStatus) {
            STATE_TRANSLATING_GALLERY -> {
                currentStatus =
                    STATE_RESULT_FROM_GALLERY_SUCCESS
                logEvent(PT__gallery_click_success)
                logEvent(
                    Trans_translate_success,
                    hashMapOf(
                        language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return
                        }"),
                        moduleType to modulePic
                    )
                )
            }
            STATE_TRANSLATING_CAMERA -> {
                currentStatus =
                    STATE_RESULT_FROM_CAMERA_SUCCESS
                logEvent(PT__camera_click_success)
                logEvent(
                    Trans_translate_success,
                    hashMapOf(
                        language to ("${translateViewModel.getSourceData(mState).value ?: return}-${
                            translateViewModel.getTargetData(
                                mState
                            ).value ?: return
                        }"),
                        moduleType to modulePic
                    )
                )
            }
            STATE_RESULT_FROM_GALLERY_FAIL -> {
                currentStatus =
                    STATE_RESULT_FROM_GALLERY_SUCCESS
            }
            STATE_RESULT_FROM_CAMERA_FAIL -> {
                currentStatus =
                    STATE_RESULT_FROM_CAMERA_SUCCESS
            }
        }

        val matrix = binding.overlay.imageMatrix
        Glide.with(context ?: return)
            .load(newBitmap)
            .listener(object : RequestListener<Drawable> {
                override fun onResourceReady(
                    resource: Drawable?,
                    model: Any?,
                    target: Target<Drawable>?,
                    dataSource: DataSource?,
                    isFirstResource: Boolean
                ): Boolean {
                    binding.overlay.visibility = View.VISIBLE
                    binding.overlay.setImageDrawable(resource)
                    binding.overlay.setDisplayMatrix(matrix)
                    return true
                }

                override fun onLoadFailed(
                    e: GlideException?,
                    model: Any?,
                    target: Target<Drawable>?,
                    isFirstResource: Boolean
                ): Boolean {
                    return false
                }

            })
            .into(binding.overlay)
        //mPreviewResult?.setImageBitmap(newBitmap)

    }


    private val combineViewModel by viewModels<CombineViewModel> {
        object : ViewModelProvider.Factory {
            override fun <T : ViewModel?> create(modelClass: Class<T>): T {
                return CombineViewModel() as T
            }

        }
    }

    private fun saveToGallery(combine: Boolean) {
        logEvent(PT_save_to_gallery)
        val overlay = mCompleteResult?.renderResult
        val preview = mPreviewBitmap
        if (combine) {
            if (overlay?.isRecycled?.not() == true && preview?.isRecycled?.not() == true) {
                combineViewModel.combine(preview, overlay).switchMap {
                    flow {
                        emit(
                            Utils.saveBitmap(
                                requireContext(),
                                it,
                                "HD-${System.currentTimeMillis()}.jpeg"
                            )
                        )
                    }.asLiveData(Dispatchers.IO)
                }.observe(this, Observer {
                    //binding.ivSelectImage.visibility = View.INVISIBLE
                    Toast.makeText(
                        context ?: return@Observer,
                        getString(R.string.save_toast_string),
                        Toast.LENGTH_SHORT
                    ).show()
                })
            }
        } else {
            if (preview?.isRecycled?.not() == true) {
                flow {
                    emit(
                        Utils.saveBitmap(
                            requireContext(),
                            preview,
                            "HD-${System.currentTimeMillis()}.jpeg"
                        )
                    )
                }.flowOn(Dispatchers.IO)
                    .asLiveData().observe(this, Observer {
                        Toast.makeText(
                            context ?: return@Observer,
                            getString(R.string.save_toast_string),
                            Toast.LENGTH_SHORT
                        ).show()
                    })
            }
        }
    }


    /**
     * Show Progress
     */
    internal fun showProgressBar() {
        binding.progress.visibility = View.VISIBLE

    }

    /**
     * Hide Progress
     */
    internal fun hideProgressBar() {
        binding.progress.visibility = View.GONE
    }

    private fun showTranslateError(throwable: Throwable? = null) {
        binding.failToast.visibility = View.VISIBLE
        binding.failToastText.text = throwable?.message ?: ""
        binding.failToast.postDelayed({ binding.failToast.visibility = View.GONE }, 2000)
    }

    /***************************CROPPER WITH TRANSLATE END*********************************/

    /***************************SELECT START*********************************/
    /**
     * 设置图片
     */
    private fun setImageFromUri(uri: Uri) {
        currentStatus =
            STATE_TRANSLATING_GALLERY
        enterTranslateState()
        doOcrTranslate(uri)
    }

    fun setImageFromShot(uri: Uri) {
        setImageFromUri(uri)
        logEvent(PT_screen_shot_iden_click)
    }

    fun setImageFromGallery(uri: Uri) {
        setImageFromUri(uri)
        logEvent(PT_gallery_iden_click)
    }

    internal var screenShotMode = false

    fun setScreenShotMode(mode: Boolean) {
        screenShotMode = mode
    }

    /***************************SELECT END*********************************/

    override fun onClick(v: View) {
        if (TextSpeech.isSpeaking()) {
            TextSpeech.stopSpeak()
        }
        when (v.id) {
            R.id.btn_control -> {
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                if (!v.context.isNetworkConnected()) {
                    Snackbar.make(v, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return
                }

                when (currentStatus) {
                    STATE_CAMERA -> {
                        currentStatus =
                            STATE_TRANSLATING_CAMERA
                        enterTranslateState()
                        takePicture(v.context)
                        when (mState.getType()) {
                            TYPE_TEXT -> {
                                logEvent(PT_camera_click)
                            }
                            TYPE_OBJECT -> {
                                logEvent(LO_camera_click)
                            }
                        }
                    }
                    STATE_RESULT_FROM_CAMERA_FAIL,
                    STATE_RESULT_FROM_CAMERA_SUCCESS,
                    STATE_RESULT_FROM_GALLERY_SUCCESS,
                    STATE_RESULT_FROM_GALLERY_FAIL -> {
                        mState.enterCameraState()
                        when (mState.getType()) {
                            TYPE_TEXT -> {
                                logEvent(PT_exit)
                                logEvent(PT_camera_refresh)
                            }
                            TYPE_OBJECT -> {
                                logEvent(LO_camera_refresh)
                            }
                        }
                    }
                }
            }
            R.id.iv_select_image -> {
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                if (!v.context.isNetworkConnected()) {
                    Snackbar.make(v, R.string.network_unavailable, Snackbar.LENGTH_SHORT)
                        .show()
                    return
                }

                try {
                    val intent = Intent(Intent.ACTION_PICK)
                    intent.type = "image/*"
                    startActivityForResult(
                        intent,
                        REQUEST_CODE_SELECT_IMAGE
                    )
                    logEvent(PT_gallery_click)
                } catch (e: ActivityNotFoundException) {
                    binding.failToast.visibility = View.VISIBLE
                    binding.failToastText.text = context?.getString(R.string.gallery_not_found)
                    binding.failToast.postDelayed(
                        { binding.failToast.visibility = View.GONE },
                        2000
                    )
                }
                mCameraSource.torch(false)
                mPreTorchState = false
            }
            R.id.torch_selector -> {
                val isChecked = (v as? Checkable)?.isChecked
                mCameraSource.torch(isChecked == true)
                mPreTorchState = isChecked == true
                //mCamera?.cameraControl?.enableTorch(isChecked == true)
                logEvent(PT_flash_click)
            }
            R.id.iv_finish -> {
                //requireActivity().finish()
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                if (currentStatus == STATE_RESULT_FROM_GALLERY_FAIL
                    || currentStatus == STATE_RESULT_FROM_GALLERY_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_CAMERA_SUCCESS
                    || currentStatus == STATE_RESULT_FROM_CAMERA_FAIL
                ) {
                    mState.enterCameraState()
                } else {
                    activity?.finish()
                }
            }
            R.id.tv_source_language, R.id.tv_target_language, R.id.language_selector -> {
                /*startActivityForSelectLanguage(
                    v.context,
                    REQUEST_CODE_SELECT_SOURCE_LANGUAGE
                )*/
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                languagePicker(v.context, v)
                //pickerClickedEvent(state = currentStatus)
            }
            /*R.id.iv_exchange -> {
                if (AntiShakeUtils.isInvalidClick(v)) {
                    return
                }
                exchangeLanguage(v.context)
                logEvent(PT_lan_click_switch)
            }*/

        }
    }

    /*private fun exchangeState() {
        val targetLanguageTag = targetLiveData.value ?: return
        val sourceLanguageTag = sourceLiveData.value ?: return
        binding.ivExchange.isEnabled = firebaseSourceSupportLanguages.contains(targetLanguageTag)
        binding.ivExchange.alpha = if (binding.ivExchange.isEnabled) {
            1.0f
        } else {
            0.5f
        }
    }*/

    override fun onActivityResult(requestCode: Int, resultCode: Int, data: Intent?) {
        super.onActivityResult(requestCode, resultCode, data)

        when (requestCode) {
            REQUEST_CODE_SELECT_IMAGE -> {
                val uri = data?.data
                //println("uri===$uri")

                if (uri != null) {
                    setImageFromGallery(uri)
                }

            }
            REQUEST_CODE_SELECT_SOURCE_LANGUAGE -> {
                val languageTag = data?.getStringExtra("languageTag") ?: return
                saveLanguage(context ?: return, PREFER_KEY_SOURCE_LANG, languageTag)
            }
            REQUEST_CODE_SELECT_TARGET_LANGUAGE -> {
                val languageTag = data?.getStringExtra("languageTag") ?: return
                saveLanguage(
                    context ?: return,
                    PREFER_KEY_TARGET_LANG, languageTag
                )
            }
        }
    }

    override fun onTouchEvent(event: MotionEvent) {
        mViewPageGestureWrapper?.onTouchEvent(event)
    }

    override fun onInterceptTouchEvent(event: MotionEvent) {
        mViewPageGestureWrapper?.onInterceptTouchEvent(event)
    }
}