package com.talpa.translate.ocr

import android.app.Application
import android.graphics.Bitmap
import android.graphics.Rect
import android.text.TextUtils
import android.util.Log
import androidx.lifecycle.AndroidViewModel
import androidx.lifecycle.LiveData
import androidx.lifecycle.MutableLiveData
import androidx.lifecycle.asLiveData
import com.photo.translation.R
import com.talpa.translate.base.state.PhotoMLState
import com.talpa.translate.factory.TranslatorFactory
import com.talpa.translate.imglabel.ImageLabelData
import com.talpa.translate.imglabel.ImageLabelWorker
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.Utils.rotateBitmap
import com.talpa.translate.ocr.datasource.CompleteResult
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.ocr.result.OcrResult
import com.talpa.translate.render.Render
import com.talpa.translate.repository.box.collins.SenseNew
import com.talpa.translate.repository.net.dictionary.DictionaryRepo
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.flow.*
import kotlinx.coroutines.withContext
import java.util.*
import kotlin.math.min


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */
class ImageTranslate(application: Application) : AndroidViewModel(application) {

    companion object {
        const val TAG = "ocr_translate"
    }

    val sourceLiveData: MutableLiveData<String> = MutableLiveData("en")
    val targetLiveData: MutableLiveData<String> = MutableLiveData("en")
    val objectSource: MutableLiveData<String> = MutableLiveData("en")
    val objectTarget: MutableLiveData<String> = MutableLiveData("en")

    fun getSourceData(state: PhotoMLState): MutableLiveData<String> {
        if (state.getType() == PhotoMLState.TYPE_TEXT) {
            return sourceLiveData
        } else {
            return objectSource
        }
    }

    fun getTargetData(state: PhotoMLState): MutableLiveData<String> {
        if (state.getType() == PhotoMLState.TYPE_TEXT) {
            return targetLiveData
        } else {
            return objectTarget
        }
    }

    private lateinit var mTranslator: Recognizer
    private val mRender: Render = TranslatorFactory.getRender(getApplication())

    fun imageLabelChangeLanguage(
        imageLabel: ImageLabelData,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<ImageLabelData>> {
        val dictRepo = DictionaryRepo()
        return flow {
            emit(
                Result.success(
                    withContext(Dispatchers.IO) {
                        val count = min(imageLabel.source.size, imageLabel.target.size)
                        val sourceSenses = arrayListOf<SenseNew>()
                        val targetSenses = arrayListOf<SenseNew>()
                        repeat(count) { index ->
                            val sourceList = async {
                                dictRepo.search(
                                    imageLabel.source[index].data?.word
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    imageLabel.source[index].targetLanTag
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    sourceLanguage,
                                    getApplication()
                                ).also {
                                    it.sourceLanTag = imageLabel.source[index].targetLanTag
                                    it.targetLanTag = sourceLanguage
                                }
                            }
                            val targetList = async {
                                dictRepo.search(
                                    imageLabel.target[index].data?.word
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    imageLabel.target[index].targetLanTag
                                        ?: throw IllegalArgumentException(
                                            getApplication<Application>().getString(
                                                R.string.translate_fail
                                            )
                                        ),
                                    targetLanguage,
                                    getApplication()
                                ).also {
                                    it.sourceLanTag = imageLabel.target[index].targetLanTag
                                    it.targetLanTag = targetLanguage
                                }
                            }
                            sourceSenses.add(sourceList.await())
                            targetSenses.add(targetList.await())
                        }
                        ImageLabelData(imageLabel.sourceBitmap, sourceSenses, targetSenses)
                    }
                )

            )

        }.catch {
            Log.d(TAG, "label imageLabelChangeLanguage error", it)
            emit(Result.failure(it))
        }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    /**
     * 物体识别
     */
    fun itemRecognize(
        bitmap: Bitmap,
        rotation: Int,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<ImageLabelData>> {
        val dictRepo = DictionaryRepo()
        val sourceBitmap = rotateBitmap(rotation, bitmap)
        return flow {
            val itemRecognizer =
                ImageLabelWorker(
                    getApplication<Application>(),
                    sourceBitmap,
                    sourceLanguage,
                    targetLanguage
                )
            val labels = itemRecognizer.doItemRecognize()
            emit(labels)
        }.flatMapConcat { labels ->

            flow {

                emit(
                    Result.success(
                        withContext(Dispatchers.IO) {
                            val sourceSenses = arrayListOf<SenseNew>()
                            val targetSenses = arrayListOf<SenseNew>()
                            labels.forEach {

                                val sourceList = async {
                                    dictRepo.search(
                                        it.text,
                                        Locale.ENGLISH.language,
                                        sourceLanguage,
                                        getApplication()
                                    ).also {
                                        it.sourceLanTag = Locale.ENGLISH.language
                                        it.targetLanTag = sourceLanguage
                                    }
                                }
                                val targetList = async {
                                    dictRepo.search(
                                        it.text,
                                        Locale.ENGLISH.language,
                                        targetLanguage,
                                        getApplication()
                                    ).also {
                                        it.sourceLanTag = Locale.ENGLISH.language
                                        it.targetLanTag = targetLanguage
                                    }
                                }
                                sourceSenses.add(sourceList.await())
                                targetSenses.add(targetList.await())

                            }
                            ImageLabelData(bitmap, sourceSenses, targetSenses)
                        }
                    )

                )
            }
        }
            .catch {
                Log.d(TAG, "label itemRecognize error", it)
                emit(Result.failure(it))
            }
            .flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun doOcrOnly(bitmap: Bitmap, rotation: Int): LiveData<Result<OcrResult>> {
        val sourceBitmap = rotateBitmap(rotation, bitmap)
        return flow {
            val ocrResult = mTranslator.doOcr(sourceBitmap)
            emit(Result.success(ocrResult))
        }.onStart {
            mTranslator = OcrDispatcher.dispatchOcrTranslator(
                getApplication<Application>(),
                Locale.ENGLISH.language
            )
            //mRender.initialize(bitmap)
        }.catch {
            //mRender.release()
            Log.d(TAG, "text doOcrOnly error", it)
            emit(Result.failure(it))
        }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    /**
     * 翻译
     */
    fun translate(
        bitmap: Bitmap,
        rotation: Int,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult(sourceLanguage, targetLanguage)
        val sourceBitmap = rotateBitmap(rotation, bitmap)
        return flow {
            val ocrResult = mTranslator.doOcr(sourceBitmap) //Step1:执行OCR
            var flag = true
            kotlin.run {
                ocrResult.blocks.forEach {
                    if (!TextUtils.isEmpty(it.text)) {
                        flag = false
                        return@run
                    }
                }
            }
            if (flag) throw NoContentException(getApplication<Application>().getString(R.string.no_content_identified))
            completeResult.ocrResult = ocrResult
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val texts = arrayListOf<String>()
                completeResult.ocrResult!!.blocks.forEach { texts.add(it.text) }
                val transResult = HiTranslator.getInstance(getApplication())
                    .postTranslate(
                        from = sourceLanguage,
                        to = targetLanguage,
                        texts = texts
                    )  //Step2:OCR结果提交翻译

                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(flow {
                val photoAnalyzer = PhotoAnalyzer.createAnalyzer(sourceBitmap)
                val ocrResult = completeResult.ocrResult!!
                val rects = arrayListOf<Rect>()
                ocrResult.blocks.forEach {
                    rects.add(it.rect)
                }
                val renderSource = photoAnalyzer.generateBgBitmap(rects = rects.toTypedArray())
                photoAnalyzer.recycle()
                emit(renderSource)
            }) { completeResult, renderSource ->
                //val list = completeResult.transResponse?.result?.texts!!
                val textsResponse = completeResult.transResponse?.result?.texts
                    ?: throw NoContentException(getApplication<Application>().getString(R.string.translate_fail))

                if (textsResponse.isEmpty()) {
                    throw NoContentException(
                        getApplication<Application>().getString(
                            R.string.no_trans_for_ocr
                        )
                    )
                }
                mRender.renderOverlay(renderSource, completeResult)
                val finalResult = rotateBitmap(-rotation, renderSource)
                completeResult.renderResult = finalResult
                Result.success(completeResult)
            }
        }
            .onStart {
                mTranslator = OcrDispatcher.dispatchOcrTranslator(
                    getApplication(),
                    sourceLanguage
                )
            }.catch {
                //mRender.release()
                Log.d(TAG, "text translate error", it)
                emit(Result.failure(it))
            }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun changeTargetLanguage(
        bitmap: Bitmap,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<CompleteResult>> {
        val completeResult = CompleteResult(sourceLanguage, targetLanguage)
        return flow {
            val history = mTranslator.getHistory() ?: throw NoContentException(
                getApplication<Application>().getString(R.string.no_trans_for_ocr)
            )
            completeResult.ocrResult = history
            emit(completeResult)
        }.flatMapConcat { completeResult ->
            flow {
                val history = completeResult.ocrResult
                val texts = arrayListOf<String>()
                history?.blocks?.forEach { block ->
                    texts.add(block.text)
                }
                val transResult = HiTranslator.getInstance(getApplication())
                    .postTranslate(from = sourceLanguage, to = targetLanguage, texts = texts)

                completeResult.transResponse = transResult
                emit(completeResult)
            }.zip(
                flow {
                    val photoAnalyzer = PhotoAnalyzer.createAnalyzer(bitmap)
                    val ocrResult = completeResult.ocrResult!!
                    val rects = arrayListOf<Rect>()
                    ocrResult.blocks.forEach {
                        rects.add(it.rect)
                    }
                    val renderSource = photoAnalyzer.generateBgBitmap(rects = rects.toTypedArray())
                    photoAnalyzer.recycle()
                    emit(renderSource)
                }
            ) { completeResult, renderSource ->

                val textsResponse = completeResult.transResponse?.result?.texts
                    ?: throw NoContentException(getApplication<Application>().getString(R.string.translate_fail))
                if (textsResponse.isEmpty()) {
                    throw NoContentException(
                        getApplication<Application>().getString(
                            R.string.no_trans_for_ocr
                        )
                    )
                }

                mRender.renderOverlay(renderSource, completeResult)
                completeResult.renderResult = renderSource
                return@zip Result.success(completeResult)
            }
        }
            .catch {
                Log.d(TAG, "text changeTargetLanguage error", it)
                emit(Result.failure(it))
            }.flowOn(Dispatchers.IO)
            .asLiveData()
    }
}