package com.talpa.translate.ocr

import android.graphics.Bitmap
import android.graphics.Canvas
import android.graphics.Rect
import android.util.Log
import androidx.annotation.Keep
import kotlinx.coroutines.Deferred
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.async
import kotlinx.coroutines.runBlocking

/**
 * Create by chenjunsheng on 2020/7/28
 */
@Keep
class PhotoAnalyzer private constructor(private val mNativePtr: Long) {

    companion object {
        init {
            System.loadLibrary("imgprocessor")
            System.loadLibrary("opencv_java4")
        }

        fun getBase64String(bitmap: Bitmap): String? {
            return getBase64Native(bitmap)
        }

        fun compressBitmap(bitmap: Bitmap): Bitmap {
            return nativeCompressBitmap(bitmap)
        }

        @JvmStatic
        private external fun nativeCompressBitmap(bitmap: Bitmap): Bitmap

        @JvmStatic
        private external fun getBase64Native(bitmap: Bitmap): String?

        @JvmStatic
        private external fun nativeCreate(bitmap: Bitmap): PhotoAnalyzer

        @JvmStatic
        private external fun nativeRecycle(ptr: Long)

        @JvmStatic
        private external fun nativeGenBitmap(ptr: Long, area: Rect): Bitmap?

        fun createAnalyzer(source: Bitmap): PhotoAnalyzer {
            val analyzer =  nativeCreate(bitmap = source)
            analyzer.mWidth = source.width
            analyzer.mHeight = source.height
            return analyzer
        }
    }

    private var mHeight: Int = 0
    private var mWidth: Int = 0

    private var mRecycled = false

    fun recycle() {
        if (!mRecycled) {
            nativeRecycle(mNativePtr)
            mRecycled = true
        }
    }

    data class AnalyzerStub(var bitmap: Bitmap?, val rect: Rect)

    fun generateBgBitmap(rects: Array<Rect>): Bitmap? {
        if (mRecycled) {
            throw IllegalStateException("analyzer has been recycled")
        }
        val bitmapDeffer = arrayListOf<Deferred<AnalyzerStub?>>()

        return runBlocking {
            rects.forEach { rect ->
                bitmapDeffer.add(async(Dispatchers.Default) {
                    AnalyzerStub(nativeGenBitmap(mNativePtr, rect), rect)

                })
            }
            var bitmap: Bitmap = Bitmap.createBitmap(mWidth, mHeight, Bitmap.Config.ARGB_8888)
            val canvas = Canvas(bitmap)
            bitmapDeffer.forEach {
                val analyzerStub = it.await() ?: return@forEach
                val resultBlock = analyzerStub.bitmap ?: return@forEach
                val area = analyzerStub.rect
                canvas.drawBitmap(resultBlock, null, area, null)
                analyzerStub.bitmap = null
            }
            bitmap
        }
        //return generateBitmapNative(rects)
    }

}