package com.talpa.camera.translate.ocr

import android.content.Context
import android.graphics.*
import android.text.TextPaint
import android.util.Log
import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.appcompat.widget.AppCompatTextView
import androidx.lifecycle.LiveData
import androidx.lifecycle.liveData
import com.google.firebase.ml.vision.FirebaseVision
import com.google.firebase.ml.vision.common.FirebaseVisionImage
import com.google.firebase.ml.vision.common.FirebaseVisionImageMetadata
import com.google.firebase.ml.vision.text.FirebaseVisionText
import com.photo.translation.R
import com.talpa.base.BaseApp
import com.talpa.base.common.FrameMetadata
import com.talpa.base.utils.BitmapUtils
import com.talpa.camera.cameraview.controls.Facing
import com.talpa.camera.translate.trans.HiTranslator
import com.talpa.camera.translate.trans.Translator
import com.talpa.camera.translate.trans.TranslatorFactory
import com.talpa.tengine.Trans
import com.talpa.tengine.lang.LANG
import com.talpa.tengine.lang.languageTagToLangMap
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.withContext
import java.util.*
import kotlin.coroutines.resume
import kotlin.coroutines.resumeWithException
import kotlin.coroutines.suspendCoroutine


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */
class ImageTranslate constructor(
    private val context: Context,
    private var sourceBitmap: Bitmap,
    private var sourceLanguage: String,
    private var targetLanguage: String,
    private var metadata: FrameMetadata?
) {

    companion object {
        const val TAG = "ocr_translate"
    }

    private val mTranslator: Translator = TranslatorFactory.getTranslator(context)

    /**
     * 翻译
     */
    fun translate(): LiveData<Result<Bitmap>> {
        val metadata = FirebaseVisionImageMetadata.Builder()
            .setFormat(FirebaseVisionImageMetadata.IMAGE_FORMAT_NV21)
            .setWidth(sourceBitmap.width)
            .setHeight(sourceBitmap.height)
            .setRotation(metadata?.rotation ?: 0)
            .build()
        val visionImage = FirebaseVisionImage.fromBitmap(
            BitmapUtils.rotateBitmap(
                sourceBitmap,
                metadata.rotation,
                Facing.BACK
            )
        )
        return liveData {
            val resultVisionText =
                try {
                    coroutineScope {
                        detectorVisionImage(visionImage)
                    }
                } catch (e: Exception) {
                    emit(Result.failure<Bitmap>(e))
                    return@liveData
                }
            visionText = resultVisionText
            val result = try {
                withContext(Dispatchers.IO) {
                    BitmapUtils.reRotateBitmap(
                        handleVisionTextToBitmap(
                            resultVisionText,
                            visionImage.bitmap
                        ),
                        this@ImageTranslate.metadata?.rotation ?: 0,
                        Facing.BACK
                    )
                }
            } catch (e: Exception) {
                emit(Result.failure<Bitmap>(e))
                return@liveData
            }
            emit(Result.success(result))
        }
    }

    private var visionText: FirebaseVisionText? = null //识别的结果
    //private var visionBitmap: Bitmap? = null //

    private suspend fun detectorVisionImage(visionImage: FirebaseVisionImage) =
        suspendCoroutine<FirebaseVisionText> { continuation ->
        val detector = FirebaseVision.getInstance().onDeviceTextRecognizer
        detector.processImage(visionImage)
            .addOnSuccessListener { visionText->
                if (visionText.textBlocks.isEmpty()) {
                    Log.d(TAG, "success with no content")
                    continuation.resumeWithException(NoContentException())
                } else {
                    Log.d(TAG, "success with text blocks, size:${visionText.textBlocks.size}")
                    continuation.resume(visionText)
                }
            }
            .addOnFailureListener { exception ->
                Log.d(TAG, "detect vision image fail", exception)
                continuation.resumeWithException(exception)
            }
    }

    fun changeLanguage(sourceLanguage: String, targetLanguage: String): LiveData<Result<Bitmap>> {
        return liveData (Dispatchers.IO) {
            if (visionText != null) {
                this@ImageTranslate.sourceLanguage = sourceLanguage
                this@ImageTranslate.targetLanguage = targetLanguage
                val bitmap = try {
                    handleVisionTextToBitmap(visionText!!, sourceBitmap)
                } catch (e: Exception) {
                    emit(Result.failure<Bitmap>(e))
                    return@liveData
                }
                emit(Result.success(bitmap))
            } else {
                emit(Result.failure<Bitmap>(IllegalAccessException("illegal access language change")))
            }
        }
    }

    /**
     * Vision Text To Bitmap
     */
    private fun handleVisionTextToBitmap(
        result: FirebaseVisionText,
        bitmap: Bitmap
    ): Bitmap {
        //println("Thread---2==${Thread.currentThread().name}")
        val newBitmap = Bitmap.createBitmap(bitmap.width, bitmap.height, Bitmap.Config.ARGB_8888)
        newBitmap.setHasAlpha(true)

        val canvas = Canvas(newBitmap)
        canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)

        //canvas.drawBitmap(bitmap, 0f, 0f, null)

        //canvas.drawRect()
        val backgroundPaint = Paint()
        backgroundPaint.color = Color.WHITE

        val textPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)
            .also { it.density = BaseApp.appContext.resources.displayMetrics.density }
        textPaint.color = Color.BLACK
        textPaint.textAlign = Paint.Align.LEFT

        val blocks = result.textBlocks
        var detectedContent = false

        blocks.forEachIndexed { index, textBlock ->
            /*Log.d("cjslog", "block:${block.boundingBox}" )
            block.cornerPoints?.forEach {

                Log.d("cjslog", "point:${it.toString()}")
            }*/
            val bounding = textBlock.boundingBox ?: return@forEachIndexed
            val blockText = textBlock.text
            //Log.d("cjslog", "text block text:${blockText}")
            val blockCornerPoints = textBlock.cornerPoints ?: return@forEachIndexed
            //canvas.drawRect(rect, backgroundPaint)
            val leftTopPoint = blockCornerPoints[0]

            var translation: String? = null
            translation = try {
                val translateResult = translateTextOnline(blockText)
                Log.d(TAG, "translation reuslt: source_lan:${translateResult?.sourceLanguage}" +
                        "\n target_lan:${translateResult?.targetLanguage}" +
                        "\n text:${translateResult?.text}" +
                        "\n translation:${translateResult?.translation}" +
                        "\n value_msg:${translateResult?.valueMessage}")
                translateResult?.translation
            } catch (e: Exception) {
                Log.d(TAG, "translate online error", e)
                e.printStackTrace()
                null
            }

            if (translation != null) {
                detectedContent = true
                /*canvas.drawText(
                    translationResult.translation,
                    leftTopPoint.x.toFloat(),
                    leftTopPoint.y.toFloat(),
                    textPaint
                )*/

                val bg = backgroundColor() //backgroundColor(bitmap, block)
                val bit =
                    createBlockView(context, textBlock, translation, bg) ?: return@forEachIndexed
                canvas.drawBitmap(bit, leftTopPoint.x.toFloat(), leftTopPoint.y.toFloat(), Paint())
            }
        }
        //canvas.save()
        //canvas.restore()
        if (!detectedContent) {
            Log.d(TAG, "dectcted no content")
            throw NoContentException()
        }
        //val blockFrame = block.boundingBox
        return newBitmap
    }

    private fun translateTextOnline(text: String): HiTranslator.Result? {
        /*return HiTranslator.Builder()
            .setSourceLanguage(sourceLanguage)
            .setTargetLanguage(targetLanguage)
            .setText(text)
            .setKey(
                if (BaseApp.serverKey == null) {
                    BaseApp.initBackgourndKey()
                    BaseApp.serverKey!!
                } else {
                    BaseApp.serverKey!!
                }
            )
            .build()
            .execute()*/
        val map = languageTagToLangMap()

        val source = if (sourceLanguage == LANG.AUTO) {
            sourceLanguage
        } else {
            map[sourceLanguage] ?: "en"
        }

        val target = map[targetLanguage] ?: "en"
        Log.d(TAG, "start_trans: source_lan:${source}, target_lan:${target}")
        val trans = Trans(from = source, to = target, text = text)
        val transResult = mTranslator.doTranslate(trans)
        //val transResult = context.translate(trans).blockingSingle().result
        transResult?.let {
            val result = HiTranslator.Result(
                sourceLanguage,
                targetLanguage,
                text,
                it.translation,
                it.code,
                it.errorMessage
            )
            //Log.d("cjslog", "translate result:${result}")
            return result
        }
        Log.d("cjslog", "result null")
        return null
    }


    private fun createBlockView(
        context: Context,
        block: FirebaseVisionText.TextBlock,
        text: String,
        backgroundColor: Int
    ): Bitmap? {

        val rect = block.boundingBox ?: return null
        //val blockText = block.text


        val textView =
            LayoutInflater.from(context).inflate(R.layout.layout_text, null) as AppCompatTextView
        textView.apply {
            layout(rect.left, rect.top, rect.right, rect.bottom)
        }
        //val textView = view.findViewById<AppCompatTextView>(R.id.tv_content)
        textView.width = rect.width()
        textView.height = rect.height()
        textView.layoutParams = ViewGroup.LayoutParams(rect.width(), rect.height())

        textView.measure(0, 0)

        textView.setBackgroundColor(backgroundColor)

        textView.text = text

        val bitmap = Bitmap.createBitmap(rect.width(), rect.height(), Bitmap.Config.ARGB_8888)

        val canvas = Canvas(bitmap)
        textView.draw(canvas)

        return bitmap
    }

    private fun backgroundColor() = Color.WHITE

}