package com.talpa.translate.render

import android.content.Context
import android.graphics.*
import android.text.TextPaint
import android.view.LayoutInflater
import android.view.ViewGroup
import androidx.appcompat.widget.AppCompatTextView
import com.photo.translation.R
import com.talpa.translate.base.BaseApp
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.base.utils.BitmapUtils
import com.talpa.translate.camera.view.controls.Facing
import com.talpa.translate.ocr.Translator
import com.talpa.translate.ocr.result.OcrResult
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext

/**
 * Create by chenjunsheng on 2020/7/14
 */
class TranslatorRender(val context: Context) :
    Render {

    override suspend fun renderOverlay(translator: Translator, transResult: List<String>): Bitmap? {
        return withContext(Dispatchers.Main) {
            BitmapUtils.reRotateBitmap(
                handleVisionTextToBitmap(
                    translator.getHistory() ?: return@withContext null,
                    translator.getFrameMetadata() ?: return@withContext null,
                    transResult
                ),
                translator.getFrameMetadata()?.rotation ?: 0,
                Facing.BACK
            )
        }
    }

    /**
     * Vision Text To Bitmap
     */
    private fun handleVisionTextToBitmap(
        ocrResult: OcrResult,
        metadata: FrameMetadata,
        transResult: List<String?>
    ): Bitmap {

        val newBitmap = Bitmap.createBitmap(metadata.width, metadata.height,
            Bitmap.Config.ARGB_8888)
        newBitmap.setHasAlpha(true)

        val canvas = Canvas(newBitmap)
        canvas.drawColor(Color.TRANSPARENT, PorterDuff.Mode.CLEAR)

        //canvas.drawBitmap(bitmap, 0f, 0f, null)

        //canvas.drawRect()
        val backgroundPaint = Paint()
        backgroundPaint.color = Color.WHITE

        val textPaint = TextPaint(Paint.ANTI_ALIAS_FLAG)
            .also { it.density = BaseApp.appContext.resources.displayMetrics.density }
        textPaint.color = Color.BLACK
        textPaint.textAlign = Paint.Align.LEFT

        ocrResult.blocks.forEachIndexed { index, block ->
            val rect = block.rect
            val bit = createBlockView(context, rect, transResult.getOrNull(index) ?: return@forEachIndexed) ?: return@forEachIndexed
            canvas.drawBitmap(bit, rect.left.toFloat(), rect.top.toFloat(), null)
        }

        //val blockFrame = block.boundingBox
        return newBitmap
    }

    private fun createBlockView(
        context: Context,
        rect: Rect,
        text: String,
        backgroundColor: Int = Color.WHITE
    ): Bitmap? {

        //val rect = block.boundingBox ?: return null
        //val blockText = block.text

        val textView =
            LayoutInflater.from(context).inflate(R.layout.layout_text, null) as AppCompatTextView
        textView.apply {
            layout(rect.left, rect.top, rect.right, rect.bottom)
        }
        //val textView = view.findViewById<AppCompatTextView>(R.id.tv_content)
        textView.width = rect.width()
        textView.height = rect.height()
        textView.layoutParams = ViewGroup.LayoutParams(rect.width(), rect.height())

        textView.measure(0, 0)

        textView.setBackgroundColor(backgroundColor)

        textView.text = text

        val bitmap = Bitmap.createBitmap(rect.width(), rect.height(), Bitmap.Config.ARGB_8888)

        val canvas = Canvas(bitmap)
        textView.draw(canvas)

        return bitmap
    }

}