package com.talpa.translate.ocr

import android.content.Context
import android.graphics.Bitmap
import android.os.Build
import android.util.Log
import androidx.lifecycle.LiveData
import androidx.lifecycle.asLiveData
import com.photo.translation.R
import com.talpa.translate.base.common.FrameMetadata
import com.talpa.translate.factory.TranslatorFactory
import com.talpa.translate.network.HiTranslator
import com.talpa.translate.ocr.datasource.DataSource
import com.talpa.translate.ocr.exception.NoContentException
import com.talpa.translate.render.Render
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.catch
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn
import kotlinx.coroutines.flow.onStart


/**
 * Photo Translation
 *
 * @author CY 2020-02-05
 */

class ImageTranslate constructor(private val context: Context) {

    companion object {
        const val TAG = "ocr_translate"
    }

    private lateinit var mTranslator: Recognizer
    private val mRender : Render

    fun cleanUp() {
        mRender.release()
    }

    init {
        val analyzer: IAnalyzer
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.N) {
            analyzer = PhotoAnalyzer()
        } else {
            analyzer = PhotoAnalyzerCompat()
        }
        mRender = TranslatorFactory.getRender(context, analyzer)
    }

    /**
     * 翻译
     */
    fun translate(
        bitmap: Bitmap,
        metadata: FrameMetadata?,
        sourceLanguage: String,
        targetLanguage: String
    ): LiveData<Result<Bitmap>> {

        return flow {

            val ocrResult = mTranslator.doOcr()

            val texts = arrayListOf<String>()
            ocrResult.blocks.forEach {
                texts.add(it.text)
            }

            val transResult = HiTranslator.getInstance()
                .postTranslate(from = sourceLanguage, to = targetLanguage, tests = texts)

            //Log.d("cjslog", "trasn result:${transResult}")
            val textsResponse = transResult.result?.texts ?: throw NoContentException(
                context.getString(R.string.translate_fail)
            )
            if (textsResponse.isEmpty()) throw NoContentException(
                context.getString(R.string.no_trans_for_ocr)
            )

            val result =
                mRender.renderOverlay(mTranslator, textsResponse) ?: throw NoContentException(
                    context.getString(R.string.generate_bitmap_fail)
                )

            emit(Result.success(result))
        }.onStart {
            mTranslator = OcrDispatcher.dispatchOcrTranslator(context, sourceLanguage)
            mRender.initialize(bitmap)

            val dataSource: DataSource =
                mRender.createDataSource(
                    bitmap,
                    mTranslator.getType(),
                    checkNotNull(metadata),
                    targetLanguage
                )
            mTranslator.setup(dataSource)
        }.catch {
            Log.d("cjslog", "translate error", it)
            emit(Result.failure(it))
        }.flowOn(Dispatchers.IO)
            .asLiveData()
    }

    fun changeLanguage(sourceLanguage: String, targetLanguage: String): LiveData<Result<Bitmap>> {
        return flow {
            mTranslator.getHistory()?.let {
                val texts = arrayListOf<String>()
                it.blocks.forEach { block ->
                    texts.add(block.text)
                }
                val transResult = HiTranslator.getInstance()
                    .postTranslate(from = sourceLanguage, to = targetLanguage, tests = texts)

                //Log.d("cjslog", "trasn result change:${transResult}")

                val textsResponse = transResult.result?.texts ?: throw NoContentException(
                    context.getString(R.string.translate_fail)
                )
                if (textsResponse.isEmpty()) throw NoContentException(
                    context.getString(R.string.no_trans_for_ocr)
                )
                val result =
                    mRender.renderOverlay(mTranslator, textsResponse) ?: throw NoContentException(
                        context.getString(R.string.generate_bitmap_fail)
                    )

                emit(Result.success(result))
            }
        }.catch { emit(Result.failure(it)) }.flowOn(Dispatchers.IO)
            .asLiveData()
    }


}