package com.talpa.rate.ratebar;

import android.animation.Animator;
import android.animation.AnimatorListenerAdapter;
import android.content.Context;
import android.graphics.Color;
import android.graphics.drawable.ClipDrawable;
import android.graphics.drawable.Drawable;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.TextView;

import androidx.annotation.IntRange;
import androidx.annotation.NonNull;

import com.airbnb.lottie.LottieAnimationView;

/**
 * Created by willy on 2017/6/3.
 */

class PartialView
        extends RelativeLayout {

    private ImageView mFilledView;
    private ImageView mEmptyView;
    private LottieAnimationView lottieAnimationView;
    private int mStarWidth = 0;
    private int mStarHeight = 0;
    private TextView mGuidanceText;

    public PartialView(Context context, int partialViewId, int starWidth, int startHeight,
                       int padding) {
        super(context);

        mStarWidth = starWidth;
        mStarHeight = startHeight;

        setTag(partialViewId);
        setPadding(padding, padding, padding, padding);
        init();
    }

    public PartialView(Context context, AttributeSet attrs) {
        super(context, attrs);
        init();
    }

    public PartialView(Context context, AttributeSet attrs, int defStyleAttr) {
        super(context, attrs, defStyleAttr);
        init();
    }

    private void init() {
        // Make PartialViews use the space when the RatingBar has more width (e.g. match_parent)
        setLayoutParams(new LinearLayout.LayoutParams(
                LayoutParams.WRAP_CONTENT,
                LayoutParams.WRAP_CONTENT,
                1f));

        LayoutParams params = new LayoutParams(
                mStarWidth == 0 ? LayoutParams.WRAP_CONTENT : mStarWidth,
                mStarHeight == 0 ? LayoutParams.WRAP_CONTENT : mStarHeight);
        params.addRule(CENTER_IN_PARENT);

        mFilledView = new ImageView(getContext());
        mFilledView.setScaleType(ImageView.ScaleType.CENTER_CROP);
        addView(mFilledView, params);

        mEmptyView = new ImageView(getContext());
        mEmptyView.setScaleType(ImageView.ScaleType.CENTER_CROP);
        addView(mEmptyView, params);

        mFilledView.setVisibility(View.INVISIBLE);
        mEmptyView.setVisibility(View.INVISIBLE);

        lottieAnimationView = new LottieAnimationView(getContext());
        lottieAnimationView.setScaleType(ImageView.ScaleType.CENTER);
        lottieAnimationView.setSpeed(2f);
        addView(lottieAnimationView, params);

        setEmpty();
    }

    public void setLottieAnim(String fileName) {
        lottieAnimationView.setAnimation(fileName);
    }

    public void playAnimation() {
        lottieAnimationView.playAnimation();
    }

    public void cancelAnimation() {
        //lottieAnimationView.setProgress(0);
        //lottieAnimationView.cancelAnimation();
        removeView(lottieAnimationView);
        mFilledView.setVisibility(View.VISIBLE);
        mEmptyView.setVisibility(View.VISIBLE);
    }

    interface EndAnimListener {
        void onEnd();
        void onCurrentEnd(PartialView current);
    }

    public void setAnimListener(PartialView nextView, EndAnimListener endAnimListener) {
        lottieAnimationView.addAnimatorListener(new Animator.AnimatorListener() {
            @Override
            public void onAnimationStart(Animator animation) {

            }

            @Override
            public void onAnimationEnd(Animator animation) {
                if (nextView != null) {
                    nextView.playAnimation();
                    if (endAnimListener != null) {
                        endAnimListener.onCurrentEnd(PartialView.this);
                    }
                } else {
                    if (endAnimListener != null) {
                        endAnimListener.onEnd();
                    }
                }
            }

            @Override
            public void onAnimationCancel(Animator animation) {

            }

            @Override
            public void onAnimationRepeat(Animator animation) {

            }
        });
    }

    public void setGuidanceText(String text) {
        LayoutParams params = new LayoutParams(LayoutParams.WRAP_CONTENT, LayoutParams.WRAP_CONTENT);
        params.addRule(CENTER_IN_PARENT);
        mGuidanceText = new TextView(getContext());
        mGuidanceText.setText(text);
        addView(mGuidanceText, params);
    }

    public void setFilledDrawable(@NonNull Drawable drawable) {
        if (drawable.getConstantState() == null) {
            return;
        }

        ClipDrawable clipDrawable =
                new ClipDrawable(drawable.getConstantState().newDrawable(), Gravity.START,
                                 ClipDrawable.HORIZONTAL);
        mFilledView.setImageDrawable(clipDrawable);
    }

    public void setEmptyDrawable(@NonNull Drawable drawable) {
        if (drawable.getConstantState() == null) {
            return;
        }

        ClipDrawable clipDrawable =
                new ClipDrawable(drawable.getConstantState().newDrawable(), Gravity.END,
                                 ClipDrawable.HORIZONTAL);
        mEmptyView.setImageDrawable(clipDrawable);
    }

    public void setFilled() {
        mFilledView.setImageLevel(10000);
        mEmptyView.setImageLevel(0);
        if (mGuidanceText != null) {
            mGuidanceText.setTextColor(Color.WHITE);
        }
    }

    public void setPartialFilled(float rating) {
        float percentage = rating % 1;
        int level = (int) (10000 * percentage);
        level = level == 0 ? 10000 : level;
        mFilledView.setImageLevel(level);
        mEmptyView.setImageLevel(10000 - level);
        if (mGuidanceText != null) {
            mGuidanceText.setTextColor(Color.WHITE);
        }
    }

    public void setEmpty() {
        mFilledView.setImageLevel(0);
        mEmptyView.setImageLevel(10000);
        if (mGuidanceText != null) {
            mGuidanceText.setTextColor(Color.BLACK);
        }
    }

    public void setStarWidth(@IntRange(from = 0) int starWidth) {
        mStarWidth = starWidth;

        ViewGroup.LayoutParams params = mFilledView.getLayoutParams();
        params.width = mStarWidth;
        mFilledView.setLayoutParams(params);
        mEmptyView.setLayoutParams(params);
    }

    public void setStarHeight(@IntRange(from = 0) int starHeight) {
        mStarHeight = starHeight;

        ViewGroup.LayoutParams params = mFilledView.getLayoutParams();
        params.height = mStarHeight;
        mFilledView.setLayoutParams(params);
        mEmptyView.setLayoutParams(params);
    }
}
