package com.talpa.tengine

import android.content.Context
import android.content.Intent
import android.text.TextUtils
import android.util.Log
import androidx.annotation.Keep
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.google.mlkit.nl.languageid.LanguageIdentification
import com.talpa.tengine.Result.Companion.ERROR_OTHER
import com.talpa.tengine.Result.Companion.SUCCESS
import com.talpa.tengine.Trans.Companion.EVENT_OFFLINE_TRANSLATE_FAILURE
import com.talpa.tengine.Trans.Companion.EVENT_OFFLINE_TRANSLATE_SUCCESS
import com.talpa.tengine.Trans.Companion.LANGUAGE_CHECK_BEFORE_TRANS
import com.talpa.tengine.lang.LANG
import com.talpa.tengine.offline.TranslateFlow
import com.talpa.tengine.offline.startOfflineTranslate
import com.talpa.tengine.store.readTranslation
import com.talpa.tengine.store.storeTranslation
import io.reactivex.*
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.functions.Consumer
import io.reactivex.functions.Function
import io.reactivex.schedulers.Schedulers
import org.reactivestreams.Publisher
import java.util.*

/**
 * build : TranslatorBuilder(application).apply(builder).build()
 *
 * @author CY 2020/3/31
 */
@Keep
class OSCTranslator internal constructor(
    private val context: Context,
    private val serverAppKey: String? = null,
    private val serverAppSecret: String? = null,
    private val googleKey: String? = null,
    private val microsoftKey: String? = null,
    private val isEnabledOfflineTranslate: Boolean,
    private val isDownloadModelIfNeeded: Boolean = true
) {

    init {
        // context.migrationTranslationDataForSharedPrefer()
    }

    /**
     * 文本翻译
     */
    fun translate(trans: Trans): Flowable<Trans> {

        //val transByServerChargeFlow = transByServerChargeFlow(trans)

        val offlineFlowFlatMapper = Function<Trans, Publisher<Trans>> {
            if (it.result?.code != SUCCESS) {
                transByServerChargeFlow(it)
            } else {
                Flowable.just(it)
            }
        }


        /*val offlineFlow =
            transByOffline(trans = trans, isDownloadModelIfNeeded = isDownloadModelIfNeeded)
                .flatMap(offlineFlowFlatMapper)
                .onErrorResumeNext(transByServerChargeFlow)*/

        val storeFlowFlatMapper = Function<Trans, Publisher<Trans>> {
            if (it.result?.code != SUCCESS) {
                if (isEnabledOfflineTranslate) {
                    transByOffline(trans = it, isDownloadModelIfNeeded = isDownloadModelIfNeeded)
                        .flatMap(offlineFlowFlatMapper)
                        .onErrorResumeNext(transByServerChargeFlow(it))
                } else transByServerChargeFlow(it)
            } else {
                Flowable.just(it)
            }
        }

        val storeNext = Consumer<Trans> {
            val translation = it.result?.translation
            if (!TextUtils.isEmpty(translation) && it.result?.code == SUCCESS) {
                saveTrans(it.from, it.to, it.text, translation!!)
            }
        }

        val languageId = Flowable.create(object : FlowableOnSubscribe<Trans> {
            override fun subscribe(emitter: FlowableEmitter<Trans>) {
                val languageId = LanguageIdentification.getClient()
                languageId.identifyLanguage(trans.text)
                    .addOnCompleteListener { languageCode ->
                        if (!languageCode.result.isNullOrEmpty() && languageCode.result != "und") {
                            Log.d("cjslog", "dected:${languageCode.result}")
                            val dectectLan = languageCode.result!!.substringBefore("-Latn", languageCode.result!!)
                            if (dectectLan == trans.to) {
                                trans.to = trans.from
                                trans.from = dectectLan
                            } else {
                                trans.from = LANG.AUTO
                            }
                        } else {
                            trans.from = LANG.AUTO
                        }
                        appendEvent(LANGUAGE_CHECK_BEFORE_TRANS, hashMapOf("language_id" to trans.from))
                        Log.d("cjslog", "language id:${trans.from}")
                        emitter.onNext(trans)
                    }
                    .addOnFailureListener { exception ->
                        Log.d("cjslog", "dected failed")
                        trans.from = LANG.AUTO
                        emitter.onNext(trans)
                    }
            }

        }, BackpressureStrategy.BUFFER)

        /**
         *  store -> 离线翻译->server->google->microsoft
         */
        Log.d("cjslog", "trans request from:${trans}")
        return if (trans.text.length > 15) {
            languageId.flatMap {
                transByStore(it)
            }.flatMap(storeFlowFlatMapper)
                .doOnNext(storeNext)
        } else {
            trans.from = LANG.AUTO
            transByStore(trans)
                .flatMap(storeFlowFlatMapper)
                .doOnNext(storeNext)
        }

    }

    private fun transByOffline(
        trans: Trans,
        isDownloadModelIfNeeded: Boolean = true
    ): Flowable<Trans> {
        val translateFlow = TranslateFlow()
        //val trans = Trans(sourceLanguage, targetLanguage, text = text, context = context)
        val mapper = Consumer<Trans> {
            if (it.result?.code == SUCCESS) {
                sendTranslateEvent(EVENT_OFFLINE_TRANSLATE_SUCCESS, it)
            } else {
                sendTranslateEvent(EVENT_OFFLINE_TRANSLATE_FAILURE, it)
            }
        }
        return translateFlow.startOfflineTranslate(trans, isDownloadModelIfNeeded).doOnNext(mapper)
    }

    private fun transByServerChargeFlow(trans: Trans): Flowable<Trans> {
        return context.translateBySCTranslator(trans) {
            setGoogleKey(this@OSCTranslator.googleKey)
            setMicrosoftKey(this@OSCTranslator.microsoftKey)
            setServerAppKey(this@OSCTranslator.serverAppKey)
            setServerAppSecret(this@OSCTranslator.serverAppSecret)
            isEnabledOfflineTranslate(this@OSCTranslator.isEnabledOfflineTranslate)
            isDownloadModelIfNeeded(this@OSCTranslator.isDownloadModelIfNeeded)
        }
    }


    /**
     * Read Store
     */
    private fun transByStore(trans: Trans): Flowable<Trans> {

        val sourceLanguage: String = trans.from
        val targetLanguage: String = trans.to
        val text: String = trans.text

        return Flowable.fromCallable {

            val translation = readTranslation(sourceLanguage, targetLanguage, text)
            val result = if (translation != null) {
                Log.d("cjslog", "read cache success")
                Trans.Result(
                    SUCCESS,
                    translation = translation,
                    isCached = true,
                    source = STORE
                )
            } else {
                Log.d("cjslog", "read cache failed")
                Trans.Result(
                    ERROR_OTHER,
                    translation = translation,
                    isCached = false,
                    source = STORE
                )
            }
            trans.result = result
            trans
        }
    }

    private fun saveTrans(
        sourceLanguage: String?,
        targetLanguage: String,
        text: String,
        translation: String
    ) {
        storeTranslation(
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage,
            text = text,
            translation = translation
        )
    }

    private fun sendTranslateEvent(eventId: String, trans: Trans) {
        appendEvent(eventId, hashMapOf("language" to "${trans.from}-${trans.to}"))
    }

    /**
     * 埋点
     */
    private fun appendEvent(eventId: String, params: HashMap<String, String>? = null) {
        try {
            val intent = Intent(Trans.ACTION_DATA_TRANS)
            intent.putExtra(Trans.EXTRA_EVENT_ID, eventId)
            intent.putExtra(Trans.EXTRA_EVENT_PARAMS, params)
            LocalBroadcastManager.getInstance(context)
                .sendBroadcast(intent)
        } catch (e: Throwable) {
            e.printStackTrace()
        }
    }

    private fun <T> transformer(): FlowableTransformer<T, T> {
        return FlowableTransformer<T, T> { upstream ->
            upstream.subscribeOn(Schedulers.io()).observeOn(AndroidSchedulers.mainThread())
        }
    }

}


/**
 * Build
 */
private fun TranslatorBuilder.buildOSCTranslator(): OSCTranslator {
    return OSCTranslator(
        context = context,
        serverAppKey = serverAppKey,
        serverAppSecret = serverAppSecret,
        googleKey = googleKey,
        microsoftKey = microsoftKey,
        isEnabledOfflineTranslate = isEnabledOfflineTranslate,
        isDownloadModelIfNeeded = isDownloadModelIfNeeded
    )
}


/**
 * Translate
 */
fun Context.translateByOSCTranslator(
    trans: Trans,
    builder: TranslatorBuilder.() -> Unit = {}
): Flowable<Trans> {
    val translator = TranslatorBuilder(applicationContext).apply(builder).buildOSCTranslator()
    return translator.translate(trans)
}