package com.talpa.tengine

import android.content.Context
import android.content.Intent
import android.text.TextUtils
import android.util.Log
import androidx.annotation.Keep
import androidx.localbroadcastmanager.content.LocalBroadcastManager
import com.talpa.tengine.Result.Companion.ERROR_OTHER
import com.talpa.tengine.Result.Companion.SUCCESS
import com.talpa.tengine.Trans.Companion.EVENT_TRANS_FAIL_FROM_GOOGLE_CHARGE
import com.talpa.tengine.Trans.Companion.EVENT_TRANS_FAIL_FROM_MICROSOFT_CHARGE
import com.talpa.tengine.Trans.Companion.EVENT_TRANS_FAIL_FROM_SERVER
import com.talpa.tengine.Trans.Companion.EVENT_TRANS_SUCCESS_FROM_GOOGLE_CHARGE
import com.talpa.tengine.Trans.Companion.EVENT_TRANS_SUCCESS_FROM_MICROSOFT_CHARGE
import com.talpa.tengine.Trans.Companion.EVENT_TRANS_SUCCESS_FROM_SERVER
import com.talpa.tengine.charge.GoogleCloudTranslate
import com.talpa.tengine.charge.MicrosoftTranslate
import com.talpa.tengine.store.readTranslation
import com.talpa.tengine.store.storeTranslation
import com.talpa.translate.HiTranslator
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.FlowableOnSubscribe
import io.reactivex.FlowableTransformer
import io.reactivex.android.schedulers.AndroidSchedulers
import io.reactivex.functions.Consumer
import io.reactivex.functions.Function
import io.reactivex.schedulers.Schedulers
import okhttp3.ResponseBody
import org.reactivestreams.Publisher

/**
 * build : TranslatorBuilder(application).apply(builder).build()
 *
 * @author CY 2020/3/31
 */
@Keep
class SCTranslator internal constructor(
    private val context: Context,
    private val googleKey: String? = null,
    private val microsoftKey: String? = null,
    private val serverAppKey: String? = null,
    private val serverAppSecret: String? = null
) {

    /**
     * Translate
     */
    @Keep
    fun translate(trans: Trans): Flowable<Trans> {

        val flowError = Flowable.create<Trans>({ emitter ->
            val result = Trans.Result(Trans.ERROR_OTHER)
            trans.result = result
            if (!emitter.isCancelled) {
                emitter.onNext(trans)
                emitter.onComplete()
            }
        }, BackpressureStrategy.LATEST)

        val transByMicrosoftFlow = microsoftKey?.let {
            transByMicrosoft(trans = trans, microsoftKey = it).onErrorResumeNext(flowError)
        }

        val transByGoogleFlow = googleKey?.let {
            transByGoogle(trans = trans, googleKey = it)
                .apply {
                    when {
                        transByMicrosoftFlow != null -> {
                            onErrorResumeNext(transByMicrosoftFlow)
                        }
                    }
                }
        }


        val googleFlowFlatMapper = Function<Trans, Publisher<Trans>> {

            if (it.result?.code != SUCCESS) {
                transByMicrosoftFlow ?: Flowable.just(it)
            } else {
                Flowable.just(it)
            }
        }

        val serverFlowFlatMapper = Function<Trans, Publisher<Trans>> {

            if (it.result?.code != SUCCESS) {
                transByGoogleFlow?.flatMap(googleFlowFlatMapper) ?: transByMicrosoftFlow
                ?: Flowable.just(it)
            } else {
                Flowable.just(it)
            }
        }
        val serverFlow = if (serverAppKey != null && serverAppSecret != null) {
            transByServerFlow(
                trans = trans,
                serverAppKey = serverAppKey,
                serverAppSecret = serverAppSecret
            ).apply {
                when {
                    transByGoogleFlow != null -> {
                        onErrorResumeNext(transByGoogleFlow)
                    }
                    transByMicrosoftFlow != null -> {
                        onErrorResumeNext(transByMicrosoftFlow)
                    }
                }
            }.flatMap(serverFlowFlatMapper)
        } else null

        /* val serverFlow = serverAppKey?.let {
             transByServerFlow(
                 trans = trans,
                 serverAppKey = serverAppKey,
                 serverAppSecret = serverAppSecret
             ).apply {
                 when {
                     transByGoogleFlow != null -> {
                         onErrorResumeNext(transByGoogleFlow)
                     }
                     transByMicrosoftFlow != null -> {
                         onErrorResumeNext(transByMicrosoftFlow)
                     }
                 }
             }
         }?.flatMap(serverFlowFlatMapper)*/

        val storeFlowFlatMapper = Function<Trans, Publisher<Trans>> {
            if (it.result?.code != SUCCESS) {
                serverFlow ?: transByGoogleFlow ?: transByMicrosoftFlow ?: Flowable.just(it)
            } else {
                Flowable.just(it)
            }
        }

        val storeNext = Consumer<Trans> {
            val translation = it.result?.translation
            if (!TextUtils.isEmpty(translation) && it.result?.code == SUCCESS) {
                saveTrans(it.from, it.to, it.text, translation!!)
            }
        }

        /**
         *  store ->server->google->microsoft
         */
        return transByStore(trans)
            .flatMap(storeFlowFlatMapper)
            .doOnNext(storeNext)
    }

    private fun transByServer(
        sourceLanguage: String,
        targetLanguage: String,
        text: String,
        serverAppKey: String,
        serverAppSecret: String
    ): HiTranslator.Result {
        return HiTranslator.Builder()
            .setSourceLanguage(sourceLanguage)
            .setTargetLanguage(targetLanguage)
            .setText(text)
            .setAppKey(serverAppKey)
            .setSecret(serverAppSecret)
            .build()
            .execute()
    }

    /**
     * Google Translate
     */
    private fun transByGoogle(
        trans: Trans,
        googleKey: String
    ): Flowable<Trans> {
        val translate = GoogleCloudTranslate(key = googleKey)
        translate.setFormData(trans.from, trans.to, trans.text)

        val mapper = Function<retrofit2.Response<ResponseBody>, Trans> { response ->

            val result = if (response.isSuccessful) {
                val list = translate.parses(response)

                val translation = list[0]

                sendTranslateEvent(EVENT_TRANS_SUCCESS_FROM_GOOGLE_CHARGE, trans)

                Trans.Result(
                    code = SUCCESS,
                    translation = translation,
                    detectLang = trans.from,
                    source = GOOGLE
                )
            } else {
                sendTranslateEvent(EVENT_TRANS_FAIL_FROM_GOOGLE_CHARGE, trans)
                Trans.Result(
                    code = ERROR_OTHER,
                    errorMessage = response.errorBody()?.string(),
                    source = GOOGLE
                )
            }
            trans.result = result

            trans
        }
        return translate.execute().map(mapper).compose(transformer())
    }

    /**
     * Microsoft Translate
     */
    private fun transByMicrosoft(
        trans: Trans,
        microsoftKey: String
    ): Flowable<Trans> {

        val translate = MicrosoftTranslate(microsoftKey)

        if (!translate.isSupportLanguage(trans.to)) {
            val result = Trans.Result(ERROR_OTHER, "not support language")
            trans.result = result
            return Flowable.just(trans)
        }

        translate.setFormData(trans.from, trans.to, trans.text)

        val mapper = Function<retrofit2.Response<ResponseBody>, Trans> { response ->

            val result = if (response.isSuccessful) {
                val map = translate.parses2(response)

                val (language, translation) = map.entries.first()

                sendTranslateEvent(EVENT_TRANS_SUCCESS_FROM_MICROSOFT_CHARGE, trans)

                Trans.Result(
                    code = SUCCESS,
                    translation = translation,
                    detectLang = language,
                    source = MICROSOFT
                )
            } else {
                sendTranslateEvent(EVENT_TRANS_FAIL_FROM_MICROSOFT_CHARGE, trans)
                Trans.Result(
                    code = ERROR_OTHER,
                    errorMessage = response.errorBody()?.string(),
                    source = MICROSOFT
                )

            }
            trans.result = result

            trans
        }

        return translate.execute().map(mapper).compose(transformer())
    }

    private fun transByServerFlow(
        trans: Trans,
        serverAppKey: String,
        serverAppSecret: String
    ): Flowable<Trans> {
        val serverSubscribe = FlowableOnSubscribe<Trans> { emitter ->
            try {

                Log.d("cjslog", "start server request")
                val supportLanguageMap = HiTranslator.supportLanguageMap
                val sourceLanguage = supportLanguageMap[trans.from]
                val targetLanguage = supportLanguageMap[trans.to]
                if (sourceLanguage != null && targetLanguage != null) {

                    val startTime = System.currentTimeMillis()
                    val result = transByServer(
                        sourceLanguage,
                        targetLanguage,
                        trans.text,
                        serverAppKey,
                        serverAppSecret
                    )
                    val endTime = System.currentTimeMillis()

                    val duration = (endTime - startTime) / 1000f

                    val transResult = if (result.isSuccess) {

                        sendTranslateEvent(EVENT_TRANS_SUCCESS_FROM_SERVER, trans, duration)

                        Trans.Result(
                            SUCCESS,
                            translation = result.translation,
                            isCached = result.isCache,
                            source = SERVER
                        )
                    } else {
                        //Log.i("Translator", "result error=${result.valueMessage}")
                        sendTranslateEvent(EVENT_TRANS_FAIL_FROM_SERVER, trans)

                        Trans.Result(
                            ERROR_OTHER,
                            errorMessage = result.valueMessage,
                            source = SERVER
                        )
                    }

                    trans.apply { this.result = transResult }
                } else {
                    val result = Trans.Result(
                        ERROR_OTHER,
                        errorMessage = "not support language ${trans.to}",
                        source = SERVER
                    )
                    trans.apply { this.result = result }
                }


                if (!emitter.isCancelled) {
                    emitter.onNext(trans)
                }
            } catch (e: Exception) {
                e.printStackTrace()
                if (!emitter.isCancelled) {
                    emitter.tryOnError(e)
                }
            } finally {
                if (!emitter.isCancelled) {
                    emitter.onComplete()
                }
            }


        }

        val errorFlow = Function<Throwable, Publisher<Trans>> {
            val transResult = Trans.Result(ERROR_OTHER, errorMessage = it.message)
            trans.apply { this.result = transResult }
            Flowable.just(trans)
        }

        return Flowable.create(serverSubscribe, BackpressureStrategy.LATEST)
            .compose(transformer())
            .onErrorResumeNext(errorFlow)
    }


    /**
     * Read Store
     */
    private fun transByStore(trans: Trans): Flowable<Trans> {

        val sourceLanguage: String = trans.from
        val targetLanguage: String = trans.to
        val text: String = trans.text

        return Flowable.fromCallable {

            val translation = readTranslation(sourceLanguage, targetLanguage, text)
            val result = if (translation != null) {
                Trans.Result(
                    SUCCESS,
                    translation = translation,
                    isCached = true,
                    source = STORE
                )
            } else {
                Trans.Result(
                    ERROR_OTHER,
                    isCached = false,
                    source = STORE
                )
            }
            trans.result = result
            trans
        }
    }

    private fun saveTrans(
        sourceLanguage: String?,
        targetLanguage: String,
        text: String,
        translation: String
    ) {
        storeTranslation(
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage,
            text = text,
            translation = translation
        )
    }

    private fun sendTranslateEvent(eventId: String, trans: Trans, duration: Float = 0f) {
        val language = "${trans.from}-${trans.to}"
        val params = if (duration > 0f) {
            hashMapOf("language" to language, "duration" to duration.toString())
        } else {
            hashMapOf("language" to language)
        }

        appendEvent(eventId, params)
    }

    /**
     * 埋点
     */
    private fun appendEvent(eventId: String, params: HashMap<String, String>? = null) {
        try {
            val intent = Intent(Trans.ACTION_DATA_TRANS)
            intent.putExtra(Trans.EXTRA_EVENT_ID, eventId)
            intent.putExtra(Trans.EXTRA_EVENT_PARAMS, params)
            LocalBroadcastManager.getInstance(context)
                .sendBroadcast(intent)
        } catch (e: Throwable) {
            e.printStackTrace()
        }
    }

    private fun <T> transformer(): FlowableTransformer<T, T> {
        return FlowableTransformer<T, T> { upstream ->
            upstream.subscribeOn(Schedulers.io()).observeOn(AndroidSchedulers.mainThread())
        }
    }

}

/**
 * Build
 */
private fun TranslatorBuilder.buildSCTranslator(): SCTranslator {
    return SCTranslator(
        context,
        googleKey,
        microsoftKey,
        serverAppKey,
        serverAppSecret = serverAppSecret
    )
}


/**
 * Translate
 */
@Keep
fun Context.translateBySCTranslator(
    trans: Trans,
    builder: TranslatorBuilder.() -> Unit = {}
): Flowable<Trans> {
    val translator = TranslatorBuilder(applicationContext).apply(builder).buildSCTranslator()
    return translator.translate(trans)
}
