package com.talpa.tengine.offline

import android.util.Log
import com.google.android.gms.tasks.OnFailureListener
import com.google.android.gms.tasks.OnSuccessListener
import com.google.mlkit.common.model.DownloadConditions
import com.google.mlkit.common.model.RemoteModelManager
import com.google.mlkit.nl.languageid.LanguageIdentification
import com.google.mlkit.nl.translate.TranslateLanguage
import com.google.mlkit.nl.translate.TranslateRemoteModel
import com.google.mlkit.nl.translate.Translation
import com.google.mlkit.nl.translate.TranslatorOptions
import com.talpa.translator.link.BuildConfig
import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import io.reactivex.FlowableEmitter
import io.reactivex.FlowableOnSubscribe
import io.reactivex.functions.Function
import io.reactivex.schedulers.Schedulers

/**
 * 离线翻译响应式流
 *
 * @author CY 2019-12-19
 */
class TranslateFlow {


    companion object {

        const val TAG = "TranslateFlow"

        /**
         * 识别语言错误
         */
        const val ERROR_IDENTIFY_LANGUAGE = 100

        /**
         * 识别语言成功
         */
        const val SUCCESS_IDENTIFY_LANGUAGE = 101

        /**
         * 翻译错误
         */
        const val ERROR_TRANSLATE = 200

        /**
         * 翻译成功
         */
        const val SUCCESS_TRANSLATE = 201

        /**
         * 下载模型错误
         */
        const val ERROR_DOWNLOAD_MODE = 300

        /**
         * 下载模型成功
         */
        const val SUCCESS_DOWNLOAD_MODE = 301


        /**
         * 未下载模型
         */
        const val ERROR_NOT_MODE = 302

        /**
         *  不支持的语言
         */
        const val ERROR_NOT_SUPPORT_LANGUAGE = 303

    }

    /**
     * Translate
     *
     * @param sourceLanguage 源语言
     * @param targetLanguage 目标语言
     * @param text 需要翻译的文本
     * @param downloadModelIfNeeded 是否自动下载翻译模型 默认 false
     * @param downloadModelRequireWifi 是否制定WIFI情况下下载模型 默认 true
     * @param flowResult 承接上一个结果，例如将识别文本语言的结果传入
     */
    fun translate(
        @TranslateLanguage.Language sourceLanguage: String,
        @TranslateLanguage.Language targetLanguage: String,
        text: String,
        downloadModelIfNeeded: Boolean = false,
        downloadModelRequireWifi: Boolean = true,
        flowResult: Result? = null
    ): Flowable<Result> {

        val result = flowResult ?: Result(
            code = SUCCESS_TRANSLATE,
            sourceText = text,
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage
        )

        val subscribe = FlowableOnSubscribe<Result> { emitter ->

            val options = TranslatorOptions.Builder()
                .setSourceLanguage(sourceLanguage)
                .setTargetLanguage(targetLanguage)
                .build()

            val translator = Translation.getClient(options)

            emitter.setCancellable {
                translator.close()
            }

            /**
             * 翻译成功
             */
            val onTranslateSuccessListener = OnSuccessListener<String> { translatedText ->

                Log.d("cjslog", "translate success .translatedText=$translatedText")
                if (BuildConfig.DEBUG) log(TAG, "translate success .translatedText=$translatedText")

                result.code =
                    SUCCESS_TRANSLATE
                result.translation = translatedText

                if (!emitter.isCancelled) {
                    emitter.onNext(result)
                    emitter.onComplete()
                }

            }

            /**
             * 翻译失败
             */
            val onTranslateFailureListener = OnFailureListener { exception ->

                exception.printStackTrace()
                Log.d("cjslog", "translate failure .translatedText=$exception.localizedMessage")
                if (BuildConfig.DEBUG) log(TAG, "translate failure .${exception.localizedMessage}")
                result.code =
                    ERROR_TRANSLATE
                result.exception = exception

                if (!emitter.isCancelled) {
                    emitter.onNext(result)
                    emitter.onComplete()
                }
            }

            /**
             * 翻译任务
             */
            val translateTask = {
                translator.translate(text)
                    .addOnSuccessListener(onTranslateSuccessListener)
                    .addOnFailureListener(onTranslateFailureListener)
            }


            /**
             * 模型下载成功
             */
            val onDownloadSuccessListener = OnSuccessListener<Void> {

                result.code =
                    SUCCESS_DOWNLOAD_MODE
                //emitter.onNext(result)
                //if (BuildConfig.DEBUG) log(TAG, "download model success .")
                translateTask.invoke()
            }

            /**
             * 模型下载失败
             */
            val onDownloadFailureListener = OnFailureListener { exception ->
                if (BuildConfig.DEBUG) log(TAG, "download model failure .")
                result.code =
                    ERROR_DOWNLOAD_MODE
                result.exception = exception
                if (!emitter.isCancelled) {
                    emitter.onNext(result)
                    emitter.onComplete()
                }
            }


            /**
             * 下载模型任务
             */
            val downloadModelTask = {

                val task = if (downloadModelRequireWifi) {
                    val conditions = DownloadConditions.Builder()
                        .requireWifi()
                        .build()
                    translator.downloadModelIfNeeded(conditions)
                } else {
                    translator.downloadModelIfNeeded()
                }

                task.addOnSuccessListener(onDownloadSuccessListener)
                    .addOnFailureListener(onDownloadFailureListener)
            }

            /*if (downloadModelIfNeeded) {
                downloadModelTask.invoke()
            } else {
                translateTask.invoke()
            }*/
            translateTask.invoke()
        }

        return Flowable.create<Result>(subscribe, BackpressureStrategy.LATEST)
    }

    /**
     * Auto Translate
     *
     * @param targetLanguage 目标语言
     * @param text 需要翻译的文本
     * @param downloadModelIfNeeded 是否自动下载翻译模型 默认 false
     * @param downloadModelRequireWifi 是否制定WIFI情况下下载模型 默认 true
     */
    fun autoTranslate(
        @TranslateLanguage.Language targetLanguage: String,
        text: String,
        downloadModelIfNeeded: Boolean = false,
        downloadModelRequireWifi: Boolean = true
    ): Flowable<Result> {

        val flatMap = Function<Result, Flowable<Result>> { result: Result ->

            val identifyLanguageCode = result.identifyLanguageCode

            var languageCode: String? = null

            if (result.code == SUCCESS_IDENTIFY_LANGUAGE && identifyLanguageCode != null) {
                languageCode = TranslateLanguage.fromLanguageTag(identifyLanguageCode)
            }
            /**
             * 识别成功语言码
             */
            if (languageCode != null) {

                val downloaded = {
                    translate(
                        sourceLanguage = languageCode,
                        targetLanguage = targetLanguage,
                        text = text,
                        downloadModelIfNeeded = downloadModelIfNeeded,
                        downloadModelRequireWifi = downloadModelRequireWifi,
                        flowResult = result
                    )
                }
                //是否下载语言模型
                isModelDownloaded(
                    languageCode,
                    downloadModelIfNeeded,
                    downloaded = downloaded
                )
            } else {
                result.code =
                    ERROR_NOT_SUPPORT_LANGUAGE
                Flowable.just(result)
            }
        }

        val downloaded = {
            identifyLanguage(text).flatMap(flatMap)
        }
        return isModelDownloaded(
            targetLanguage,
            downloadModelIfNeeded,
            downloaded = downloaded
        )

    }

    /**
     * Identify Language
     *
     * @param text 需要识别的文本
     */
    private fun identifyLanguage(text: String): Flowable<Result> {

        val sub = FlowableOnSubscribe<Result> { emitter ->

            val onSuccessListener = OnSuccessListener<String> { languageCode ->

                val result = if (languageCode != "und") {
                    if (BuildConfig.DEBUG) log(
                        TAG,
                        "identify language success .languageCode=$languageCode"
                    )
                    Result(
                        code = SUCCESS_IDENTIFY_LANGUAGE,
                        sourceText = text,
                        identifyLanguageCode = languageCode
                    )
                } else {
                    if (BuildConfig.DEBUG) log(TAG, "Can't identify language.")
                    Result(
                        code = ERROR_IDENTIFY_LANGUAGE,
                        sourceText = text,
                        identifyLanguageCode = languageCode
                    )
                }

                if (!emitter.isCancelled) {
                    emitter.onNext(result)
                    emitter.onComplete()
                }


            }

            val onFailureListener = OnFailureListener { exception ->
                val result = Result(
                    code = ERROR_IDENTIFY_LANGUAGE,
                    exception = exception
                )

                if (!emitter.isCancelled) {
                    emitter.onNext(result)
                    emitter.onComplete()
                }
                if (BuildConfig.DEBUG) log(
                    TAG,
                    "identify language failure .${exception.localizedMessage}"
                )
            }

            LanguageIdentification.getClient().identifyLanguage(text)
                .addOnSuccessListener { languageCode ->
                    val result = if (languageCode == "und") {
                        if (BuildConfig.DEBUG) log(TAG, "Can't identify language.")
                        Result(
                            code = ERROR_IDENTIFY_LANGUAGE,
                            sourceText = text,
                            identifyLanguageCode = languageCode
                        )
                    } else {
                        if (BuildConfig.DEBUG) log(
                            TAG,
                            "identify language success .languageCode=$languageCode"
                        )
                        Result(
                            code = SUCCESS_IDENTIFY_LANGUAGE,
                            sourceText = text,
                            identifyLanguageCode = languageCode
                        )
                    }
                    emitter.onNext(result)
                    emitter.onComplete()
                }
                .addOnFailureListener { exception ->
                    val result = Result(
                        code = ERROR_IDENTIFY_LANGUAGE,
                        exception = exception
                    )

                    if (!emitter.isCancelled) {
                        emitter.onNext(result)
                        emitter.onComplete()
                    }
                    if (BuildConfig.DEBUG) log(
                        TAG,
                        "identify language failure .${exception.localizedMessage}"
                    )

                }


            /*emitter.setCancellable {
                languageIdentifier.close()
            }*/
        }

        return Flowable.create<Result>(sub, BackpressureStrategy.LATEST)


        /**
         * 计数器
         */
        /*val countDownLatch = CountDownLatch(1)
        countDownLatch.await()*/
    }

    /**
     * 是否下载语言模型，
     * 如果已经下载，将执行高阶函数downloaded
     * 如果未下载，根据 downloadModelIfNeeded 决定是否启动下载，同时返回错误结果
     */
    private fun isModelDownloaded(
        @TranslateLanguage.Language language: String,
        downloadModelIfNeeded: Boolean,
        downloaded: () -> Flowable<Result>
    ): Flowable<Result> {


        val targetFlow = isModelDownloaded(language)
        return targetFlow.flatMap { isModelDownloaded ->
            if (isModelDownloaded) {// || downloadModelIfNeeded
                downloaded.invoke()
            } else {
                if (downloadModelIfNeeded) {
                    val downloadModel = downloadModel(language).subscribeOn(Schedulers.io())
                        .subscribe({}, Throwable::printStackTrace)
                }
                Flowable.just(Result(code = ERROR_NOT_MODE))
            }
        }
    }

    /**
     * 是否下载了模型
     */
    fun isModelDownloaded(@TranslateLanguage.Language language: String): Flowable<Boolean> {


        val subscribe = FlowableOnSubscribe { emitter: FlowableEmitter<Boolean> ->

            val manager = RemoteModelManager.getInstance()
            val remoteModel = TranslateRemoteModel.Builder(language).build()
            //val targetRemoteModel = FirebaseTranslateRemoteModel.Builder(targetLanguage).build()

            val onSuccessListener = OnSuccessListener<Boolean> {
                if (BuildConfig.DEBUG) {
                    val code = TranslateLanguage.fromLanguageTag(language)
                    //val code = FirebaseTranslateLanguage.languageCodeForLanguage(language)
                    log(TAG, "$code isModelDownloaded $it .")
                }
                if (!emitter.isCancelled) {
                    emitter.onNext(it)
                    emitter.onComplete()
                }
            }
            val onFailureListener = OnFailureListener {
                if (BuildConfig.DEBUG) {
                    val code = TranslateLanguage.fromLanguageTag(language)
                    log(TAG, "$code isModelDownloaded failure .")
                }
                if (!emitter.isCancelled) {
                    emitter.tryOnError(it)
                }
            }
            manager.isModelDownloaded(remoteModel)
                .addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener)
        }

        return Flowable.create(subscribe, BackpressureStrategy.LATEST)
    }

    /**
     * 下载模型
     */
    fun downloadModel(
        @TranslateLanguage.Language language: String,
        requireWifi: Boolean = true
    ): Flowable<Boolean> {

        val subscribe = FlowableOnSubscribe { emitter: FlowableEmitter<Boolean> ->

            val onSuccessListener = OnSuccessListener<Void> {
                if (BuildConfig.DEBUG) {
                    log(TAG, "download model success .")
                }
                if (!emitter.isCancelled) {
                    emitter.onNext(true)
                    emitter.onComplete()
                }
            }
            val onFailureListener = OnFailureListener {
                if (BuildConfig.DEBUG) {
                    log(TAG, "download model failure .")
                }
                if (!emitter.isCancelled) {
                    emitter.tryOnError(it)
                    emitter.onComplete()
                }
            }

            val manager = RemoteModelManager.getInstance()
            val remoteModel = TranslateRemoteModel.Builder(language).build()

            val conditions =
                if (requireWifi) {
                    DownloadConditions.Builder().requireWifi().build()
                } else {
                    DownloadConditions.Builder().build()
                }
            manager.download(remoteModel, conditions).addOnSuccessListener(onSuccessListener)
                .addOnFailureListener(onFailureListener)

        }
        return Flowable.create<Boolean>(subscribe, BackpressureStrategy.BUFFER)
    }

    data class Result(
        var code: Int,
        var sourceText: String? = null,
        var translation: String? = null,
        var exception: Exception? = null,
        var identifyLanguageCode: String? = null,
        var sourceLanguage: String? = null,
        var targetLanguage: String? = null
    ) {
        override fun toString(): String {
            return "Result(code=$code, sourceText=$sourceText, translation=$translation, exception=$exception, identifyLanguageCode=$identifyLanguageCode, sourceLanguage=$sourceLanguage, targetLanguage=$targetLanguage)"
        }
    }

    fun log(tag: String = TAG, message: String) {
        Log.d(tag, message)
    }
}

