package com.talpa.tengine.request

import io.reactivex.BackpressureStrategy
import io.reactivex.Flowable
import okhttp3.*
import retrofit2.Call
import retrofit2.Retrofit
import retrofit2.http.*
import java.io.File
import java.util.concurrent.TimeUnit


/**
 * 网络请求
 * @author CY 2019/1/28
 */
object RequestHelper {

    private val map: HashMap<String, ApiService> = hashMapOf()

    var cacheFile: File? = null

    /**
     * 云端响应头拦截器，用来配置缓存策略
     * Dangerous interceptor that rewrites the server's cache-control header.
     */
    private val REWRITE_CACHE_CONTROL_INTERCEPTOR = { chain: Interceptor.Chain ->
        val request = chain.request().newBuilder()
            .cacheControl(CacheControl.Builder().maxStale(365, TimeUnit.DAYS).build())
            .build()
        val originalResponse = chain.proceed(request)
        val cacheControl = request.cacheControl.toString()
        originalResponse.newBuilder()
            .header("Cache-Control", cacheControl)
            .removeHeader("Pragma")
            .build()
    }

    private fun apiService(baseUrl: String): ApiService {

        if (map.containsKey(baseUrl)) {
            return map[baseUrl]!!
        }

        val maxCacheSize: Long = 1024 * 1024 * 10

        val cacheFile = cacheFile
        val cache = if (cacheFile != null && cacheFile.exists()) {
            Cache(cacheFile, maxCacheSize)
        } else {
            null
        }


        val okHttpClient = OkHttpClient
            .Builder()
            // .cache(cache)
            .retryOnConnectionFailure(false)
            //retrofit 缓存 需要addInterceptor，addNetworkInterceptor 同时添加
//            .addInterceptor(REWRITE_CACHE_CONTROL_INTERCEPTOR)
//            .addNetworkInterceptor(REWRITE_CACHE_CONTROL_INTERCEPTOR)
            //.addNetworkInterceptor(StethoInterceptor())
            .callTimeout(6000, TimeUnit.MILLISECONDS)
            .build()
        val retrofit = Retrofit
            .Builder()
            .client(okHttpClient)
            .baseUrl(baseUrl)
            .build()
        val apiService = retrofit.create(ApiService::class.java)
        map[baseUrl] = apiService

        return apiService
    }

    fun get(
        baseUrl: String,
        paths: String = "",
        headerMap: Map<String, String> = hashMapOf(),
        params: Map<String, String> = hashMapOf()
    ) = apiService(baseUrl)
        .get(headerMap, paths, params).toFlow()//.timeout(6, TimeUnit.SECONDS)

    fun post(
        baseUrl: String,
        paths: String = "",
        headerMap: Map<String, String> = hashMapOf(),
        params: Map<String, String> = hashMapOf()
    ) = apiService(baseUrl)
        .post(headerMap, paths, params).toFlow()//.timeout(6, TimeUnit.SECONDS)

    fun postBody(
        baseUrl: String,
        paths: String = "",
        headerMap: Map<String, String> = hashMapOf(),
        requestBody: RequestBody,
        params: Map<String, String> = hashMapOf()
    ) = apiService(baseUrl)
        .postBody(headerMap, paths, requestBody, params).toFlow()


    interface ApiService {

        /**
         * 直接返回 Flowable<ResponseBody> timeout not work
         */
        @GET
        fun get(
            @HeaderMap headerMap: Map<String, String> = hashMapOf(),
            @Url url: String = "",
            @QueryMap params: Map<String, String> = hashMapOf()
        ): Call<ResponseBody>//Flowable<ResponseBody>

        @FormUrlEncoded
        @POST
        fun post(
            @HeaderMap headerMap: Map<String, String> = hashMapOf(),
            @Url url: String = "",
            @FieldMap params: Map<String, String> = hashMapOf()
        ): Call<ResponseBody>

        @POST
        fun postBody(
            @HeaderMap headerMap: Map<String, String> = hashMapOf(),
            @Url url: String = "",
            @Body requestBody: RequestBody,
            @QueryMap params: Map<String, String> = hashMapOf()
        ): Call<ResponseBody>

    }

    private fun Call<ResponseBody>.toFlow(): Flowable<retrofit2.Response<ResponseBody>> {

        return Flowable.create<retrofit2.Response<ResponseBody>>({ emitter ->
            try {
                //设置取消回调
                emitter.setCancellable {
                    if (!isCanceled) {
                        cancel()
                    }
                }

                val response = execute()
                if (!emitter.isCancelled) {
                    //val rawResponse = response.raw()
                    emitter.onNext(response)
                    /*
                    if (response.isSuccessful) {
                        emitter.onNext(response.body()!!)
                    } else {
                        emitter.onNext(response.errorBody()!!)
                    }
                    */
                }
            } catch (e: Exception) {
                e.printStackTrace()
                if (!emitter.isCancelled) {
                    emitter.tryOnError(e)
                }
            } finally {
                if (!emitter.isCancelled) {
                    emitter.onComplete()
                }
            }


        }, BackpressureStrategy.LATEST)
    }

}


