package com.talpa.translate

import android.content.Context
import android.text.TextUtils
import com.talpa.tengine.store.readTranslation
import com.talpa.tengine.store.storeTranslation
import org.json.JSONObject
import java.io.InputStream
import java.io.OutputStream
import java.net.HttpURLConnection
import java.net.URL

/**
 * Hi Translator Sync
 *
 * @author CY 20-3-5
 */
class HiTranslatorV1 private constructor(
    private val context: Context,
    private val sourceLanguage: String?,
    private val targetLanguage: String,
    private val text: String,
    private val key: String
) {

    class Builder(private val context: Context) {//private val context: Context? = null

        private var sourceLanguage: String? = null
        private var targetLanguage: String? = null
        private var text: String? = null
        private var key: String? = null

        fun setSourceLanguage(sourceLanguage: String): Builder {
            this.sourceLanguage = sourceLanguage
            return this
        }

        fun setTargetLanguage(targetLanguage: String): Builder {
            this.targetLanguage = targetLanguage
            return this
        }

        fun setText(text: String): Builder {
            this.text = text
            return this
        }

        fun setKey(key: String): Builder {
            this.key = key
            return this
        }

        fun build(): HiTranslatorV1 {

            targetLanguage ?: throw NullPointerException("targetLanguage can`t null.")
            text ?: throw NullPointerException("text can`t null.")
            key ?: throw NullPointerException("key can`t null.")

            return HiTranslatorV1(
                context = context,
                sourceLanguage = sourceLanguage,
                targetLanguage = targetLanguage!!,
                text = text!!,
                key = key!!
            )
        }
    }

    data class Result constructor(
        val sourceLanguage: String?,
        val targetLanguage: String,
        val text: String,
        val translation: String?,
        val valueCode: Int = 0,
        val valueMessage: String? = null,
        val isCache: Boolean = false,
        val isSuccess: Boolean = valueCode == 1000
    )

    //private val preferKey by lazy { generateKey(sourceLanguage, targetLanguage, text) }

    /**
     * Translate
     */
    fun execute(): Result {

        val translation = readTranslation(
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage,
            text = text
        )

        if (!TextUtils.isEmpty(translation)) {
            return Result(
                sourceLanguage,
                targetLanguage,
                text,
                translation,
                valueCode = 1000,
                isCache = true
            )
        }

        val result = connect()

        if (!TextUtils.isEmpty(result.translation)) {
            storeTranslation(
                sourceLanguage = sourceLanguage,
                targetLanguage = targetLanguage,
                text = text,
                translation = result.translation!!
            )
        }

        return result
    }

    private fun connect(): Result {
        val url = URL(BASE_URL)
        val connection = url.openConnection() as HttpURLConnection
        connection.requestMethod =
            REQUEST_METHOD
        connection.setRequestProperty(
            "content-type",
            CONTENT_TYPE
        )
        connection.connectTimeout =
            CONNECT_TIME_OUT
        connection.readTimeout =
            READ_TIME_OUT
        connection.defaultUseCaches = true
        connection.doOutput = true
        val outputStream = connection.outputStream
        writeOutputStream(outputStream)
        outputStream.close()
        val inputStream = connection.inputStream
        val result = readInputStream(inputStream)
        inputStream.close()
        connection.disconnect()

        return result
    }

    private fun writeOutputStream(outputStream: OutputStream) {

        val jsonObj = JSONObject()
        jsonObj.put("to", targetLanguage)
        jsonObj.put("text", text)
        jsonObj.put("k", key)
        if (sourceLanguage != null) {
            jsonObj.put("from", sourceLanguage)
        }

        val json = jsonObj.toString()
        outputStream.write(json.toByteArray())
        outputStream.flush()
    }

    private fun readInputStream(inputStream: InputStream): Result {

        val bytes = inputStream.readBytes()

        val content = String(bytes)

        return parserJson(content)
    }

    /**
     * Parser Content
     *
     * {"code":1000,"message":"success","result":{"text":"你好吗"}}
     *
     */
    private fun parserJson(json: String): Result {

        val jsonObject = org.json.JSONObject(json)

        val keyCode = "code"
        val keyMessage = "message"
        val keyResult = "result"
        val keyText = "text"

        var valueCode: Int = 0
        var valueMessage: String? = null
        var valueText: String? = null

        if (jsonObject.has(keyCode)) {
            valueCode = jsonObject.getInt(keyCode)
        }
        if (jsonObject.has(keyMessage)) {
            valueMessage = jsonObject.getString(keyMessage)
        }
        if (jsonObject.has(keyResult)) {
            val valueResult = jsonObject.getJSONObject(keyResult)
            valueText = valueResult.getString(keyText)
        }

        return Result(
            sourceLanguage = sourceLanguage,
            targetLanguage = targetLanguage,
            text = text,
            translation = valueText,
            valueCode = valueCode,
            valueMessage = valueMessage
        )
    }

    /* private fun storeTranslation(context: Context, preferKey: String, translation: String) {
         val prefer = context.getSharedPreferences(PREFER_TRANSLATOR, Context.MODE_PRIVATE)
         prefer.edit().putString(preferKey, translation).apply()
     }

     private fun readTranslation(context: Context, preferKey: String): String? {
         val prefer = context.getSharedPreferences(PREFER_TRANSLATOR, Context.MODE_PRIVATE)
         return prefer.getString(preferKey, null)
     }

     private fun generateKey(
         sourceLanguage: String?, targetLanguage: String, text: String
     ): String {
         val content = "$sourceLanguage-$targetLanguage-$text"
         //return Base64.getEncoder().encodeToString(content.toByteArray())
         return Base64.encodeToString(content.toByteArray(), Base64.DEFAULT)
     }*/

    private fun zip() {
        //val zipInputStream = ZipInputStream(ByteArrayInputStream(content.toByteArray()))
        /*val baos = ByteArrayOutputStream()
        val zipOutputStream = ZipOutputStream(baos)
        zipOutputStream.putNextEntry(ZipEntry("key"))
        zipOutputStream.write(content.toByteArray())
        zipOutputStream.closeEntry()
        zipOutputStream.flush()
        val zipString = String(baos.toByteArray())

        println("content=$content")
        println("zipString=$zipString")*/
    }


    companion object {
        //        http://tapi.translasion.com/ 测试
//        http://api.translasion.com/ 正式
        private const val BASE_URL = "https://api.translasion.com/v1/translate"
        private const val CONNECT_TIME_OUT = 6000
        private const val READ_TIME_OUT = 6000
        private const val REQUEST_METHOD = "POST"
        private const val CONTENT_TYPE = "application/json"
        private const val PREFER_TRANSLATOR = "prefer_translator_data"
    }
}