package com.tencent.imsdk.manager;

import android.content.Context;
import android.util.Log;

import com.tencent.imsdk.BaseConstants;
import com.tencent.imsdk.common.IMCallback;
import com.tencent.imsdk.common.IMContext;
import com.tencent.imsdk.common.IMLog;
import com.tencent.imsdk.common.NetworkInfoCenter;
import com.tencent.imsdk.common.SystemUtil;
import com.tencent.imsdk.conversation.ConversationManager;
import com.tencent.imsdk.group.GroupManager;
import com.tencent.imsdk.message.MessageCenter;
import com.tencent.imsdk.relationship.RelationshipManager;
import com.tencent.imsdk.relationship.UserInfo;
import com.tencent.imsdk.relationship.UserStatus;
import com.tencent.imsdk.signaling.SignalingManager;

import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

public class BaseManager implements NetworkInfoCenter.INetworkChangeListener {
    private static final String TAG = BaseManager.class.getSimpleName();

    public static final int TUI_COMPONENT_UNKNOWN = 0;
    public static final int TUI_COMPONENT_CORE = 1;
    public static final int TUI_COMPONENT_CONVERSATION = 2;
    public static final int TUI_COMPONENT_CHAT = 3;
    public static final int TUI_COMPONENT_CONTACT = 4;
    public static final int TUI_COMPONENT_GROUP = 5;
    public static final int TUI_COMPONENT_SEARCH = 6;
    public static final int TUI_COMPONENT_OFFLINEPUSH = 7;
    public static final int TUI_COMPONENT_COMMUNITY = 8;

    // 检查 TUI 组件的最大次数限制
    private static final int TUI_COMPONENT_CHECK_COUNT_LIMIT = 5;
    // 检查 TUI 组件的堆栈遍历深度
    private static final int TUI_COMPONENT_STACK_LAYER_LIMIT = 10;

    // UIPlatform
    private static final int UI_PLATFORM_UNKNOWN = 0;
    private static final int UI_PLATFORM_FLUTTER = 1;
    private static final int UI_PLATFORM_FLUTTER_UIKIT = 2;
    private static final int UI_PLATFORM_UNITY = 5;
    private static final int UI_PLATFORM_UNITY_UIKIT = 6;
    private static final int UI_PLATFORM_TUIKIT = 15;

    private boolean mInvokeFromTUIKit = false;
    private boolean mInvokeFromTUICore = false;

    List<Integer> mTUIComponentList = new ArrayList<>();
    private HashMap<Integer, Integer> mTUIComponentCheckCountMap = new HashMap<>();

    private SDKConfig.NetworkInfo mLastNetworkInfo = new SDKConfig.NetworkInfo();

    private SDKConfig.ProxyInfo mProxyInfo = new SDKConfig.ProxyInfo();

    private SDKConfig.DatabaseEncryptInfo mDatabaseEncryptInfo = new SDKConfig.DatabaseEncryptInfo();

    private SDKConfig.PacketRetryInfo mPacketRetryInfo = new SDKConfig.PacketRetryInfo();

    private WeakReference<SDKListener> sdkListenerWeakReference;

    private static boolean mLoadLibrarySuccess = false;
    static {
        try {
            mLoadLibrarySuccess = SystemUtil.loadIMLibrary();
        } catch (Exception e) {
            Log.e(TAG, e.toString());
        }
    }

    private String mStringUIPlatform = "";
    private int mNumberUIPlatform = 0;

    private boolean isInit = false;
    private boolean isTestEnvironment = false;
    private boolean forceUseQuicChannel = false;
    private boolean isIPv6Prior = false;

    private UserPreference userPreference = new UserPreference();

    private static class BaseManagerHolder {
        private static final BaseManager baseManager = new BaseManager();
    }

    public static BaseManager getInstance() {
        return BaseManager.BaseManagerHolder.baseManager;
    }


    public boolean initSDK(Context context, SDKConfig sdkConfig, boolean needLogCallback, SDKListener listener) {
        if (false == mLoadLibrarySuccess) {
            IMLog.e(TAG, "libimsdk.so is not loaded");
            return false;
        }

        if (sdkConfig.sdkAppId <= 0) {
            IMLog.e(TAG, "invalid sdkAppID:" + sdkConfig.sdkAppId);
            return false;
        }

        if (null == context) {
            IMLog.e(TAG, "null context");
            return false;
        }

        if (isInit) {
            IMLog.w(TAG, "Has initSDK");
            return true;
        }

        IMContext.getInstance().init(context.getApplicationContext());
        NetworkInfoCenter.getInstance().init(context.getApplicationContext(), this);
        MessageCenter.getInstance().init();
        GroupManager.getInstance().init();
        ConversationManager.getInstance().init();
        RelationshipManager.getInstance().init();
        SignalingManager.getInstance().init();

        sdkConfig.sdkInitPath = SystemUtil.getSDKInitPath();
        sdkConfig.sdkInstanceType = SystemUtil.getInstanceType();
        sdkConfig.isTestEnvironment = isTestEnvironment;
        sdkConfig.forceUseQuicChannel = forceUseQuicChannel;
        sdkConfig.isIPv6Prior = isIPv6Prior;
        sdkConfig.deviceInfo.deviceType = SystemUtil.getDeviceType();
        sdkConfig.deviceInfo.deviceId = SystemUtil.getDeviceID();
        sdkConfig.deviceInfo.deviceBrand = SystemUtil.getInstanceType();
        sdkConfig.deviceInfo.systemVersion = SystemUtil.getSystemVersion();
        sdkConfig.networkInfo.networkType = NetworkInfoCenter.getInstance().getNetworkType();
        sdkConfig.networkInfo.ipType = NetworkInfoCenter.getInstance().getIPType();
        sdkConfig.networkInfo.networkId = NetworkInfoCenter.getInstance().getNetworkID();
        sdkConfig.networkInfo.wifiNetworkHandle = NetworkInfoCenter.getInstance().getWifiNetworkHandle();
        sdkConfig.networkInfo.xgNetworkHandle = NetworkInfoCenter.getInstance().getXgNetworkHandle();
        sdkConfig.networkInfo.networkConnected = NetworkInfoCenter.getInstance().isNetworkConnected();
        sdkConfig.proxyInfo = mProxyInfo;
        sdkConfig.databaseEncryptInfo = mDatabaseEncryptInfo;
        sdkConfig.packetRetryInfo = mPacketRetryInfo;
        sdkConfig.logSetting.enableConsoleLog = true;
        sdkConfig.logSetting.logFilePath = SystemUtil.getSDKLogPath();
        sdkConfig.stringUIPlatform = mStringUIPlatform;
        sdkConfig.numberUIPlatform = getUIPlatform();

        mLastNetworkInfo = sdkConfig.networkInfo;

        nativeInitSDK(sdkConfig, needLogCallback, listener);

        sdkListenerWeakReference = new WeakReference<>(listener);

        reportTUIComponentUsage();

        isInit = true;
        return true;
    }

    public void unInitSDK() {
        nativeUninitSDK();
        mStringUIPlatform = "";
        mNumberUIPlatform = 0;
        isInit = false;
        isTestEnvironment = false;
        forceUseQuicChannel = false;
        isIPv6Prior = false;
        mLastNetworkInfo.clean();
        mProxyInfo.clean();
        mDatabaseEncryptInfo.clean();
        mPacketRetryInfo.clean();
        mInvokeFromTUIKit = false;
        mInvokeFromTUICore = false;
        mTUIComponentList.clear();
        mTUIComponentCheckCountMap.clear();
    }

    private int getUIPlatform() {
        if (mNumberUIPlatform != UI_PLATFORM_UNKNOWN) {
            return mNumberUIPlatform;
        }

        mInvokeFromTUIKit = isTUIKit();
        boolean has_flutter = isFlutter();
        boolean has_unity = isUnity();
        if (has_flutter) {
            if (mInvokeFromTUIKit) {
                return UI_PLATFORM_FLUTTER_UIKIT;
            } else {
                return UI_PLATFORM_FLUTTER;
            }
        } else if (has_unity) {
            if (mInvokeFromTUIKit) {
                return UI_PLATFORM_UNITY_UIKIT;
            } else {
                return UI_PLATFORM_UNITY;
            }
        } else if (mInvokeFromTUIKit) {
            return UI_PLATFORM_TUIKIT;
        }

        return UI_PLATFORM_UNKNOWN;
    }

    private boolean isTUIKit() {
        try {
            Class classTUIKit = Class.forName("com.tencent.qcloud.tim.uikit.TUIKit");
            if (classTUIKit != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 TUIKit
        }

        try {
            Class classTUICore = Class.forName("com.tencent.qcloud.tuicore.TUICore");
            if (classTUICore != null) {
                return true;
            }
        } catch (ClassNotFoundException e) {
            // 不含 TUIKit
        }

        // 如果包名被修改，再判断下调用栈的是否包含类的关键字
        StackTraceElement[] stacks = Thread.currentThread().getStackTrace();
        String callName = "";
        for (int i = 0; i < stacks.length; i++) {
            if (i > 15) {
                return false;
            }
            callName = stacks[i].getClassName();
            String lowerCaseCallName = callName.toLowerCase();
            if (lowerCaseCallName.contains("tuikitimpl") || lowerCaseCallName.contains("tuicore")) {
                return true;
            }
        }

        return false;
    }

    private boolean isTUICore() {
        try {
            Class classTUICore = Class.forName("com.tencent.qcloud.tuicore.TUICore");
            if (classTUICore != null) {
                return true;
            }
        } catch (ClassNotFoundException e) {
            // 不含 TUIKit
        }

        // 如果包名被修改，再判断下调用栈的是否包含类的关键字
        StackTraceElement[] stacks = Thread.currentThread().getStackTrace();
        String callName = "";
        for (int i = 0; i < stacks.length; i++) {
            if (i > 15) {
                return false;
            }
            callName = stacks[i].getClassName();
            String lowerCaseCallName = callName.toLowerCase();
            if (lowerCaseCallName.contains("tuicore")) {
                return true;
            }
        }

        return false;
    }

    private boolean isFlutter() {
        try {
            Class c = Class.forName("com.qq.qcloud.tencent_im_sdk_plugin.tencent_im_sdk_plugin");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 Flutter
        }
        return false;
    }

    private boolean isUnity() {
        try {
            Class c = Class.forName("com.qcloud.tencentimsdk.TencentImSDKPluginUnity");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 Unity
        }
        return false;
    }

    public void checkTUIComponent(int componentType) {
        if (!isInit) {
            return;
        }
        if (false == mInvokeFromTUICore) {
            return;
        }

        if (mTUIComponentList.contains(componentType)) {
            return;
        }

        if (!mTUIComponentCheckCountMap.containsKey(componentType)) {
            return;
        }

        String keyCallName = "";
        if (componentType == TUI_COMPONENT_CONVERSATION) {
            keyCallName = "conversationprovider";
        } else if (componentType == TUI_COMPONENT_CHAT) {
            keyCallName = "chatprovider";
        } else if (componentType == TUI_COMPONENT_GROUP) {
            keyCallName = "groupInfoprovider";
        } else if (componentType == TUI_COMPONENT_CONTACT) {
            keyCallName = "contactprovider";
        } else if (componentType == TUI_COMPONENT_SEARCH) {
            keyCallName = "searchdataprovider";
        } else if (componentType == TUI_COMPONENT_OFFLINEPUSH) {
            keyCallName = "tuiofflinepushmanager";
        } else if (componentType == TUI_COMPONENT_COMMUNITY) {
            keyCallName = "communityprovider";
        } else {
            IMLog.e(TAG, "unknown tui component type:" + componentType);
            return;
        }

        int checkIndex = mTUIComponentCheckCountMap.get(componentType);
        if (checkIndex < TUI_COMPONENT_CHECK_COUNT_LIMIT) {
            checkIndex++;
            mTUIComponentCheckCountMap.put(componentType, checkIndex);

            StackTraceElement[] stacks = Thread.currentThread().getStackTrace();
            String callName = "";
            // 限制调用栈查找层级
            for (int i = 0; i < stacks.length; i++) {
                if (i >= TUI_COMPONENT_STACK_LAYER_LIMIT) {
                    return;
                }

                callName = stacks[i].getClassName();
                String lowerCaseCallName = callName.toLowerCase();
                if (lowerCaseCallName.contains(keyCallName)) {
                    List<Integer> tuiComponentList = new ArrayList<>();
                    tuiComponentList.add(componentType);
                    nativeReportTUIComponentUsage(tuiComponentList);

                    mTUIComponentList.add(componentType);
                    break;
                }
            }
        }
    }

    private void reportTUIComponentUsage() {
        mInvokeFromTUICore = isTUICore();

        mTUIComponentCheckCountMap.put(TUI_COMPONENT_CONVERSATION, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_CHAT, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_CONTACT, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_GROUP, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_SEARCH, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_OFFLINEPUSH, 0);
        mTUIComponentCheckCountMap.put(TUI_COMPONENT_COMMUNITY, 0);

        if (mInvokeFromTUIKit) {
            if (mInvokeFromTUICore) {
                mTUIComponentList.add(TUI_COMPONENT_CORE);
                if (hasTUIConversation() && !mTUIComponentList.contains(TUI_COMPONENT_CONVERSATION)) {
                    mTUIComponentList.add(TUI_COMPONENT_CONVERSATION);
                }

                if (hasTUIChat() && !mTUIComponentList.contains(TUI_COMPONENT_CHAT)) {
                    mTUIComponentList.add(TUI_COMPONENT_CHAT);
                }

                if (hasTUIContact() && !mTUIComponentList.contains(TUI_COMPONENT_CONTACT)) {
                    mTUIComponentList.add(TUI_COMPONENT_CONTACT);
                }

                if (hasTUIGroup() && !mTUIComponentList.contains(TUI_COMPONENT_GROUP)) {
                    mTUIComponentList.add(TUI_COMPONENT_GROUP);
                }

                if (hasTUISearch() && !mTUIComponentList.contains(TUI_COMPONENT_SEARCH)) {
                    mTUIComponentList.add(TUI_COMPONENT_SEARCH);
                }

                if (hasTUIOfflinePush() && !mTUIComponentList.contains(TUI_COMPONENT_OFFLINEPUSH)) {
                    mTUIComponentList.add(TUI_COMPONENT_OFFLINEPUSH);
                }

                if (hasTUICommunity() && !mTUIComponentList.contains(TUI_COMPONENT_COMMUNITY)) {
                    mTUIComponentList.add(TUI_COMPONENT_COMMUNITY);
                }
            } else {
                mTUIComponentList.add(TUI_COMPONENT_UNKNOWN);
            }

            nativeReportTUIComponentUsage(mTUIComponentList);
        }
    }

    private boolean hasTUIConversation() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuiconversation.model.ConversationProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 conversation
        }
        return false;
    }

    private boolean hasTUIChat() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuichat.model.ChatProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 chat
        }
        return false;
    }

    private boolean hasTUIContact() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuicontact.model.ContactProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 contact
        }
        return false;
    }

    private boolean hasTUIGroup() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuigroup.model.GroupInfoProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 group
        }
        return false;
    }

    private boolean hasTUISearch() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuisearch.model.SearchDataProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 search
        }
        return false;
    }

    private boolean hasTUIOfflinePush() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tim.tuiofflinepush.TUIOfflinePushManager");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 offlinepush
        }
        return false;
    }

    private boolean hasTUICommunity() {
        try {
            Class c = Class.forName("com.tencent.qcloud.tuikit.tuicommunity.model.CommunityProvider");
            if (c != null) {
                return true;
            }
        } catch (Exception e) {
            // 不含 community
        }
        return false;
    }

    public void login(final String userID, final String userSig, final IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeLogin(userID, userSig, callback);
    }

    public void logout(final IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        nativeLogout(callback);
    }

    public boolean setLibraryPath(String libraryPath) {
        mLoadLibrarySuccess = SystemUtil.loadIMLibrary(libraryPath);
        return mLoadLibrarySuccess;
    }

    public void setCustomUIPlatform(String stringUIPlatform, int numberUIPlatform) {
        mStringUIPlatform = stringUIPlatform;
        mNumberUIPlatform = numberUIPlatform;
    }

    public void setTestEnvironment(boolean testEnvironment) {
        isTestEnvironment = testEnvironment;
    }

    public void setForceUseQuicChannel(boolean force) {
        forceUseQuicChannel = force;
    }

    public void setIPv6Prior(boolean prior) {
        isIPv6Prior = prior;
    }

    public void setCustomServerInfo(CustomServerInfo customServerInfo) {
        nativeSetCustomServerInfo(customServerInfo);
    }

    public void setProxyInfo(SDKConfig.ProxyInfo proxyInfo) {
        mProxyInfo = proxyInfo;
    }

    public void setDatabaseEncryptInfo(SDKConfig.DatabaseEncryptInfo databaseEncryptInfo) {
        mDatabaseEncryptInfo = databaseEncryptInfo;
    }

    public void isCommercialAbilityEnabled(long commercialAbilityKey, IMCallback<Object> callback) {
        nativeIsCommercialAbilityEnabled(commercialAbilityKey, callback);
    }

    public void setPacketRetryInfo(SDKConfig.PacketRetryInfo info) {
        mPacketRetryInfo = info;
    }

    public String getLoginUser() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return null;
        }
        return nativeGetLoginUser();
    }

    public int getLoginStatus() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 3; // 无登录状态
        }
        return nativeGetLoginStatus();
    }

    public String getVersion() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return null;
        }
        return nativeGetSDKVersion();
    }

    public boolean isInited() {
        return isInit;
    }

    public long getClockTickInHz() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 0;
        }
        return nativeGetClockTickInHz();
    }

    public long getTimeTick() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 0;
        }
        return nativeGetTimeTick();
    }

    public long getServerTime() {
        if (!isInit) {
            Log.e(TAG, "sdk not init");
            return 0;
        }
        return nativeGetServerTime();
    }

    public void initLocalStorage(String userID, IMCallback callback) {
        if (!isInit) {
            if (callback != null) {
                callback.fail(BaseConstants.ERR_SDK_NOT_INITIALIZED, "sdk not init");
            }
            return;
        }
        
        nativeInitLocalStorage(userID, callback);
    }

    @Override
    public void onNetworkChange(boolean connected, int networkType, int ipType, String networkID,
                                long wifiNetworkHandle, long xgNetworkHandle) {
        if (connected == mLastNetworkInfo.networkConnected
                && networkType == mLastNetworkInfo.networkType
                && ipType == mLastNetworkInfo.ipType
                && (null != networkID && networkID.equals(mLastNetworkInfo.networkId))) {
            Log.w(TAG, "onNetworkChange, networkinfo is same");
            return;
        }

        mLastNetworkInfo.networkConnected = connected;
        mLastNetworkInfo.networkType = networkType;
        mLastNetworkInfo.ipType = ipType;
        mLastNetworkInfo.networkId = networkID;
        nativeNotifyNetworkChange(connected, networkType, ipType, networkID,
                wifiNetworkHandle, xgNetworkHandle);
    }

    public void notifySelfInfoUpdated(UserInfo selfInfo) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onSelfInfoUpdated(selfInfo);
            }
        }
    }

    public void notifyUserStatusChanged(List<UserStatus> userStatusList) {
        if (sdkListenerWeakReference != null) {
            SDKListener listener = sdkListenerWeakReference.get();
            if (listener != null) {
                listener.onUserStatusChanged(userStatusList);
            }
        }
    }

    public void enableSignaling(boolean enable) {
        userPreference.setEnableSignaling(enable);
        nativeSetUserPreference(userPreference);
    }

    public long getAudioPermission() {
        return nativeGetAudioPermission();
    }

    //////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

    protected native void nativeInitSDK(SDKConfig sdkConfig, boolean needLogCallback, SDKListener sdkListener);

    protected native void nativeUninitSDK();

    protected native void nativeNotifyNetworkChange(boolean connected, int networkType, int ipType, String networkID, long wifiNetworkHandle, long xgNetworkHandle);

    protected native void nativeLogin(String identifier, String userSig, IMCallback callBack);

    protected native void nativeLogout(IMCallback callBack);

    protected native void nativeSetCustomServerInfo(CustomServerInfo customServerInfo);

    protected native void nativeIsCommercialAbilityEnabled(long commercialAbilityKey, IMCallback callBack);

    protected native String nativeGetLoginUser();

    protected native int nativeGetLoginStatus();

    protected native String nativeGetSDKVersion();

    protected native long nativeGetClockTickInHz();

    protected native long nativeGetTimeTick();

    protected native long nativeGetServerTime();

    protected native void nativeInitLocalStorage(String identifier, IMCallback callBack);

    protected native void nativeReportTUIComponentUsage(List<Integer> tuiComponentSet);

    protected native void nativeSetUserPreference(UserPreference userPreference);

    protected native long nativeGetAudioPermission();
}
