package com.transsin.networkmonitor

import android.os.SystemClock
import android.util.Log
import com.transsin.networkmonitor.Consts.FAILURE
import com.transsin.networkmonitor.Consts.SUCCESS
import com.transsin.networkmonitor.ErrorCode.CODE_CONNECT_ERROR
import com.transsin.networkmonitor.ErrorCode.CODE_FAKE_NETWORK
import com.transsin.networkmonitor.ErrorCode.CODE_NO_NETWORK
import com.transsin.networkmonitor.ErrorCode.CODE_NO_ROUTE_TO_HOST
import com.transsin.networkmonitor.ErrorCode.CODE_PROTOCOL_ERROR
import com.transsin.networkmonitor.ErrorCode.CODE_SOCKET_TIME_OUT
import com.transsin.networkmonitor.ErrorCode.CODE_SSL_HANDSHAKE_ERROR
import com.transsin.networkmonitor.ErrorCode.CODE_UNKNOWN
import com.transsin.networkmonitor.ErrorCode.CODE_UNKNOWN_HOST
import com.transsin.networkmonitor.ErrorCode.MSG_FAKE_NETWORK
import com.transsin.networkmonitor.ErrorCode.MSG_NETWORK_DISCONNECT
import com.transsin.networkmonitor.ErrorCode.MSG_UNKNOWN
import okhttp3.*
import java.io.IOException
import java.lang.Long.max
import java.net.*
import javax.net.ssl.SSLHandshakeException
import kotlin.random.Random

/**
 * @Author:huangxiushuo
 * @Date: 2022-07-22 17:01
 * @Description:
 */
internal class MonitorEventListener(
    private val listener: EventListener?,
    private val usage: Int,
    shouldUploadData: Boolean,
    channel: String? = "All"
) : EventListener() {

    private var mCallStartTime = 0L
    private var mDnsStartTime = 0L
    private var mSecureConnectStartTime = 0L
    private var mTcpStartTime = 0L
    private var mReqHeadersStartTime = 0L
    private var mReqHeadersEndTime = 0L
    private var mReqBodyEndTime = 0L
    private var mRespHeadersStartTime = 0L
    private val mMonitorData = MonitorData.create(usage, shouldUploadData, channel)
    private var mRespHeadersEndTime = 0L
    private var httpCode = CODE_UNKNOWN
    private var errorCode = CODE_UNKNOWN
    private var errorMsg = MSG_UNKNOWN
    private val randomNumber = (1..10).random()

    override fun callStart(call: Call) {
        super.callStart(call)
        "[callStart]".printEvent()
        try {
            val isDownload = call.request().headers().get("isDownload")
            isDownload?.let {
                mMonitorData.usage = if (it == "true") 1 else 0
            }
            val offlineAd = call.request().headers().get("offlineAd")
            offlineAd?.let { mMonitorData.offlineAd = it.toInt() }
            val host = call.request().url().host()
            mMonitorData.host = host ?: ""
            val url = (call.request().url().toString())
            mMonitorData.completeApi = if (randomNumber <= 2) {
                url
            } else {
                ""
            }
            mMonitorData.serverApi = if (usage == 0 && url.contains("?")) {
                url.substring(0, url.indexOf("?"))
            } else url
        } catch (e: Exception) {
            "[get header exception] ${e::class.java.simpleName} : ${e.message}".printErrEvent()
        }
        mCallStartTime = realtime()
        listener?.callStart(call)
    }


    override fun dnsStart(call: Call, domainName: String) {
        super.dnsStart(call, domainName)
        "[dnsStart] domainName: $domainName".printEvent()
        mDnsStartTime = realtime()
        listener?.dnsStart(call, domainName)
    }

    override fun dnsEnd(call: Call, domainName: String, inetAddressList: List<InetAddress>) {
        super.dnsEnd(call, domainName, inetAddressList)
        val addresses = inetAddressList.joinToString(
            separator = ",",
            prefix = "[",
            postfix = "]"
        )
        "[dnsEnd] inetAddressList $addresses".printEvent()
        mMonitorData.dnsTime = realtimeDiff(mDnsStartTime)
        listener?.dnsEnd(call, domainName, inetAddressList)
    }

    override fun connectStart(call: Call, inetSocketAddress: InetSocketAddress, proxy: Proxy) {
        super.connectStart(call, inetSocketAddress, proxy)
        mTcpStartTime = realtime()
        mMonitorData.ipAddr = inetSocketAddress.address.hostAddress.orEmpty()
        "[connectStart] inetSocketAddress : $inetSocketAddress, proxy : $proxy".printEvent()
        listener?.connectStart(call, inetSocketAddress, proxy)
    }

    override fun secureConnectStart(call: Call) {
        super.secureConnectStart(call)
        "[secureConnectStart]".printEvent()
        mSecureConnectStartTime = realtime()
        listener?.secureConnectStart(call)
    }

    override fun secureConnectEnd(call: Call, handshake: Handshake?) {
        super.secureConnectEnd(call, handshake)
        "[secureConnectEnd]".printEvent()
        mMonitorData.sslTime = realtimeDiff(mSecureConnectStartTime)
        listener?.secureConnectEnd(call, handshake)
    }

    override fun connectEnd(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?
    ) {
        super.connectEnd(call, inetSocketAddress, proxy, protocol)
        "[connectEnd] inetSocketAddress $inetSocketAddress".printEvent()

        mMonitorData.tcpTime = realtimeDiff(mTcpStartTime)
        listener?.connectEnd(call, inetSocketAddress, proxy, protocol)
    }

    override fun connectFailed(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?,
        ioe: IOException
    ) {
        super.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
        "[connectFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        listener?.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
    }

    override fun connectionReleased(call: Call, connection: Connection) {
        super.connectionReleased(call, connection)
        "[connectionReleased]".printEvent()
        listener?.connectionReleased(call, connection)
    }

    override fun connectionAcquired(call: Call, connection: Connection) {
        super.connectionAcquired(call, connection)
        "[connectionAcquired] connection: ${connection}".printEvent()
        listener?.connectionAcquired(call, connection)
    }

    override fun requestHeadersStart(call: Call) {
        super.requestHeadersStart(call)
        "[requestHeadersStart]".printEvent()
        mReqHeadersStartTime = realtime()
        listener?.requestHeadersStart(call)
    }

    override fun requestHeadersEnd(call: Call, request: Request) {
        super.requestHeadersEnd(call, request)
        "[requestHeadersEnd] request :${request}".printEvent()
        mReqHeadersEndTime = realtime()
        mMonitorData.sendTime = mReqHeadersEndTime - mReqHeadersStartTime
        listener?.requestHeadersEnd(call, request)
    }

    //may not call
    override fun requestBodyStart(call: Call) {
        super.requestBodyStart(call)
        "[requestBodyStart]".printEvent()
        listener?.requestBodyStart(call)
    }

    //may not call
    override fun requestBodyEnd(call: Call, byteCount: Long) {
        super.requestBodyEnd(call, byteCount)
        "[requestBodyEnd] byteCount: ${byteCount}".printEvent()
        mReqBodyEndTime = realtime()
        mMonitorData.run {
            reqBodySize = byteCount
            sendTime = mReqBodyEndTime - mReqHeadersStartTime
        }
        listener?.requestBodyEnd(call, byteCount)
    }

    override fun requestFailed(call: Call, ioe: IOException) {
        super.requestFailed(call, ioe)
        "[requestFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        listener?.requestFailed(call, ioe)
    }

    override fun responseHeadersStart(call: Call) {
        super.responseHeadersStart(call)
        "[responseHeadersStart]".printEvent()
        mRespHeadersStartTime = realtime()
        mMonitorData.waitTime = mRespHeadersStartTime - max(mReqBodyEndTime, mReqHeadersEndTime)
        listener?.responseHeadersStart(call)
    }

    override fun responseHeadersEnd(call: Call, response: Response) {
        super.responseHeadersEnd(call, response)
        "[responseHeadersEnd] code : ${response.code()}".printEvent()
        val provider = response.headers().get("x-response-cdn") ?: ""
        provider.printEvent()
        mMonitorData.cdnProvider = provider
        mRespHeadersEndTime = realtime()
        mMonitorData.recTime = realtimeDiff(mRespHeadersStartTime)
        mMonitorData.cdnReqTime = realtimeDiff(mCallStartTime)
        httpCode = response.code()
        listener?.responseHeadersEnd(call, response)
    }

    //may not call
    override fun responseBodyStart(call: Call) {
        super.responseBodyStart(call)
        "[responseBodyStart]".printEvent()
        listener?.responseBodyStart(call)
    }

    //may not call
    override fun responseBodyEnd(call: Call, byteCount: Long) {
        super.responseBodyEnd(call, byteCount)
        "[responseBodyEnd] byteCount: ${byteCount}".printEvent()
        mMonitorData.run {
            resBodySize = byteCount
            recTime = realtimeDiff(mRespHeadersStartTime)
            cdnDownTime = realtimeDiff(mRespHeadersEndTime)
            cdnSpeed = "${(byteCount / 1024f) / (cdnDownTime / 1000f)}"
        }
        listener?.responseBodyEnd(call, byteCount)
    }

    override fun responseFailed(call: Call, ioe: IOException) {
        super.responseFailed(call, ioe)
        "[responseFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        listener?.responseFailed(call, ioe)
    }

    override fun callEnd(call: Call) {
        super.callEnd(call)
        "[callEnd]".printEvent()
        mMonitorData.run {
            callResultCode = SUCCESS
            reqTime = realtimeDiff(mCallStartTime)
            errorCode = httpCode
            //请求成功  1. errorCode = -1，命中缓存 2. 304命中缓存
            if (errorCode != CODE_UNKNOWN && errorCode != 304 && !mMonitorData.shouldUploadData) {
                track()
            }
        }
        listener?.callEnd(call)
    }

    /**
     * [IOException]
     * -[UnknownHostException]
     * -[SSLException]
     * --[SSLHandshakeException]
     * -[SocketTimeoutException]
     */
    override fun callFailed(call: Call, ioe: IOException) {
        super.callFailed(call, ioe)
        "[callFailed]  ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        mMonitorData.callResultCode = FAILURE
        if (isNetDone()) {
            // 无网
            if (!NetworkMonitor.isNetworkConnected) {
                errorCode = CODE_NO_NETWORK
                errorMsg = MSG_NETWORK_DISCONNECT
            }
            // 假网
            else if (NetworkMonitor.isFakeNetwork) {
                errorCode = CODE_FAKE_NETWORK
                errorMsg = MSG_FAKE_NETWORK
            } else {
                errorCode = fetchCode(ioe)
            }
        } else {
            errorCode = fetchCode(ioe)
        }


        mMonitorData.errorCode = errorCode
        mMonitorData.errorMsg = if (errorMsg == MSG_UNKNOWN) ioe.shortMsg() else errorMsg
        if (!mMonitorData.shouldUploadData) {
            mMonitorData.track()
        }
        listener?.callFailed(call, ioe)
    }

    private fun isNetDone(): Boolean {
        if (NetworkMonitor.isNetInitDone || SystemClock.uptimeMillis() - NetworkMonitor.initStartTime > 3000) {
            return true
        }
        return false
    }

    private fun fetchCode(ioe: IOException): Int {
        var code = httpCode
        if (httpCode == CODE_UNKNOWN) {
            code = when(ioe) {
                is SocketTimeoutException -> CODE_SOCKET_TIME_OUT
                is UnknownHostException -> CODE_UNKNOWN_HOST
                is NoRouteToHostException -> CODE_NO_ROUTE_TO_HOST
                is ProtocolException -> CODE_PROTOCOL_ERROR
                is SSLHandshakeException -> CODE_SSL_HANDSHAKE_ERROR
                is ConnectException -> CODE_CONNECT_ERROR
                else -> CODE_UNKNOWN
            }
        }
        return code
    }
}