package com.transsin.networkmonitor

import android.util.Log
import com.transsin.networkmonitor.Consts.FAILURE
import com.transsin.networkmonitor.Consts.SUCCESS
import com.transsin.networkmonitor.ErrorCode.CODE_CALL_FAILED
import com.transsin.networkmonitor.ErrorCode.CODE_FAKE_NETWORK
import com.transsin.networkmonitor.ErrorCode.CODE_UNKNOWN
import com.transsin.networkmonitor.ErrorCode.MSG_FAKE_NETWORK
import com.transsin.networkmonitor.ErrorCode.MSG_UNKNOWN
import okhttp3.*
import java.io.IOException
import java.lang.Long.max
import java.net.InetAddress
import java.net.InetSocketAddress
import java.net.Proxy

/**
 * @Author:huangxiushuo
 * @Date: 2022-07-22 17:01
 * @Description:
 */
internal class MonitorEventListener(
    private val listener: EventListener?,
    private val usage: Int,
    private val isTest: Boolean
) : EventListener() {

    private var mCallStartTime = 0L
    private var mDnsStartTime = 0L
    private var mSecureConnectStartTime = 0L
    private var mTcpStartTime = 0L
    private var mReqHeadersStartTime = 0L
    private var mReqHeadersEndTime = 0L
    private var mReqBodyEndTime = 0L
    private var mRespHeadersStartTime = 0L
    private val mMonitorData = MonitorData.create(usage, isTest)
    private var mRespHeadersEndTime = 0L
    private var httpCode = CODE_UNKNOWN
    private var errorCode = CODE_UNKNOWN
    private var errorMsg = MSG_UNKNOWN

    override fun callStart(call: Call) {
        super.callStart(call)
        "[callStart]".printEvent()
        mMonitorData.serverApi = (call.request().url().toString())
        mCallStartTime = realtime()
        listener?.callStart(call)
    }


    override fun dnsStart(call: Call, domainName: String) {
        super.dnsStart(call, domainName)
        "[dnsStart] domainName: $domainName".printEvent()
        mDnsStartTime = realtime()
        listener?.dnsStart(call, domainName)
    }

    override fun dnsEnd(call: Call, domainName: String, inetAddressList: List<InetAddress>) {
        super.dnsEnd(call, domainName, inetAddressList)
        val addresses = inetAddressList.joinToString(
            separator = ",",
            prefix = "[",
            postfix = "]"
        )
        "[dnsEnd] inetAddressList $addresses".printEvent()
        mMonitorData.dnsTime = realtimeDiff(mDnsStartTime)
        listener?.dnsEnd(call, domainName, inetAddressList)
    }

    override fun connectStart(call: Call, inetSocketAddress: InetSocketAddress, proxy: Proxy) {
        super.connectStart(call, inetSocketAddress, proxy)
        mTcpStartTime = realtime()
        mMonitorData.ipAddr = inetSocketAddress.address.hostAddress.orEmpty()
        "[connectStart] inetSocketAddress : $inetSocketAddress, proxy : $proxy".printEvent()
        listener?.connectStart(call, inetSocketAddress, proxy)
    }

    override fun secureConnectStart(call: Call) {
        super.secureConnectStart(call)
        "[secureConnectStart]".printEvent()
        mSecureConnectStartTime = realtime()
        listener?.secureConnectStart(call)
    }

    override fun secureConnectEnd(call: Call, handshake: Handshake?) {
        super.secureConnectEnd(call, handshake)
        "[secureConnectEnd]".printEvent()
        mMonitorData.sslTime = realtimeDiff(mSecureConnectStartTime)
        listener?.secureConnectEnd(call, handshake)
    }

    override fun connectEnd(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?
    ) {
        super.connectEnd(call, inetSocketAddress, proxy, protocol)
        "[connectEnd] inetSocketAddress $inetSocketAddress".printEvent()

        mMonitorData.tcpTime = realtimeDiff(mTcpStartTime)
        listener?.connectEnd(call, inetSocketAddress, proxy, protocol)
    }

    override fun connectFailed(
        call: Call,
        inetSocketAddress: InetSocketAddress,
        proxy: Proxy,
        protocol: Protocol?,
        ioe: IOException
    ) {
        super.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
        "[connectFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        errorCode = ErrorCode.CODE_CONNECT_FAILED
        errorMsg = ioe.shortMsg()
        listener?.connectFailed(call, inetSocketAddress, proxy, protocol, ioe)
    }

    override fun connectionReleased(call: Call, connection: Connection) {
        super.connectionReleased(call, connection)
        "[connectionReleased]".printEvent()
        listener?.connectionReleased(call, connection)
    }

    override fun connectionAcquired(call: Call, connection: Connection) {
        super.connectionAcquired(call, connection)
        "[connectionAcquired] connection: ${connection}".printEvent()
        listener?.connectionAcquired(call, connection)
    }

    override fun requestHeadersStart(call: Call) {
        super.requestHeadersStart(call)
        "[requestHeadersStart]".printEvent()
        mReqHeadersStartTime = realtime()
        listener?.requestHeadersStart(call)
    }

    override fun requestHeadersEnd(call: Call, request: Request) {
        super.requestHeadersEnd(call, request)
        "[requestHeadersEnd] request :${request}".printEvent()
        val isDownload = request.headers().get("isDownload")
        isDownload?.let {
            mMonitorData.usage = if (it == "true") 1 else 0
        }
        mMonitorData.host = request.headers().get("Host") ?: ""
        mReqHeadersEndTime = realtime()
        mMonitorData.sendTime = mReqHeadersEndTime - mReqHeadersStartTime
        listener?.requestHeadersEnd(call, request)
    }

    //may not call
    override fun requestBodyStart(call: Call) {
        super.requestBodyStart(call)
        "[requestBodyStart]".printEvent()
        listener?.requestBodyStart(call)
    }

    //may not call
    override fun requestBodyEnd(call: Call, byteCount: Long) {
        super.requestBodyEnd(call, byteCount)
        "[requestBodyEnd] byteCount: ${byteCount}".printEvent()
        mReqBodyEndTime = realtime()
        mMonitorData.run {
            reqBodySize = byteCount
            sendTime = mReqBodyEndTime - mReqHeadersStartTime
        }
        listener?.requestBodyEnd(call, byteCount)
    }

    override fun requestFailed(call: Call, ioe: IOException) {
        super.requestFailed(call, ioe)
        "[requestFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        errorCode = ErrorCode.CODE_REQUEST_FAILED
        errorMsg = ioe.shortMsg()
        listener?.requestFailed(call, ioe)
    }

    override fun responseHeadersStart(call: Call) {
        super.responseHeadersStart(call)
        "[responseHeadersStart]".printEvent()
        mRespHeadersStartTime = realtime()
        mMonitorData.waitTime = mRespHeadersStartTime - max(mReqBodyEndTime, mReqHeadersEndTime)
        listener?.responseHeadersStart(call)
    }

    override fun responseHeadersEnd(call: Call, response: Response) {
        super.responseHeadersEnd(call, response)
        "[responseHeadersEnd] code : ${response.code()}".printEvent()
        val provider = response.headers().get("x-response-cdn") ?: ""
        provider.printEvent()
        mMonitorData.cdnProvider = provider
        mRespHeadersEndTime = realtime()
        mMonitorData.recTime = realtimeDiff(mRespHeadersStartTime)
        mMonitorData.cdnReqTime = realtimeDiff(mCallStartTime)
        httpCode = response.code()
        listener?.responseHeadersEnd(call, response)
    }

    //may not call
    override fun responseBodyStart(call: Call) {
        super.responseBodyStart(call)
        "[responseBodyStart]".printEvent()
        listener?.responseBodyStart(call)
    }

    //may not call
    override fun responseBodyEnd(call: Call, byteCount: Long) {
        super.responseBodyEnd(call, byteCount)
        "[responseBodyEnd] byteCount: ${byteCount}".printEvent()
        mMonitorData.run {
            resBodySize = byteCount
            recTime = realtimeDiff(mRespHeadersStartTime)
            cdnDownTime = realtimeDiff(mRespHeadersEndTime)
            cdnSpeed = "${(byteCount / 1024f) / (cdnDownTime / 1000f)}"
        }
        listener?.responseBodyEnd(call, byteCount)
    }

    override fun responseFailed(call: Call, ioe: IOException) {
        super.responseFailed(call, ioe)
        "[responseFailed] ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        errorCode = ErrorCode.CODE_RESPONSE_FAILED
        errorMsg = ioe.shortMsg()
        listener?.responseFailed(call, ioe)
    }

    override fun callEnd(call: Call) {
        super.callEnd(call)
        "[callEnd]".printEvent()
        mMonitorData.run {
            callResultCode = SUCCESS
            reqTime = realtimeDiff(mCallStartTime)
            errorCode = httpCode
            //请求成功  1. errorCode = -1，命中缓存 2. 304命中缓存
            if (errorCode != CODE_UNKNOWN && errorCode != 304 && !mMonitorData.isTestModel) {
                track()
            }
        }
        listener?.callEnd(call)
    }

    /**
     * [IOException]
     * -[UnknownHostException]
     * -[SSLException]
     * --[SSLHandshakeException]
     * -[SocketTimeoutException]
     */
    override fun callFailed(call: Call, ioe: IOException) {
        super.callFailed(call, ioe)
        "[callFailed]  ${ioe::class.java.simpleName} : ${ioe.message}".printErrEvent()
        mMonitorData.callResultCode = FAILURE
        if (!NetworkMonitor.isNetworkValid) {
            errorCode = CODE_FAKE_NETWORK
            errorMsg = MSG_FAKE_NETWORK
        }
        mMonitorData.errorCode = if (errorCode == CODE_UNKNOWN)
            CODE_CALL_FAILED else errorCode
        mMonitorData.errorMsg = if (errorMsg == MSG_UNKNOWN)
            ioe.shortMsg() else errorMsg
        if (errorCode != CODE_UNKNOWN && !mMonitorData.isTestModel) {
            mMonitorData.track()
        }
        listener?.callFailed(call, ioe)
    }

}