package com.transsion.iot.communication.tcp

import android.os.Handler
import android.os.HandlerThread
import android.util.Log
import com.google.gson.Gson
import com.transsion.iot.communication.ConnectConstants
import com.transsion.iot.communication.ConnectListener
import com.transsion.iot.communication.MessageReceiveListener
import com.transsion.iot.communication.ResponseCode
import com.transsion.iot.communication.packet.ResponseData
import com.transsion.iot.communication.packet.ServiceIdConstants
import com.transsion.iot.communication.packet.resp.BaseResult
import kotlinx.coroutines.*
import kotlinx.coroutines.channels.Channel
import org.json.JSONException
import org.json.JSONObject
import java.io.IOException
import java.io.InputStream
import java.io.OutputStream
import java.net.Socket
import java.util.concurrent.atomic.AtomicBoolean

object TCPClient {
    private val TAG = TCPClient::class.java.simpleName
    private var ip = "192.168.188.1"
    private var port = 8889
    const val HEART_BEAT_INTERVAL = 30 * 1000L
    private const val REQUEST_TIME_OUT = 10 * 1000L
    lateinit var mSocket: Socket
    private lateinit var mOutStream: OutputStream
    private lateinit var mInStream: InputStream
    private lateinit var mReceiveThread: SocketReceiveThread
    private lateinit var mHeartBeatThread: HandlerThread
    private lateinit var mHeartBeatHandler: Handler
    private var mChannelMap = HashMap<String, Channel<String>>()
    private var mJobMap = HashMap<String, Job>()
    private var mConnectState = ConnectConstants.STATE_DISCONNECTED
    private var mConnectListener: ConnectListener? = null
    private var mMessageReceiveListener: MessageReceiveListener? = null
    private lateinit var mHeartBeatJob: Job

    private var mHeartBeatRunnable = object : Runnable {
        override fun run() {
            sendHeartBeat()
            mHeartBeatHandler.postDelayed(this, HEART_BEAT_INTERVAL)
        }
    }

    private fun initThread() {
        mReceiveThread = SocketReceiveThread()
        mReceiveThread.start()

        mHeartBeatThread = HandlerThread("heart-beat")
        mHeartBeatThread.start()
        mHeartBeatHandler = Handler(mHeartBeatThread.looper)
        mHeartBeatHandler.post(mHeartBeatRunnable)
    }

    fun setIP(ip: String) {
        this.ip = ip
    }

    fun setPort(port: Int) {
        this.port = port
    }

    fun setOnConnectListener(connectListener: ConnectListener?) {
        this.mConnectListener = connectListener
    }

    fun setMessageReceiveListener(messageReceiveListener: MessageReceiveListener?) {
        this.mMessageReceiveListener = messageReceiveListener
    }

    fun getConnectState(): Int {
        return mConnectState
    }

    fun isConnected(): Boolean {
        return mConnectState == ConnectConstants.STATE_CONNECTED
    }

    fun connect() {
        Log.i(TAG, "startConnect")
        if (ip.isEmpty() || port <= 0) {
            mConnectListener?.onError("ip or port is invalid")
            return
        }
        val mConnectThread = SocketConnectThread()
        mConnectThread.start()

        mConnectState = ConnectConstants.STATE_CONNECTING
        mConnectListener?.onConnectStateChanged(ConnectConstants.STATE_CONNECTING)
    }

    fun send(serviceId: String, message: String): String? {
        return runBlocking(Dispatchers.IO){
            val channel = Channel<String>(1)
            mChannelMap[serviceId] = channel
            val job = GlobalScope.launch {
                delay(REQUEST_TIME_OUT)
                mChannelMap[serviceId]?.send(Gson().toJson(ResponseData(ResponseCode.CODE_TIME_OUT, BaseResult(), System.currentTimeMillis().toString(), serviceId, "time out")))
                consumeChannel(serviceId)
            }
            mJobMap[serviceId] = job
            writeToOutputStream(message.toByteArray())
            mChannelMap[serviceId]?.receive()

        }
    }

    fun sendHeartBeat() {
        mHeartBeatJob = GlobalScope.launch {
            delay(HEART_BEAT_INTERVAL * 2)
            disconnect()
        }
        writeToOutputStream(ServiceIdConstants.HEART_BEAT_PING.toByteArray())
    }

    private fun writeToOutputStream(data: ByteArray) {
        if (data.isEmpty()) return
        Log.d(TAG, "writeToOutputStream data=${data}")
        try {
            mOutStream.write(data)
            mOutStream.flush()
        } catch (e: Exception) {
            e.printStackTrace()
            mConnectListener?.onError("send failed:${e.message}")
        }
    }

    private fun parseToRespData(receiveData: String) {
        if (receiveData == ServiceIdConstants.HEART_BEAT_PONG) {
            mHeartBeatJob.cancel()
            return
        }
        var serviceId: String = ServiceIdConstants.GET_DEVICE_BASIC_INFO
        try {
            val obj = JSONObject(receiveData)
            serviceId = obj.optString("serviceId")
        } catch (e: JSONException) {
            e.printStackTrace()
        }
        Log.d(TAG, "parseToRespData serviceId=$serviceId")
        Log.d(TAG, "parseToRespData mChannelMap=${mChannelMap.keys}")
        if (serviceId == ServiceIdConstants.GET_SEARCHED_MESH_NODES) {
            mMessageReceiveListener?.onMessageReceived(receiveData)
            return
        }
        GlobalScope.launch {
            mChannelMap[serviceId]?.send(receiveData)
            consumeChannel(serviceId)
            consumeJob(serviceId)
        }
    }

    fun disconnect() {
        if (!isConnected()) {
            return
        }
        mReceiveThread.threadExit()
        mConnectState = ConnectConstants.STATE_DISCONNECTED
        mConnectListener?.onConnectStateChanged(ConnectConstants.STATE_DISCONNECTED)
        mHeartBeatHandler.removeCallbacks(mHeartBeatRunnable)
        clearChannelMap()
        clearJobMap()
        try {
            mOutStream.close() //关闭输出流
            mInStream.close() //关闭输入流
            mSocket.close() //关闭socket
        } catch (e: IOException) {
            e.printStackTrace()
        }
    }

    fun onDestroy() {
        disconnect()
        mMessageReceiveListener = null
        mConnectListener = null
    }

    private fun consumeChannel(serviceId: String) {
        mChannelMap[serviceId]?.close()
        mChannelMap.remove(serviceId)
    }

    private fun consumeJob(serviceId: String) {
        mJobMap[serviceId]?.cancel()
        mJobMap.remove(serviceId)
    }

    private fun clearChannelMap() {
        if (mChannelMap.isNotEmpty()) {
            mChannelMap.clear()
        }
    }

    private fun clearJobMap() {
        if (mJobMap.isNotEmpty()) {
            for (key in mJobMap.keys) {
                mJobMap[key]?.cancel()
            }
            mJobMap.clear()
        }
    }

    class SocketConnectThread : Thread() {
        override fun run() {
            try {
                mSocket = Socket(ip, port)
                mOutStream = mSocket.getOutputStream()
                mInStream = mSocket.getInputStream()
            } catch (e: Exception) {
                e.printStackTrace()
                mConnectListener?.onError("connect failed:${e.message}")
                mConnectState = ConnectConstants.STATE_DISCONNECTED
                mConnectListener?.onConnectStateChanged(ConnectConstants.STATE_DISCONNECTED)
                return
            }
            Log.i(TAG, "connect success")
            mConnectState = ConnectConstants.STATE_CONNECTED
            mConnectListener?.onConnectStateChanged(ConnectConstants.STATE_CONNECTED)
            initThread()
        }
    }

    class SocketReceiveThread : Thread() {
        private var threadExit = AtomicBoolean(false)
        override fun run() {
            val buffer = ByteArray(40 * 1024)
            while (!threadExit.get()) {
                try {
                    //读取数据，返回值表示读到的数据长度。-1表示结束
                    val count = mInStream.read(buffer)
                    if (count == -1) {
                        Log.i(TAG, "read read -1")
                        mConnectListener?.onError("receive failed: -1")
                        break
                    } else {
                        val receiveData = String(buffer, 0, count)
                        Log.i(TAG, "read buffer:$receiveData,count=$count")
                        parseToRespData(receiveData)
                    }
                } catch (e: IOException) {
                    e.printStackTrace()
                    mConnectListener?.onError("receive failed: ${e.message}")
                    threadExit()
                }
            }
        }

        fun threadExit() {
            threadExit.set(true)
        }
    }
}