package com.ts.lib.player.ui;

import android.content.Context;
import android.graphics.Point;
import android.media.MediaPlayer;
import android.net.Uri;
import android.util.AttributeSet;
import android.view.Gravity;
import android.view.MotionEvent;
import android.view.TextureView;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.FrameLayout;
import android.widget.ImageView;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;
import android.widget.SeekBar;
import android.widget.TextView;
import android.widget.Toast;

import com.ts.lib.common.Logger;
import com.ts.lib.common.util.ToolUtils;
import com.ts.lib.common.util.Util;
import com.ts.lib.player.MediaDataSource;
import com.ts.lib.player.Player;
import com.ts.lib.player.TsPlayer;
import com.ts.lib.player.model.VideoSize;

import androidx.annotation.NonNull;
import androidx.annotation.Nullable;

/**
 * 播放器View抽象类
 *
 * @author 💎 Li Junchao
 */
public abstract class PlayerView extends FrameLayout implements View.OnClickListener
        , SeekBar.OnSeekBarChangeListener, View.OnTouchListener, Player, ControlView
        , Player.PlayerListener
{
    public static final String TAG = "TS/view";

    protected Context mContext;

    protected MediaDataSource dataSource;
    /* ------------------------------------------------------ */
    public SeekBar progressBar;
    public ViewGroup textureViewContainer;
    public View displaySurface;
    public ImageView fullscreenButton;
    public TextView currentTimeTextView, totalTimeTextView;
    public ImageView startButton;
    /* ------------------------------------------------------ */
    /** 屏幕 */
    protected int mScreenWidth, mScreenHeight;

    @ViewState
    public int state = ControlView.STATE_IDLE;
    public int screen = SCREEN_NORMAL;

    protected float mDownX;
    protected float mDownY;
    protected boolean mChangeVolume;
    protected boolean mChangePosition;
    protected boolean mChangeBrightness;
    protected long mGestureDownPosition;
    protected int mGestureDownVolume;
    protected float mGestureDownBrightness;
    protected long mSeekTimePosition;
    /* ------------------------------------------------------ */
    /** 缓冲状态 */
    public static int backUpBufferState = -1;

    public static boolean WIFI_TIP_DIALOG_SHOWED = false;
    /* ------------------------------------------------------ */

    public ViewGroup topContainer, bottomContainer;

    public PlayerView(@NonNull Context context)
    {
        this(context, null);
    }

    public PlayerView(@NonNull Context context, @Nullable AttributeSet attrs)
    {
        this(context, attrs, 0);
    }

    public PlayerView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr)
    {
        this(context, attrs, defStyleAttr, 0);
    }

    public PlayerView(@NonNull Context context, @Nullable AttributeSet attrs, int defStyleAttr, int defStyleRes)
    {
        super(context, attrs, defStyleAttr, defStyleRes);

        init(context);
    }

    protected abstract int getLayoutId();

    protected void init(Context context)
    {
        View.inflate(context, getLayoutId(), this);

        mContext = context;

        startButton = findViewById(R.id.start);
        fullscreenButton = findViewById(R.id.fullscreen);
        progressBar = findViewById(R.id.bottom_seek_progress);
        currentTimeTextView = findViewById(R.id.current);
        totalTimeTextView = findViewById(R.id.total);
        bottomContainer = findViewById(R.id.layout_bottom);
        textureViewContainer = findViewById(R.id.surface_container);
        topContainer = findViewById(R.id.layout_top);

//        displaySurface = findViewById(R.id.display_view);

        newViewIfNotFound(context);

/*        ViewGroup.LayoutParams params =
                new ViewGroup.LayoutParams(
                        ViewGroup.LayoutParams.MATCH_PARENT, ViewGroup.LayoutParams.MATCH_PARENT);*/

        progressBar.setFocusable(false);
        progressBar.setEnabled(false);

        startButton.setOnClickListener(this);
        fullscreenButton.setOnClickListener(this);
        progressBar.setOnSeekBarChangeListener(this);
        bottomContainer.setOnClickListener(this);
        textureViewContainer.setOnClickListener(this);
        textureViewContainer.setOnTouchListener(this);

        state = ControlView.STATE_IDLE;
    }

    protected void newViewIfNotFound(Context context)
    {
        if (startButton == null) {
            startButton = new ImageView(context);
        }
        if (fullscreenButton == null) {
            fullscreenButton = new ImageView(context);
        }
        if (progressBar == null) {
            progressBar = new SeekBar(context);
        }
        if (currentTimeTextView == null) {
            currentTimeTextView = new TextView(context);
        }
        if (totalTimeTextView == null) {
            totalTimeTextView = new TextView(context);
        }
        if (bottomContainer == null) {
            bottomContainer = new LinearLayout(context);
        }
        if (textureViewContainer == null) {
            textureViewContainer = new FrameLayout(context);
        }
        if (topContainer == null) {
            topContainer = new RelativeLayout(context);
        }
    }



    @Override
    public void onClick(View v)
    {
        int i = v.getId();
        if (i == R.id.start) {
            clickStart();
        } else if (i == R.id.fullscreen) {
            clickFullscreen();
        }
    }

    @Override
    public boolean onTouch(View v, MotionEvent event)
    {
        float x = event.getX();
        float y = event.getY();
        int id = v.getId();
        if (id == R.id.surface_container) {
            switch (event.getAction()) {
                case MotionEvent.ACTION_DOWN:
                    touchActionDown(x, y);
                    break;
                case MotionEvent.ACTION_MOVE:
                    touchActionMove(x, y);
                    break;
                case MotionEvent.ACTION_UP:
                    touchActionUp();
                    break;
            }
        }
        return false;
    }

    @Override
    public void onProgressChanged(SeekBar seekBar, int progress, boolean fromUser)
    {

    }

    @Override
    public void onStartTrackingTouch(SeekBar seekBar)
    {

    }

    @Override
    public void onStopTrackingTouch(SeekBar seekBar)
    {
        final TsPlayer player = PlayerManager.getInstance().getPlayer();

        if (player != null) {
            player.seekTo(seekBar.getProgress());
        }
    }

    protected boolean canPlay()
    {
        if (dataSource == null || dataSource.playbackConfiguration.uri == null) {
            Toast.makeText(mContext, getResources().getString(R.string.no_url), Toast.LENGTH_SHORT).show();
            Logger.w(TAG, "未设置播放资源");
            return false;
        }

        final Uri uri = dataSource.playbackConfiguration.uri;

        if (Util.isLocalFileUri(uri)) return true;

        // 网络地址 + 已缓存完
        if (isCacheCompleted(dataSource)) {
            Logger.i(TAG,"已缓存完毕的视频，可无网播放-->" + uri);
        } else if (ToolUtils.isNetworkAvailable(mContext)) {
            if (!ToolUtils.isWifiConnected(mContext)) {
                Logger.i(TAG, "未缓存且网络已连接但不是WiFi，是否播放-->" + uri);
                showWifiDialog();
                return false;
            }
        } else {
            Logger.w(TAG, "未缓存或者未缓存完毕的视频，需联网播放-->" + uri);
            showNetworkDialog();
            return false;
        }

        return true;
    }

    protected void clickStart()
    {
        Logger.d(TAG, "onClick start");

        final TsPlayer player = PlayerManager.getInstance().getPlayer();

        if (player == null) {
            Logger.e(TAG,"clickStart--->未初始化播放器");
            return;
        }

        if (state == STATE_NORMAL) {

            if (!canPlay()) {
                return;
            }

            setup(player);

        } else if (state == ControlView.STATE_PLAYING) {
            player.pause();
//            onStatePause();
        } else if (state == ControlView.STATE_PAUSE) {
            player.play();
//            onStatePlaying();
        } else if (state == STATE_AUTO_COMPLETE) {
            player.play();
//            onStatePlaying();
        }

/*        if (player.isPlaying()) {
            player.pause();
            onStatePause();
        } else {
            player.play();
            onStatePlaying();
        }*/
    }

    protected void startVideo()
    {
        final TsPlayer player = PlayerManager.getInstance().getPlayer();

        setup(player);
    }

    private void setup(TsPlayer player)
    {
        if (displaySurface == null) {
            switchPlayerView();

            addDisplaySurface();
            player.addListener(this);
            player.setDataSource(dataSource);
            player.setVideoTextureView((TextureView) displaySurface);
        } else if (!PlayerManager.getInstance().isSameView(textureViewContainer.hashCode())) {
            player.clearVideoSurface();
            textureViewContainer.removeView(displaySurface);

            switchPlayerView();

            addDisplaySurface();
            player.addListener(this);
            player.setDataSource(dataSource);
            player.setVideoTextureView((TextureView) displaySurface);
        }

        ToolUtils.scanForActivity(getContext()).getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);

        player.play();
    }

    private void switchPlayerView()
    {
        final PlayerView playerView = PlayerManager.getInstance().getPlayerView();
        if (playerView != null) {
            playerView.progressBar.setFocusable(false);
            playerView.progressBar.setEnabled(false);
            playerView.progressBar.setSelected(false);
            playerView.progressBar.setOnSeekBarChangeListener(null);
            if (playerView.isPlaying()) {
                playerView.pause();
            }

            playerView.onStateNormal();

            PlayerManager.getInstance().getPlayer().clearVideoSurface();
            PlayerManager.getInstance().setDisplayViewId(-1);
            PlayerManager.getInstance().setPlayerView(null);

            playerView.displaySurface = null;
            playerView.textureViewContainer.removeAllViews();
        }

        PlayerManager.getInstance().setPlayerView(this);

        progressBar.setFocusable(true);
        progressBar.setSelected(true);
        progressBar.setEnabled(true);
        progressBar.setOnSeekBarChangeListener(this);
    }

    private void addDisplaySurface()
    {
        if (displaySurface != null) textureViewContainer.removeView(displaySurface);

        displaySurface = new TextureView(mContext);

        PlayerManager.getInstance().setDisplayViewId(textureViewContainer.hashCode());

        FrameLayout.LayoutParams layoutParams =
                new FrameLayout.LayoutParams(
                        ViewGroup.LayoutParams.MATCH_PARENT,
                        ViewGroup.LayoutParams.MATCH_PARENT,
                        Gravity.CENTER);

        textureViewContainer.addView(displaySurface, layoutParams);
    }

    protected void clickFullscreen()
    {
        Logger.d(TAG, "onClick fullscreen");


    }

    protected void touchActionDown(float x, float y)
    {

    }

    protected void touchActionMove(float x, float y)
    {

    }

    protected void touchActionUp()
    {

    }

    public void onStatePlaying() {
        Logger.i(TAG, "onStatePlaying " + " [" + this.hashCode() + "] ");
        if (state == ControlView.STATE_PREPARED) {//如果是准备完成视频后第一次播放，先判断是否需要跳转进度。
            Logger.d(TAG, "onStatePlaying:STATE_PREPARED ");

        }
        state = ControlView.STATE_PLAYING;
        startProgressTimer();
    }

    public void onStatePause() {
        Logger.i(TAG, "onStatePause " + " [" + this.hashCode() + "] ");
        state = ControlView.STATE_PAUSE;
        startProgressTimer();
    }

    public void onStateError() {
        Logger.i(TAG, "onStateError " + " [" + this.hashCode() + "] ");
        state = ControlView.STATE_ERROR;
        cancelProgressTimer();
    }

    public void onStateAutoComplete()
    {
        Logger.i(TAG, "onStateAutoComplete" + " [" + this.hashCode() + "] ");
        state = STATE_AUTO_COMPLETE;
    }

    public void cancelProgressTimer() {

    }

    public void onStateNormal() {
        Logger.i(TAG, "onStateNormal " + " [" + this.hashCode() + "] ");
        state = STATE_NORMAL;
    }

    /**
     * 正在 prepare
     */
    public void onStatePreparing() {
        Logger.i(TAG, "onStatePreparing " + " [" + this.hashCode() + "] ");
        state = STATE_PREPARING;


    }

    public void onStatePreparingPlaying() {
        Logger.i(TAG, "onStatePreparingPlaying " + " [" + this.hashCode() + "] ");
        state = STATE_PREPARING_PLAYING;
    }

    public void onStatePreparingChangeUrl() {
        Logger.i(TAG, "onStatePreparingChangeUrl " + " [" + this.hashCode() + "] ");
        state = STATE_PREPARING_CHANGE_URL;
    }

    public void startProgressTimer() {

    }

    public void setState(int state) {
        switch (state) {
            case STATE_NORMAL:
                onStateNormal();
                break;
            case STATE_PREPARING:
                onStatePreparing();
                break;
            case STATE_PREPARING_PLAYING:
                onStatePreparingPlaying();
                break;
            case STATE_PREPARING_CHANGE_URL:
                onStatePreparingChangeUrl();
                break;
            case ControlView.STATE_PLAYING:
                onStatePlaying();
                break;
            case ControlView.STATE_PAUSE:
                onStatePause();
                break;
            case ControlView.STATE_ERROR:
                onStateError();
                break;
            case STATE_AUTO_COMPLETE:
                onStateAutoComplete();
                break;
        }
    }

    public void reset()
    {
        Logger.i(TAG,"reset--->view:" + this.hashCode() + "  progressBar:" + progressBar.getProgress());

        if (state == ControlView.STATE_PLAYING || state == ControlView.STATE_PAUSE
                || state == STATE_PREPARING_PLAYING) {
            Logger.i(TAG,"reset--->currentPosition:" + getCurrentPosition());
        }

        progressBar.setFocusable(false);
        progressBar.setSelected(false);
        progressBar.setEnabled(false);

        onStateNormal();

        if (state == ControlView.STATE_PLAYING
                || state == ControlView.STATE_PREPARING_PLAYING) {
            pause();
        }
        PlayerManager.getInstance().getPlayer().clearVideoSurface();
        PlayerManager.getInstance().setDisplayViewId(-1);
        PlayerManager.getInstance().setPlayerView(null);

        textureViewContainer.removeAllViews();

        ToolUtils.scanForActivity(getContext()).getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
    }

    public void showWifiDialog()
    {

    }

    public void showNetworkDialog()
    {

    }

    public void changeUrl(MediaDataSource dataSource, long positionMs)
    {

    }

    @Override
    public void onPrepared()
    {
        Logger.d(TAG, "player--------->onPrepared");
        state = ControlView.STATE_PREPARED;

        final String uriPath = dataSource.playbackConfiguration.uri.getPath();

        // 如果是audio，则开始播放
        if (Util.isAudio(uriPath)) {
            onStatePlaying();
        }
    }

    @Override
    public void onCompletion()
    {
        Logger.d(TAG, "player--------->onCompletion");
        onStateAutoComplete();

        ToolUtils.scanForActivity(getContext()).getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
    }

    @Override
    public void onVideoSizeChanged(VideoSize videoSize)
    {
        layoutByVideoRatio(videoSize);
    }

    private void layoutByVideoRatio(VideoSize videoSize)
    {
        final Point displayModeSize = Util.getCurrentDisplayModeSize(mContext);

        int containerWidth = textureViewContainer.getWidth();
        int containerHeight = textureViewContainer.getHeight();

        // 根据视频分辨率的比率计算view应该在所在容器的大小
        final Point displaySize = Util.measureSizeByRatio(containerWidth, containerHeight, videoSize.width, videoSize.height);

        Logger.d("屏幕分辨率-->" + displayModeSize + ",  video Container: "
                + containerWidth + "x" + containerHeight + " displaySize:" + videoSize);

        final View view = displaySurface;

        if (view.getWidth() / (float) view.getHeight()
                == displaySize.x / (float) displaySize.y) {
            return;
        }

        FrameLayout.LayoutParams layoutParams =
                new FrameLayout.LayoutParams(displaySize.x,displaySize.y,
                        Gravity.CENTER);

        view.setLayoutParams(layoutParams);
    }

    @Override
    public void onProgressChanged(PositionInfo newPosition)
    {
        final int durationMs = (int) newPosition.durationMs;
        if (progressBar.getMax() != durationMs) {
            progressBar.setMax(durationMs);
        }

        progressBar.setProgress((int) newPosition.currentPositionMs);
        progressBar.setSecondaryProgress((int) newPosition.bufferedPositionMs);

//            seekBar.incrementSecondaryProgressBy();

        currentTimeTextView.setText(newPosition.formatCurrentPositionElapsedTime());
        totalTimeTextView.setText(newPosition.formatDurationTime());
    }

    @Override
    public void onInfo(int what, int extra)
    {
        Logger.d(TAG, "onInfo what - " + what + " extra - " + extra);
        if (what == MediaPlayer.MEDIA_INFO_VIDEO_RENDERING_START) {
            Logger.i(TAG, "onInfo-->MEDIA_INFO_VIDEO_RENDERING_START");
            if (state == ControlView.STATE_PREPARED
                    || state == ControlView.STATE_PAUSE
                    || state == ControlView.STATE_PREPARING_PLAYING) {
                //开始渲染图像，进入playing状态
                onStatePlaying();
            }
        } else if (what == MediaPlayer.MEDIA_INFO_BUFFERING_START) {    // 开始 buffering
            Logger.i(TAG, "onInfo-->MEDIA_INFO_BUFFERING_START");
            backUpBufferState = state;
            setState(STATE_PREPARING_PLAYING);
        } else if (what == MediaPlayer.MEDIA_INFO_BUFFERING_END) {      // ready
            Logger.i(TAG, "onInfo-->MEDIA_INFO_BUFFERING_END");
            if (backUpBufferState != -1) {
                setState(backUpBufferState);
                backUpBufferState = -1;
            }
        } else if (what == Player.STATE_PAUSE) {
            onStatePause();
        }
    }

    @Override
    public void onError(int what, String extra)
    {
        Logger.e(TAG,"onError " + what + " - " + extra + " [" + this.hashCode() + "] ");
        onStateError();
    }

    @Override
    public void setDataSource(@NonNull MediaDataSource dataSource)
    {
        this.dataSource = dataSource;
        onStateNormal();
        Logger.d(TAG, "setDataSource-->" + dataSource.playbackConfiguration.uri);
    }

    @Override
    public void addListener(PlayerListener listener)
    {
        PlayerManager.getInstance().getPlayer().addListener(this);
    }

    @Override
    public void removeListener(PlayerListener listener)
    {
        PlayerManager.getInstance().getPlayer().removeListener(this);
    }

    @Override
    public void prepare()
    {
        PlayerManager.getInstance().getPlayer().prepare();
    }

    @Override
    public void play()
    {
        PlayerManager.getInstance().getPlayer().play();
    }

    @Override
    public void setPlayWhenReady(boolean playWhenReady)
    {
        PlayerManager.getInstance().getPlayer().setPlayWhenReady(playWhenReady);
    }

    @Override
    public boolean getPlayWhenReady()
    {
        return PlayerManager.getInstance().getPlayer().getPlayWhenReady();
    }

    @Override
    public boolean isPlaying()
    {
        return PlayerManager.getInstance().getPlayer().isPlaying();
    }

    @Override
    public void pause()
    {
        PlayerManager.getInstance().getPlayer().pause();
    }

    @Override
    public void stop()
    {
        PlayerManager.getInstance().getPlayer().stop();
    }

    @Override
    public long getCurrentPosition()
    {
        return PlayerManager.getInstance().getPlayer().getCurrentPosition();
    }

    @Override
    public long getBufferedPosition()
    {
        return PlayerManager.getInstance().getPlayer().getBufferedPosition();
    }

    @Override
    public long getDuration()
    {
        return PlayerManager.getInstance().getPlayer().getDuration();
    }

    @Override
    public void setRepeatMode(int repeatMode)
    {
        PlayerManager.getInstance().getPlayer().setRepeatMode(repeatMode);
    }

    @Override
    public int getRepeatMode()
    {
        return PlayerManager.getInstance().getPlayer().getRepeatMode();
    }

    @Override
    public void setSeekType(int type)
    {
        PlayerManager.getInstance().getPlayer().setSeekType(type);
    }

    @Override
    public void seekTo(long positionMs)
    {
        PlayerManager.getInstance().getPlayer().seekTo(positionMs);
    }

    @Override
    public void setVolume(float vol)
    {
        PlayerManager.getInstance().getPlayer().setVolume(vol);
    }

    @Override
    public float getVolume()
    {
        return PlayerManager.getInstance().getPlayer().getVolume();
    }

    @Override
    public void setSpeed(float speed)
    {
        PlayerManager.getInstance().getPlayer().setSpeed(speed);
    }

    @Override
    public float getSpeed()
    {
        return PlayerManager.getInstance().getPlayer().getSpeed();
    }

    @Override
    public int getPlaybackState()
    {
        return PlayerManager.getInstance().getPlayer().getPlaybackState();
    }

    @Override
    public boolean isCacheCompleted(@NonNull MediaDataSource dataSource)
    {
        return PlayerManager.getInstance().getPlayer().isCacheCompleted(dataSource);
    }

    @Override
    public void release()
    {
        /* no-op:PlayerView 不释放player */
    }
}
