package com.unity3d.ads.core.data.repository

import com.unity3d.ads.core.utils.CoroutineTimer
import com.unity3d.services.UnityAdsConstants
import com.unity3d.services.core.log.DeviceLog
import gateway.v1.DiagnosticEventRequestOuterClass.DiagnosticEventType
import gateway.v1.DiagnosticEventRequestOuterClass.DiagnosticEvent
import gateway.v1.NativeConfigurationOuterClass.DiagnosticEventsConfiguration
import kotlinx.coroutines.channels.BufferOverflow
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.getAndUpdate
import kotlinx.coroutines.flow.update

class AndroidDiagnosticEventRepository(
    private val flushTimer: CoroutineTimer
) : DiagnosticEventRepository {

    private val batch = MutableStateFlow<MutableList<DiagnosticEvent>>(value = mutableListOf())
    private var maxBatchSize = Int.MAX_VALUE
    private val allowedEvents = mutableSetOf<DiagnosticEventType>()
    private val blockedEvents = mutableSetOf<DiagnosticEventType>()
    private val enabled = MutableStateFlow(false)
    private val configured = MutableStateFlow(false)

    private val _diagnosticEvents = MutableSharedFlow<List<DiagnosticEvent>>(
        replay = UnityAdsConstants.SharedFlow.REPLAY,
        extraBufferCapacity = UnityAdsConstants.SharedFlow.EXTRA_CAPACITY,
        onBufferOverflow = BufferOverflow.DROP_OLDEST
    )

    override val diagnosticEvents = _diagnosticEvents.asSharedFlow()

    override fun addDiagnosticEvent(diagnosticEvent: DiagnosticEvent) {
        if (!configured.value) {
            batch.update { it.add(diagnosticEvent); it }
        } else {
            if (!enabled.value) return
            batch.update { it.add(diagnosticEvent); it }
            if (batch.value.size >= maxBatchSize) {
                flush()
            }
        }
    }

    override fun flush() {
        val events = batch.getAndUpdate { mutableListOf() }
            .asSequence()
            .filter { allowedEvents.isEmpty() || it.eventType in allowedEvents }
            .filter { it.eventType !in blockedEvents }
            .toList()

        if (events.isNotEmpty()) {
            DeviceLog.debug("Unity Ads Sending diagnostic batch enabled: ${enabled.value} size: ${events.size} :: $events")
            _diagnosticEvents.tryEmit(events)
        }
    }

    override fun clear() {
        batch.update { mutableListOf() }
    }

    override fun configure(diagnosticsEventsConfiguration: DiagnosticEventsConfiguration) {
        configured.value = true
        enabled.value = diagnosticsEventsConfiguration.enabled
        if (!enabled.value) {
            clear()
            return
        }
        this.maxBatchSize = diagnosticsEventsConfiguration.maxBatchSize
        this.allowedEvents.addAll(diagnosticsEventsConfiguration.allowedEventsList)
        this.blockedEvents.addAll(diagnosticsEventsConfiguration.blockedEventsList)

        val maxBatchIntervalMs = diagnosticsEventsConfiguration.maxBatchIntervalMs

        flushTimer.start(
            delayStartMillis = 0,
            repeatMillis = maxBatchIntervalMs.toLong(),
        ) { flush() }
    }
}