package com.vungle.ads.internal

import android.content.Context
import com.vungle.ads.BidTokenCallback
import com.vungle.ads.BuildConfig
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.VungleAds
import com.vungle.ads.internal.bidding.BidTokenEncoder
import com.vungle.ads.internal.executor.FutureResult
import com.vungle.ads.internal.executor.SDKExecutors
import com.vungle.ads.internal.privacy.PrivacyManager
import com.vungle.ads.internal.util.ConcurrencyTimeoutProvider
import java.util.concurrent.Callable
import java.util.concurrent.TimeUnit

internal class VungleInternal {

    /**
     * Will return an encoded string of advertisement bid tokens.
     * This method might be called from adapter side.
     *
     * Optional: Pass in null or empty String to make this filter return all encoded advertisements.

     * @return an encoded string contains available bid tokens digest. In rare cases, this can return null value
     */
    fun getAvailableBidTokens(context: Context): String? {
        if (!VungleAds.isInitialized()) {
            PrivacyManager.init(context.applicationContext)
        }
        val provider: ConcurrencyTimeoutProvider by inject(context)
        val sdkExecutors: SDKExecutors by inject(context)
        val bidTokenEncoder: BidTokenEncoder by inject(context)
        val futureResult: FutureResult<String?> = FutureResult(sdkExecutors.apiExecutor
            .submit(Callable<String?> {
                bidTokenEncoder.encode().bidToken
            })
        )
        return futureResult.get(provider.getTimeout(), TimeUnit.MILLISECONDS)
    }

    /**
     * Will return an encoded string of advertisement bidding token.
     */
    fun getAvailableBidTokensAsync(context: Context, callback: BidTokenCallback) {
        if (!VungleAds.isInitialized()) {
            PrivacyManager.init(context.applicationContext)
        }

        val bidTokenEncoder: BidTokenEncoder by inject(context)
        val sdkExecutors: SDKExecutors by inject(context)
        sdkExecutors.apiExecutor.execute {
            val bidTokenInfo = bidTokenEncoder.encode()
            if (bidTokenInfo.bidToken.isNotEmpty()) {
                callback.onBidTokenCollected(bidTokenInfo.bidToken)
            } else {
                callback.onBidTokenError(bidTokenInfo.errorMessage)
            }
        }
    }

    fun getSdkVersion() = BuildConfig.VERSION_NAME
}
