@file:JvmName("BaseAdLoader")

package com.vungle.ads.internal.load

import android.content.Context
import android.webkit.URLUtil
import androidx.annotation.WorkerThread
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.AssetDownloadError
import com.vungle.ads.InternalError
import com.vungle.ads.MraidJsError
import com.vungle.ads.NativeAdInternal
import com.vungle.ads.ServiceLocator.Companion.inject
import com.vungle.ads.SingleValueMetric
import com.vungle.ads.TimeIntervalMetric
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.downloader.AssetDownloadListener
import com.vungle.ads.internal.downloader.AssetDownloadListener.DownloadError.Companion.DEFAULT_SERVER_CODE
import com.vungle.ads.internal.downloader.DownloadRequest
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.Executors
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.model.AdPayload
import com.vungle.ads.internal.network.TpatSender
import com.vungle.ads.internal.network.VungleApiClient
import com.vungle.ads.internal.omsdk.OMInjector
import com.vungle.ads.internal.protos.Sdk
import com.vungle.ads.internal.signals.SignalManager
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.Logger
import com.vungle.ads.internal.util.PathProvider
import com.vungle.ads.internal.util.UnzipUtility
import java.io.File
import java.io.IOException
import java.util.concurrent.atomic.AtomicBoolean
import java.util.concurrent.atomic.AtomicLong


abstract class BaseAdLoader(
    val context: Context,
    val vungleApiClient: VungleApiClient,
    val sdkExecutors: Executors,
    private val omInjector: OMInjector,
    private val downloader: Downloader,
    val pathProvider: PathProvider,
    val adRequest: AdRequest
) {

    companion object {
        private const val TAG = "BaseAdLoader"
        private const val DOWNLOADED_FILE_NOT_FOUND = "Downloaded file not found!"
    }

    private val downloadCount: AtomicLong = AtomicLong(0)

    private val downloadRequiredCount: AtomicLong = AtomicLong(0)

    private var adLoaderCallback: AdLoaderCallback? = null

    private var notifySuccess = AtomicBoolean(false)
    private var notifyFailed = AtomicBoolean(false)

    private val adAssets: MutableList<AdAsset> = mutableListOf()

    internal var advertisement: AdPayload? = null
    private var fullyDownloaded = AtomicBoolean(true)
    private var requiredAssetDownloaded = AtomicBoolean(true)

    private var mainVideoSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.ASSET_FILE_SIZE)

    private var templateSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_ZIP_SIZE)

    private var templateHtmlSizeMetric =
        SingleValueMetric(Sdk.SDKMetric.SDKMetricType.TEMPLATE_HTML_SIZE)

    private var assetDownloadDurationMetric =
        TimeIntervalMetric(Sdk.SDKMetric.SDKMetricType.ASSET_DOWNLOAD_DURATION_MS)

    // check if all the assets downloaded successfully.
    private val assetDownloadListener: AssetDownloadListener
        get() {
            return object : AssetDownloadListener {
                override fun onError(
                    error: AssetDownloadListener.DownloadError?,
                    downloadRequest: DownloadRequest
                ) {
                    Logger.e(TAG, "onError called: reason ${error?.reason}; cause ${error?.cause}")
                    sdkExecutors.backgroundExecutor.execute {

                        fullyDownloaded.set(false)
                        if (downloadRequest.asset.isRequired) {
                            requiredAssetDownloaded.set(false)
                        }

                        if (downloadRequest.asset.isRequired && downloadRequiredCount.decrementAndGet() <= 0) {
                            // call failure callback if all required assets are not downloaded.
                            onAdLoadFailed(AssetDownloadError())
                            // cancel the rest of optional download requests
                            cancel()
                            return@execute
                        }

                        if (downloadCount.decrementAndGet() <= 0) {
                            onAdLoadFailed(AssetDownloadError())
                        }
                    }
                }

                override fun onSuccess(file: File, downloadRequest: DownloadRequest) {
                    sdkExecutors.backgroundExecutor.execute {
                        if (!file.exists()) {
                            onError(
                                AssetDownloadListener.DownloadError(
                                    DEFAULT_SERVER_CODE,
                                    IOException(DOWNLOADED_FILE_NOT_FOUND),
                                    AssetDownloadListener.DownloadError.ErrorReason.FILE_NOT_FOUND_ERROR
                                ),
                                downloadRequest
                            ) //AdAsset table will be updated in onError callback
                            return@execute
                        }

                        val adAsset = downloadRequest.asset
                        adAsset.fileSize = file.length()
                        adAsset.status = AdAsset.Status.DOWNLOAD_SUCCESS

                        if (downloadRequest.isTemplate) {
                            downloadRequest.stopRecord()
                            val templateFileSizeMetric = if (downloadRequest.isHtmlTemplate) {
                                templateHtmlSizeMetric
                            } else {
                                templateSizeMetric
                            }
                            templateFileSizeMetric.value = file.length()
                            AnalyticsClient.logMetric(
                                templateFileSizeMetric,
                                placementId = adRequest.placement.referenceId,
                                creativeId = advertisement?.getCreativeId(),
                                eventId = advertisement?.eventId(),
                                metaData = adAsset.serverPath
                            )
                        } else if (downloadRequest.isMainVideo) {
                            mainVideoSizeMetric.value = file.length()
                            AnalyticsClient.logMetric(
                                mainVideoSizeMetric,
                                placementId = adRequest.placement.referenceId,
                                creativeId = advertisement?.getCreativeId(),
                                eventId = advertisement?.eventId(),
                                metaData = adAsset.serverPath
                            )
                        }

                        // Update the asset path in the mraid file map
                        advertisement?.updateAdAssetPath(adAsset)

                        if (downloadRequest.isTemplate) {
                            if (!processVmTemplate(adAsset, advertisement)) {
                                fullyDownloaded.set(false)
                                if (adAsset.isRequired) {
                                    requiredAssetDownloaded.set(false)
                                }
                            }
                        }

                        if (adAsset.isRequired && downloadRequiredCount.decrementAndGet() <= 0) {
                            // onAdLoaded callback will be triggered when isRequired assets are downloaded.
                            if (requiredAssetDownloaded.get()) {
                                onAdReady()
                            } else {
                                onAdLoadFailed(AssetDownloadError())
                                cancel()
                                return@execute
                            }
                        }

                        // check if all the assets downloaded successfully.
                        if (downloadCount.decrementAndGet() <= 0) {
                            // set advertisement state to READY
                            if (fullyDownloaded.get()) {
                                onDownloadCompleted(adRequest, advertisement?.eventId())
                            } else {
                                onAdLoadFailed(AssetDownloadError())
                            }
                        }
                    }
                }
            }
        }

    fun loadAd(adLoaderCallback: AdLoaderCallback) {
        this.adLoaderCallback = adLoaderCallback

        sdkExecutors.backgroundExecutor.execute {
            AnalyticsClient.logMetric(
                Sdk.SDKMetric.SDKMetricType.LOAD_AD_API,
                placementId = adRequest.placement.referenceId
            )

            requestAd()
        }
    }

    protected abstract fun requestAd()

    abstract fun onAdLoadReady()

    fun cancel() {
        downloader.cancelAll()
    }

    private fun downloadAssets(advertisement: AdPayload) {
        assetDownloadDurationMetric.markStart()
        downloadCount.set(adAssets.size.toLong())
        downloadRequiredCount.set(adAssets.filter { it.isRequired }.size.toLong())
        for (asset in adAssets) {
            val downloadRequest =
                DownloadRequest(
                    getAssetPriority(asset),
                    asset,
                    adRequest.placement.referenceId,
                    advertisement.getCreativeId(),
                    advertisement.eventId()
                )
            if (downloadRequest.isTemplate) {
                downloadRequest.startRecord()
            }
            downloader.download(downloadRequest, assetDownloadListener)
        }
    }

    fun onAdLoadFailed(error: VungleError) {
        if (!notifySuccess.get() && notifyFailed.compareAndSet(false, true)) {
            adLoaderCallback?.onFailure(error)
        }
    }

    private fun onAdReady() {
        advertisement?.let {

            // onSuccess can only be called once. For ADO case, it will be called right after
            // template is downloaded. After all assets are downloaded, onSuccess will not be called.
            if (!notifyFailed.get() && notifySuccess.compareAndSet(false, true)) {
                // After real time ad loaded, will send win notifications.
                // Use this abstract method to notify sub ad loader doing some clean up job.
                onAdLoadReady()
                adLoaderCallback?.onSuccess(it)
            }
        }
    }

    private fun isUrlValid(url: String?): Boolean {
        return !url.isNullOrEmpty() && (URLUtil.isHttpsUrl(url) || URLUtil.isHttpUrl(url))
    }

    private fun fileIsValid(file: File, adAsset: AdAsset): Boolean {
        return file.exists() && file.length() == adAsset.fileSize
    }

    private fun unzipFile(
        advertisement: AdPayload,
        downloadedFile: File,
        destinationDir: File
    ): Boolean {
        val existingPaths: MutableList<String> = ArrayList()
        for (asset in adAssets) {
            if (asset.fileType == AdAsset.FileType.ASSET) {
                existingPaths.add(asset.localPath)
            }
        }
        try {
            UnzipUtility.unzip(downloadedFile.path, destinationDir.path,
                object : UnzipUtility.Filter {
                    override fun matches(extractPath: String?): Boolean {
                        if (extractPath.isNullOrEmpty()) {
                            return true
                        }
                        val toExtract = File(extractPath)
                        for (existing in existingPaths) {
                            val existingFile = File(existing)
                            if (existingFile == toExtract)
                                return false
                            if (toExtract.path.startsWith(existingFile.path + File.separator))
                                return false
                        }
                        return true
                    }
                })

            val file = File(destinationDir.path, Constants.AD_INDEX_FILE_NAME)
            if (!file.exists()) {
                AnalyticsClient.logError(
                    VungleError.INVALID_INDEX_URL,
                    "Failed to retrieve indexFileUrl from the Ad.",
                    adRequest.placement.referenceId,
                    advertisement.getCreativeId(),
                    advertisement.eventId(),
                )
                return false
            }

        } catch (ex: Exception) {
            AnalyticsClient.logError(
                VungleError.TEMPLATE_UNZIP_ERROR,
                "Unzip failed: ${ex.message}",
                adRequest.placement.referenceId,
                advertisement.getCreativeId(),
                advertisement.eventId(),
            )
            return false
        }

        FileUtility.delete(downloadedFile)
        return true
    }

    private fun getDestinationDir(advertisement: AdPayload): File? {
        return pathProvider.getDownloadsDirForAd(advertisement.eventId())
    }

    private fun injectMraidJS(destinationDir: File): Boolean {
        try {
            val adMraidJS = File(destinationDir.path, Constants.AD_MRAID_JS_FILE_NAME)
            val mraidJsPath = pathProvider.getJsAssetDir(ConfigManager.getMraidJsVersion())
            val mraidJsFile = File(mraidJsPath, Constants.MRAID_JS_FILE_NAME)
            if (mraidJsFile.exists()) {
                mraidJsFile.copyTo(adMraidJS, true)
            }
        } catch (e: Exception) {
            Logger.e(TAG, "Failed to inject mraid.js: ${e.message}")
            return false
        }
        return true
    }

    private fun processVmTemplate(
        asset: AdAsset,
        advertisement: AdPayload?
    ): Boolean {
        if (advertisement == null) {
            return false
        }
        if (asset.status != AdAsset.Status.DOWNLOAD_SUCCESS) {
            return false
        }
        if (asset.localPath.isEmpty()) {
            return false
        }
        val vmTemplate = File(asset.localPath)
        if (!fileIsValid(vmTemplate, asset)) {
            return false
        }

        val destinationDir = getDestinationDir(advertisement)
        if (destinationDir == null || !destinationDir.isDirectory) {
            Logger.e(TAG, "Unable to access Destination Directory")
            return false
        }

        if (asset.fileType == AdAsset.FileType.ZIP && !unzipFile(
                advertisement,
                vmTemplate,
                destinationDir
            )
        ) {
            return false
        }

        // Inject OMSDK
        if (advertisement.omEnabled()) {
            try {
                omInjector.injectJsFiles(destinationDir)
            } catch (e: Exception) {
                Logger.e(TAG, "Failed to inject OMSDK: ${e.message}")
                return false
            }
        }

        // Inject MRAID JS
        return injectMraidJS(destinationDir).also {
            FileUtility.printDirectoryTree(destinationDir)
        }

    }

    @WorkerThread
    open fun onDownloadCompleted(request: AdRequest, advertisementId: String?) {
        Logger.d(TAG, "download completed $request")
        advertisement?.setAssetFullyDownloaded()
        onAdReady()

        assetDownloadDurationMetric.markEnd()
        val placementId = advertisement?.placementId()
        val creativeId = advertisement?.getCreativeId()
        val eventId = advertisement?.eventId()
        AnalyticsClient.logMetric(assetDownloadDurationMetric, placementId, creativeId, eventId)
    }

    internal fun handleAdMetaData(advertisement: AdPayload, metric: SingleValueMetric?= null) {
        this.advertisement = advertisement

        // Update config
        advertisement.config()?.let { config ->
            ConfigManager.initWithConfig(context, config, false, metric)
        }

        val error = validateAdMetadata(advertisement)
        if (error != null) {

            AnalyticsClient.logError(
                error.reason,
                error.description,
                adRequest.placement.referenceId,
                advertisement.getCreativeId(),
                advertisement.eventId(),
            )
            onAdLoadFailed(InternalError(error.reason, error.descriptionExternal))
            return
        }

        val destinationDir: File? = getDestinationDir(advertisement)
        if (destinationDir == null || !destinationDir.isDirectory || !destinationDir.exists()) {
            onAdLoadFailed(AssetDownloadError())
            return
        }
        val signalManager: SignalManager by inject(context)

        // URLs for start to load ad notification when load ad after get the adm and
        // before assets start to download.
        advertisement.adUnit()?.loadAdUrls?.also { loadAdUrls ->
            val tpatSender = TpatSender(
                vungleApiClient,
                advertisement.placementId(),
                advertisement.getCreativeId(),
                advertisement.eventId(),
                sdkExecutors.ioExecutor,
                pathProvider,
                signalManager = signalManager
            )
            loadAdUrls.forEach {
                tpatSender.sendTpat(it, sdkExecutors.jobExecutor)
            }
        }

        if (adAssets.isNotEmpty()) {
            adAssets.clear()
        }
        adAssets.addAll(advertisement.getDownloadableAssets(destinationDir))

        if (adAssets.isEmpty()) {
            onAdLoadFailed(AssetDownloadError())
            return
        }

        // Move mraid.js download here, before start downloading other assets.
        MraidJsLoader.downloadJs(pathProvider, downloader, sdkExecutors.backgroundExecutor,
            object : MraidJsLoader.DownloadResultListener {
                override fun onDownloadResult(downloadResult: Int) {
                    if (downloadResult == MraidJsLoader.MRAID_AVAILABLE
                        || downloadResult == MraidJsLoader.MRAID_DOWNLOADED
                    ) {
                        if (downloadResult == MraidJsLoader.MRAID_DOWNLOADED) {
                            AnalyticsClient.logMetric(
                                Sdk.SDKMetric.SDKMetricType.MRAID_DOWNLOAD_JS_RETRY_SUCCESS,
                                placementId = adRequest.placement.referenceId
                            )
                        }
                        downloadAssets(advertisement)
                    } else {
                        adLoaderCallback?.onFailure(MraidJsError())
                    }
                }
            })

    }

    private fun getAssetPriority(adAsset: AdAsset): DownloadRequest.Priority {
        return if (adAsset.isRequired) {
            DownloadRequest.Priority.CRITICAL
        } else {
            DownloadRequest.Priority.HIGHEST
        }
    }

    private fun validateAdMetadata(adPayload: AdPayload): ErrorInfo? {

        var reason: Int
        var description: String
        if (adPayload.adUnit()?.sleep != null) {
            return getErrorInfo(adPayload)
        }

        if (adRequest.placement.referenceId != advertisement?.placementId()) {
            reason = VungleError.AD_RESPONSE_EMPTY
            description = "Requests and responses don't match the placement Id."
            return ErrorInfo(reason, description)
        }

        val templateSettings = adPayload.adUnit()?.templateSettings
        if (templateSettings == null) {
            reason = VungleError.ASSET_RESPONSE_DATA_ERROR
            description = "Missing template settings"
            return ErrorInfo(reason, description)
        }
        val cacheableReplacements = templateSettings.cacheableReplacements
        if (adPayload.isNativeTemplateType()) {
            cacheableReplacements?.let {
                if (it[NativeAdInternal.TOKEN_MAIN_IMAGE]?.url == null) {
                    reason = VungleError.NATIVE_ASSET_ERROR
                    description = "Unable to load main image."
                    return ErrorInfo(reason, description)
                }
                if (it[NativeAdInternal.TOKEN_VUNGLE_PRIVACY_ICON_URL]?.url == null) {
                    reason = VungleError.NATIVE_ASSET_ERROR
                    description = "Unable to load privacy image."
                    return ErrorInfo(reason, description)
                }
            }
        } else {
            val templateUrl = adPayload.adUnit()?.templateURL
            val vmUrl = adPayload.adUnit()?.vmURL
            if (templateUrl.isNullOrEmpty() && vmUrl.isNullOrEmpty()) {
                reason = VungleError.INVALID_TEMPLATE_URL
                description = "Failed to prepare vmURL or templateURL for downloading."
                return ErrorInfo(reason, description)
            }

            if (!templateUrl.isNullOrEmpty() && !isUrlValid(templateUrl)) {
                reason = VungleError.ASSET_REQUEST_ERROR
                description = "Failed to load template asset."
                return ErrorInfo(reason, description)
            }

            if (!vmUrl.isNullOrEmpty() && !isUrlValid(vmUrl)) {
                reason = VungleError.ASSET_REQUEST_ERROR
                description = "Failed to load vm url asset."
                return ErrorInfo(reason, description)
            }
        }

        if (adPayload.hasExpired()) {
            reason = VungleError.AD_EXPIRED
            description = "The ad markup has expired for playback."
            return ErrorInfo(reason, description)
        }

        if (adPayload.eventId().isNullOrEmpty()) {
            reason = VungleError.INVALID_EVENT_ID_ERROR
            description = "Event id is invalid."
            return ErrorInfo(reason, description)
        }

        cacheableReplacements?.forEach {
            val httpUrl = it.value.url
            if (httpUrl.isNullOrEmpty()) {
                reason = VungleError.INVALID_ASSET_URL
                description = "Invalid asset URL $httpUrl"
                return ErrorInfo(reason, description)
            }
            if (!isUrlValid(httpUrl)) {
                reason = VungleError.ASSET_REQUEST_ERROR
                description = "Invalid asset URL $httpUrl"
                return ErrorInfo(reason, description)
            }
        }

        return null
    }

    private fun getErrorInfo(adPayload: AdPayload): ErrorInfo {

        val errorCode: Int = adPayload.adUnit()?.errorCode ?: VungleError.PLACEMENT_SLEEP
        val sleep = adPayload.adUnit()?.sleep
        val info = adPayload.adUnit()?.info
        when (errorCode) {
            Sdk.SDKError.Reason.AD_NO_FILL_VALUE,
            Sdk.SDKError.Reason.AD_LOAD_TOO_FREQUENTLY_VALUE,
            Sdk.SDKError.Reason.AD_SERVER_ERROR_VALUE,
            Sdk.SDKError.Reason.AD_PUBLISHER_MISMATCH_VALUE,
            Sdk.SDKError.Reason.AD_INTERNAL_INTEGRATION_ERROR_VALUE,
            -> {
                return ErrorInfo(
                    errorCode, "Response error: $sleep",
                    "Request failed with error: $errorCode, $info"
                )
            }

            else -> {
                return ErrorInfo(
                    VungleError.PLACEMENT_SLEEP, "Response error: $sleep",
                    "Request failed with error: ${VungleError.PLACEMENT_SLEEP}, $info"
                )
            }
        }

    }

    class ErrorInfo(
        val reason: Int,
        val description: String,
        val descriptionExternal: String = description,
        val errorIsTerminal: Boolean = false,
    )

}
