package com.vungle.ads.internal.load

import com.vungle.ads.internal.util.Logger
import com.vungle.ads.AnalyticsClient
import com.vungle.ads.MraidJsError
import com.vungle.ads.VungleError
import com.vungle.ads.internal.ConfigManager
import com.vungle.ads.internal.Constants
import com.vungle.ads.internal.downloader.AssetDownloadListener
import com.vungle.ads.internal.downloader.DownloadRequest
import com.vungle.ads.internal.downloader.Downloader
import com.vungle.ads.internal.executor.VungleThreadPoolExecutor
import com.vungle.ads.internal.model.AdAsset
import com.vungle.ads.internal.util.FileUtility
import com.vungle.ads.internal.util.PathProvider
import java.io.File
import java.util.concurrent.CopyOnWriteArrayList
import java.util.concurrent.atomic.AtomicBoolean

object MraidJsLoader {

    private const val TAG = "MraidJsLoader"

    const val MRAID_DOWNLOADED = 10
    const val MRAID_INVALID_ENDPOINT = 11
    const val MRAID_DOWNLOAD_FAILED = 12
    const val MRAID_AVAILABLE = 13

    private val isDownloading = AtomicBoolean(false)

    private val listeners = CopyOnWriteArrayList<DownloadResultListener>()

    interface DownloadResultListener {
        fun onDownloadResult(downloadResult: Int)
    }

    fun downloadJs(
        pathProvider: PathProvider,
        downloader: Downloader,
        executor: VungleThreadPoolExecutor,
        downloadResultListener: DownloadResultListener? = null
    ) {
        executor.execute {
            try {
                downloadResultListener?.let { listeners.add(it) }

                if (isDownloading.getAndSet(true)) {
                    Logger.w(TAG, "mraid js is downloading, waiting for the previous request.")
                    return@execute
                }

                // Check if mraid js endpoint valid
                val mraidEndpoint = ConfigManager.getMraidEndpoint()
                if (mraidEndpoint.isNullOrEmpty()) {
                    notifyListeners(MRAID_INVALID_ENDPOINT)
                    return@execute
                }

                // Check if mraid js is already downloaded
                val mraidJsPath = pathProvider.getJsAssetDir(ConfigManager.getMraidJsVersion())
                val mraidJsFile = File(mraidJsPath, Constants.MRAID_JS_FILE_NAME)
                if (mraidJsFile.exists()) {
                    Logger.w(TAG, "mraid js already downloaded")
                    notifyListeners(MRAID_AVAILABLE)
                    return@execute
                }

                // delete invalid js first
                val jsPath = pathProvider.getJsDir()
                FileUtility.deleteContents(jsPath)

                // download latest js
                val mraidJsAsset = AdAsset(
                    Constants.MRAID_JS_FILE_NAME,
                    "$mraidEndpoint/${Constants.MRAID_JS_FILE_NAME}",
                    mraidJsFile.absolutePath,
                    AdAsset.FileType.ASSET,
                    true
                )
                val mraidDownloadRequest = DownloadRequest(DownloadRequest.Priority.HIGH, mraidJsAsset)
                downloader.download(mraidDownloadRequest, object : AssetDownloadListener {
                    override fun onError(
                        error: AssetDownloadListener.DownloadError?,
                        downloadRequest: DownloadRequest
                    ) {
                        executor.execute {
                            try {
                                val errorMessage = "download mraid js error: ${error?.serverCode}. " +
                                        "Failed to load asset ${downloadRequest.asset.serverPath}"
                                Logger.d(TAG, errorMessage)
                                MraidJsError(errorMessage).logErrorNoReturnValue()
                                FileUtility.deleteContents(jsPath)
                            } catch (e: Exception) {
                                Logger.e(TAG, "Failed to delete js assets", e)
                            } finally {
                                notifyListeners(MRAID_DOWNLOAD_FAILED)
                            }
                        }
                    }

                    override fun onSuccess(file: File, downloadRequest: DownloadRequest) {
                        executor.execute {
                            try {
                                if (file.exists() && file.length() > 0) {
                                    notifyListeners(MRAID_DOWNLOADED)
                                } else {
                                    AnalyticsClient.logError(
                                        VungleError.MRAID_JS_WRITE_FAILED,
                                        "Mraid js downloaded but write failure: ${mraidJsFile.absolutePath}"
                                    )

                                    FileUtility.deleteContents(jsPath)
                                    notifyListeners(MRAID_DOWNLOAD_FAILED)
                                }
                            } catch (e: Exception) {
                                Logger.e(TAG, "Failed to delete js assets", e)
                                notifyListeners(MRAID_DOWNLOAD_FAILED)
                            }
                        }
                    }
                })
            } catch (e: Exception) {
                Logger.e(TAG, "Failed to download mraid js", e)
            }

        }
    }

    private fun notifyListeners(downloadResult: Int) {
        listeners.forEach {
            it.onDownloadResult(downloadResult)
        }
        listeners.clear()

        isDownloading.set(false)
    }

}
