package com.zaz.translate.app

import android.app.Application
import android.content.Context
import android.content.Intent
import android.os.Bundle
import androidx.annotation.Keep
import com.zaz.translate.log.log
import kotlinx.serialization.Serializable
import java.lang.reflect.Method
import kotlin.coroutines.suspendCoroutine


@Keep
fun Application.getServerAppKey(): String? {

    val application = this

    val methodName = "getServerAppKey"
    val getServerAppKeyMethod = application.javaClass.getMethod(methodName)

    return getServerAppKeyMethod.invoke(application)?.toString()

}

@Keep
fun Application.getServerAppSecret(): String? {
    val application = this
    val methodName = "getServerAppSecret"
    val getServerAppKeyMethod = application.javaClass.getMethod(methodName)

    return getServerAppKeyMethod.invoke(application)?.toString()
}

/**
 * 翻译扩展，通过反射实现，对应 Application 中如下方法，如未实现，请实现。
 * @Keep
 * fun translate(text: String, sourceLanguage: String? = null, targetLanguage: String): String
 */
@Keep
fun Application.translate(
    text: String,
    sourceLanguage: String? = null,
    targetLanguage: String
): String? {

    val application = this

    val methodName = "translate"
    val translateMethod = application.javaClass.getMethod(
        methodName,
        String::class.java,
        String::class.java,
        String::class.java
    )

    val result =
        translateMethod.invoke(application, text, sourceLanguage, targetLanguage)?.toString()

    log(message = "translate-result==$result")
    return result
}

/**
 * return bundle
 * @link EXTRA_MULTI_TEXTS,EXTRA_MULTI_SOURCE_LANGUAGES,EXTRA_MULTI_TARGET_LANGUAGES
 */
@Keep
fun Application.multiTranslate(bundle: Bundle): Bundle? {
    val application = this

    val methodName = "multiTranslate"
    val translateMethod = application.javaClass.getMethod(
        methodName,
        Bundle::class.java,
    )
    return translateMethod.invoke(application, bundle) as? Bundle
}

/**
 * 对应 Application 中如下方法，如未实现，请实现。方法作用为：获取悬浮球文本翻译目标语言，对应于“我的语言”
 * @Keep
 * fun readTextLanguage(): String?
 */
@Keep
fun Application.readTextLanguage(): String? {

    val application = this

    val methodName = "readTextLanguage"

    val method = application.javaClass.getMethod(methodName)

    return method.invoke(application) as? String
}

/**
 * 对应 Application 中如下方法，如未实现，请实现。方法作用为：获取悬浮球文本编辑框翻译目标语言，对应于“朋友的语言”
 * @Keep
 * fun readEditLanguage(): String?
 */
@Keep
fun Application.readEditLanguage(): String? {

    val application = this

    val methodName = "readEditLanguage"

    val method = application.javaClass.getMethod(methodName)

    return method.invoke(application) as? String
}

//private fun Application.audioLanguageCode() = readEditLanguage()


/**
 * 对应 Application 中如下方法，如未实现，请实现。方法作用为：获取 Google Speech to Text app Key
 * @Keep
 * fun getGoogleSpeechToTextKey(): String?
 */
@Keep
fun Application.getGoogleSpeechToTextKey(): String? {

    val application = this

    val methodName = "getGoogleSpeechToTextKey"

    val method = application.javaClass.getMethod(methodName)

    return method.invoke(application) as? String //?: "AIzaSyAy_ifrpf7lwMWW_pv93wLZTtZQtL0vBwA"
}

/**
 * 翻译扩展，通过反射实现，对应 Application 中如下方法，如未实现，请实现。方法作用为：实现数据分析，埋点功能
 * @Keep
 * fun logEvent(): String?
 */
@Keep
fun Application.logEvent(
    context: Context,
    eventId: String,
    params: HashMap<String, String> = hashMapOf()
) {
    val application = this

    val methodName = "logEvent"

    val method =
        application.javaClass.getMethod(
            methodName,
            Context::class.java,
            String::class.java,
            HashMap::class.java
        )

    method.invoke(application, context, eventId, params)
}


/**
 * 文本复制功能
 * @Keep
 * class ImplApplication
 * fun copy(): String?
 */
@Keep
fun Application.copy(label: CharSequence, text: CharSequence) {
    val application = this

    val methodName = "copy"

    val method =
        application.javaClass.getMethod(
            methodName,
            CharSequence::class.java,
            CharSequence::class.java
        )

    method.invoke(application, label, text)
}

/**
 * showLanguageSheet
 * @Keep
 * class ImplApplication
 * fun showLanguageSheet(context: Context,@LanguageType languageType: Int,autoDetect: Boolean = false,excludeLanguage: String? = null,recent: ArrayList<String> = arrayListOf(),requestCode: Int = 100)
 */
@Keep
fun Application.showLanguageSheet(
    context: Context,
    languageType: Int,
    autoDetect: Boolean = false,
    excludeLanguage: String? = null,
    recent: ArrayList<String> = arrayListOf(),
    requestCode: Int = 100
) {
    val application = this

    val methodName = "showLanguageSheet"

    val method =
        application.javaClass.getMethod(
            methodName,
            Context::class.java,
            Int::class.java,
            Boolean::class.java,
            String::class.java,
            ArrayList::class.java,
            Int::class.java
        )

    method.invoke(
        application,
        context,
        languageType,
        autoDetect,
        excludeLanguage,
        recent,
        requestCode
    )
}

/**
 * showLanguageSheetIntent
 * @Keep
 * class ImplApplication
 * fun showLanguageSheetIntent(context: Context,@LanguageType languageType: Int,autoDetect: Boolean = false,excludeLanguage: String? = null)
 */
@Keep
fun Application.showLanguageSheetIntent(
    context: Context,
    languageType: Int,
    autoDetect: Boolean = false,
    excludeLanguage: String? = null
): Intent {

    val application = this

    val methodName = "showLanguageSheetIntent"

    val method =
        application.javaClass.getMethod(
            methodName,
            Context::class.java,
            Int::class.java,
            Boolean::class.java,
            String::class.java
        )

    return method.invoke(
        application,
        context,
        languageType,
        autoDetect,
        excludeLanguage
    ) as Intent
}

@Serializable
@Keep
data class TransResult(
    val translation: String? = null,
    val errorMessage: String? = null,
    val source: String? = null
) {
    var text: String? = null
    var sourceLanguage: String? = null
    var targetLanguage: String? = null
}

suspend fun Method.invokeSuspend(obj: Any, vararg args: Any?): Any? =
    suspendCoroutine {
        invoke(obj, *args, it)
    }

//    Bundle().apply {
//        putStringArrayList("texts", ts)
//        putStringArrayList("sourceLanguages", sls)
//        putStringArrayList("targetLanguages", tls)
const val EXTRA_MULTI_TEXTS = "texts"
const val EXTRA_MULTI_SOURCE_LANGUAGES = "sourceLanguages"
const val EXTRA_MULTI_TARGET_LANGUAGES = "targetLanguages"